<?php
/**
 * Arquivo de configuração principal
 * Sistema de Gerenciamento Eletrônico de Documentos (GED)
 */

// Definições de caminho
define('BASE_PATH', dirname(__DIR__));
define('UPLOAD_PATH', BASE_PATH . '/uploads');
define('INCLUDE_PATH', BASE_PATH . '/includes');
define('MODEL_PATH', BASE_PATH . '/models');
define('VIEW_PATH', BASE_PATH . '/views');
define('CONTROLLER_PATH', BASE_PATH . '/controllers');
define('ASSET_PATH', BASE_PATH . '/assets');

// Configurações de sessão
ini_set('session.cookie_httponly', 1);
ini_set('session.use_only_cookies', 1);
ini_set('session.cookie_secure', 1);
session_start();

// Configurações de segurança
error_reporting(E_ALL);
ini_set('display_errors', 0);
ini_set('log_errors', 1);
ini_set('error_log', BASE_PATH . '/logs/error.log');

// Criar diretório de logs se não existir
if (!file_exists(BASE_PATH . '/logs')) {
    mkdir(BASE_PATH . '/logs', 0755, true);
}

// Configurações de banco de dados
$db_config = [
    'host' => 'localhost',
    'dbname' => 'ged_db1',
    'username' => 'root',
    'password' => 'password'
];

// Função para carregar automaticamente classes
spl_autoload_register(function ($class_name) {
    // Verificar em models
    if (file_exists(MODEL_PATH . '/' . strtolower($class_name) . '.php')) {
        require_once MODEL_PATH . '/' . strtolower($class_name) . '.php';
        return;
    }
    
    // Verificar em controllers
    if (file_exists(CONTROLLER_PATH . '/' . strtolower($class_name) . '.php')) {
        require_once CONTROLLER_PATH . '/' . strtolower($class_name) . '.php';
        return;
    }
    
    // Verificar em includes
    if (file_exists(INCLUDE_PATH . '/' . strtolower($class_name) . '.php')) {
        require_once INCLUDE_PATH . '/' . strtolower($class_name) . '.php';
        return;
    }
});

// Função para sanitizar entrada
function sanitize($input) {
    if (is_array($input)) {
        foreach ($input as $key => $value) {
            $input[$key] = sanitize($value);
        }
    } else {
        $input = htmlspecialchars(trim($input), ENT_QUOTES, 'UTF-8');
    }
    return $input;
}

// Função para redirecionar
function redirect($url) {
    header("Location: $url");
    exit;
}

// Função para verificar se o usuário está autenticado
function isAuthenticated() {
    return isset($_SESSION['user_id']);
}

// Função para verificar permissão
function hasPermission($permission) {
    if (!isAuthenticated()) {
        return false;
    }
    
    // Administradores têm todas as permissões
    if ($_SESSION['user_type'] === 'admin') {
        return true;
    }
    
    return in_array($permission, $_SESSION['permissions'] ?? []);
}

// Função para exigir autenticação
function requireAuth() {
    if (!isAuthenticated()) {
        $_SESSION['redirect_after_login'] = $_SERVER['REQUEST_URI'];
        redirect('/login.php');
    }
}

// Função para exigir permissão
function requirePermission($permission) {
    requireAuth();
    
    if (!hasPermission($permission)) {
        redirect('/access-denied.php');
    }
}

// Função para gerar token CSRF
function generateCSRFToken() {
    if (empty($_SESSION['csrf_token'])) {
        $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
    }
    return $_SESSION['csrf_token'];
}

// Função para verificar token CSRF
function verifyCSRFToken($token) {
    return isset($_SESSION['csrf_token']) && hash_equals($_SESSION['csrf_token'], $token);
}

// Função para registrar log de auditoria
function logAudit($user_id, $action, $resource_type, $resource_id = null, $details = null) {
    global $db_config;
    
    try {
        $db = new PDO(
            "mysql:host={$db_config['host']};dbname={$db_config['dbname']}",
            $db_config['username'],
            $db_config['password']
        );
        $db->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
        
        $query = "INSERT INTO audit_log (user_id, action, resource_type, resource_id, details, ip_address, user_agent)
                  VALUES (?, ?, ?, ?, ?, ?, ?)";
        
        $stmt = $db->prepare($query);
        $stmt->execute([
            $user_id,
            $action,
            $resource_type,
            $resource_id,
            $details,
            $_SERVER['REMOTE_ADDR'] ?? null,
            $_SERVER['HTTP_USER_AGENT'] ?? null
        ]);
        
        return true;
    } catch (PDOException $e) {
        error_log("Erro ao registrar log de auditoria: " . $e->getMessage());
        return false;
    }
}
?>
