<?php
/**
 * Controlador de Logs de Auditoria
 * Sistema de Gerenciamento Eletrônico de Documentos (GED)
 */

// Incluir arquivos necessários
require_once __DIR__ . '/../config/database.php';
require_once __DIR__ . '/../models/audit_log.php';

class AuditLogController {
    private $db;
    private $audit_log;
    
    /**
     * Construtor
     */
    public function __construct() {
        $database = new Database();
        $this->db = $database->getConnection();
        $this->audit_log = new AuditLog($this->db);
    }
    
    /**
     * Lista logs de auditoria com filtros e paginação
     * @param array $filters Filtros
     * @param int $page Página atual
     * @param int $per_page Itens por página
     * @return array Logs e informações de paginação
     */
    public function listLogs($filters = [], $page = 1, $per_page = 50) {
        $result = $this->audit_log->readAll($filters, $page, $per_page);
        
        return [
            'success' => true,
            'logs' => $result['logs'],
            'pagination' => $result['pagination']
        ];
    }
    
    /**
     * Obtém um log de auditoria pelo ID
     * @param int $id ID do log
     * @return array Log ou erro
     */
    public function getLog($id) {
        if ($this->audit_log->readOne($id)) {
            return [
                'success' => true,
                'log' => [
                    'id' => $this->audit_log->id,
                    'user_id' => $this->audit_log->user_id,
                    'action' => $this->audit_log->action,
                    'entity_type' => $this->audit_log->entity_type,
                    'entity_id' => $this->audit_log->entity_id,
                    'details' => $this->audit_log->details,
                    'ip_address' => $this->audit_log->ip_address,
                    'user_agent' => $this->audit_log->user_agent,
                    'created_at' => $this->audit_log->created_at
                ]
            ];
        } else {
            return [
                'success' => false,
                'message' => 'Log não encontrado'
            ];
        }
    }
    
    /**
     * Obtém estatísticas de atividade por usuário
     * @param string $start_date Data de início (formato Y-m-d)
     * @param string $end_date Data de fim (formato Y-m-d)
     * @return array Estatísticas
     */
    public function getUserActivityStats($start_date = null, $end_date = null) {
        $stats = $this->audit_log->getUserActivityStats($start_date, $end_date);
        
        return [
            'success' => true,
            'stats' => $stats
        ];
    }
    
    /**
     * Obtém estatísticas de atividade por tipo de entidade
     * @param string $start_date Data de início (formato Y-m-d)
     * @param string $end_date Data de fim (formato Y-m-d)
     * @return array Estatísticas
     */
    public function getEntityActivityStats($start_date = null, $end_date = null) {
        $stats = $this->audit_log->getEntityActivityStats($start_date, $end_date);
        
        return [
            'success' => true,
            'stats' => $stats
        ];
    }
    
    /**
     * Obtém estatísticas de atividade por ação
     * @param string $start_date Data de início (formato Y-m-d)
     * @param string $end_date Data de fim (formato Y-m-d)
     * @return array Estatísticas
     */
    public function getActionStats($start_date = null, $end_date = null) {
        $stats = $this->audit_log->getActionStats($start_date, $end_date);
        
        return [
            'success' => true,
            'stats' => $stats
        ];
    }
    
    /**
     * Obtém histórico de atividade de um documento específico
     * @param int $document_id ID do documento
     * @return array Histórico de atividade
     */
    public function getDocumentHistory($document_id) {
        $history = $this->audit_log->getDocumentHistory($document_id);
        
        return [
            'success' => true,
            'history' => $history
        ];
    }
    
    /**
     * Exporta logs para CSV
     * @param array $filters Filtros
     * @return string Caminho do arquivo CSV gerado
     */
    public function exportLogsCSV($filters = []) {
        // Obter todos os logs sem paginação
        $result = $this->audit_log->readAll($filters, 1, 10000);
        $logs = $result['logs'];
        
        // Criar arquivo CSV temporário
        $temp_file = tempnam(sys_get_temp_dir(), 'logs_');
        $file = fopen($temp_file, 'w');
        
        // Escrever cabeçalho
        fputcsv($file, ['ID', 'Usuário', 'Ação', 'Tipo de Entidade', 'ID da Entidade', 'Detalhes', 'IP', 'User Agent', 'Data/Hora']);
        
        // Escrever dados
        foreach ($logs as $log) {
            fputcsv($file, [
                $log['id'],
                $log['username'],
                $log['action'],
                $log['entity_type'],
                $log['entity_id'],
                $log['details'],
                $log['ip_address'],
                $log['user_agent'],
                $log['created_at']
            ]);
        }
        
        fclose($file);
        
        return $temp_file;
    }
    
    /**
     * Exporta logs para PDF
     * @param array $filters Filtros
     * @return string Caminho do arquivo PDF gerado
     */
    public function exportLogsPDF($filters = []) {
        // Obter todos os logs sem paginação
        $result = $this->audit_log->readAll($filters, 1, 1000);
        $logs = $result['logs'];
        
        // Criar arquivo PDF temporário usando uma biblioteca como FPDF ou TCPDF
        // Este é um exemplo simplificado, em uma implementação real você usaria uma biblioteca PDF
        
        $temp_file = tempnam(sys_get_temp_dir(), 'logs_') . '.pdf';
        
        // Aqui você usaria uma biblioteca PDF para gerar o arquivo
        // Por exemplo, com FPDF:
        /*
        require_once 'fpdf/fpdf.php';
        $pdf = new FPDF();
        $pdf->AddPage();
        $pdf->SetFont('Arial', 'B', 16);
        $pdf->Cell(40, 10, 'Logs de Auditoria');
        // ... adicionar dados dos logs ...
        $pdf->Output($temp_file, 'F');
        */
        
        // Como este é apenas um exemplo, vamos criar um arquivo de texto simples
        $file = fopen($temp_file, 'w');
        fwrite($file, "Logs de Auditoria\n\n");
        
        foreach ($logs as $log) {
            fwrite($file, "ID: " . $log['id'] . "\n");
            fwrite($file, "Usuário: " . $log['username'] . "\n");
            fwrite($file, "Ação: " . $log['action'] . "\n");
            fwrite($file, "Entidade: " . $log['entity_type'] . " #" . $log['entity_id'] . "\n");
            fwrite($file, "Detalhes: " . $log['details'] . "\n");
            fwrite($file, "Data/Hora: " . $log['created_at'] . "\n\n");
        }
        
        fclose($file);
        
        return $temp_file;
    }
}
?>
