<?php
/**
 * Controlador de Documentos
 * Sistema de Gerenciamento Eletrônico de Documentos (GED)
 */

// Incluir arquivos necessários
require_once __DIR__ . '/../config/database.php';
require_once __DIR__ . '/../models/document.php';
require_once __DIR__ . '/../models/category.php';

class DocumentController {
    private $db;
    private $document;
    
    /**
     * Construtor
     */
    public function __construct() {
        $database = new Database();
        $this->db = $database->getConnection();
        $this->document = new Document($this->db);
    }
    
    /**
     * Processa o upload de um novo documento
     * @param array $data Dados do documento
     * @param array $file Dados do arquivo
     * @param int $user_id ID do usuário que está fazendo o upload
     * @return array Resultado do upload
     */
    public function upload($data, $file, $user_id) {
        // Verificar campos obrigatórios
        if (empty($data['title'])) {
            return [
                'success' => false,
                'message' => 'O título do documento é obrigatório'
            ];
        }
        
        if (empty($file['name'])) {
            return [
                'success' => false,
                'message' => 'Nenhum arquivo foi selecionado'
            ];
        }
        
        // Verificar se houve erro no upload
        if ($file['error'] !== UPLOAD_ERR_OK) {
            $error_message = $this->getUploadErrorMessage($file['error']);
            return [
                'success' => false,
                'message' => 'Erro no upload: ' . $error_message
            ];
        }
        
        // Verificar tipo de arquivo permitido
        $allowed_types = ['application/pdf', 'application/msword', 'application/vnd.openxmlformats-officedocument.wordprocessingml.document', 
                         'image/jpeg', 'image/png', 'image/gif', 'application/vnd.ms-excel', 
                         'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet', 'text/plain'];
        
        if (!in_array($file['type'], $allowed_types)) {
            return [
                'success' => false,
                'message' => 'Tipo de arquivo não permitido. Tipos permitidos: PDF, DOC, DOCX, XLS, XLSX, JPG, PNG, GIF, TXT'
            ];
        }
        
        // Verificar tamanho máximo (50MB)
        $max_size = 50 * 1024 * 1024; // 50MB em bytes
        if ($file['size'] > $max_size) {
            return [
                'success' => false,
                'message' => 'O arquivo excede o tamanho máximo permitido (50MB)'
            ];
        }
        
        // Criar diretório de uploads se não existir
        $upload_dir = UPLOAD_PATH . '/' . date('Y/m');
        if (!file_exists($upload_dir)) {
            mkdir($upload_dir, 0755, true);
        }
        
        // Gerar nome de arquivo único
        $file_extension = pathinfo($file['name'], PATHINFO_EXTENSION);
        $file_name = uniqid() . '_' . time() . '.' . $file_extension;
        $file_path = $upload_dir . '/' . $file_name;
        
        // Mover arquivo para o diretório de uploads
        if (!move_uploaded_file($file['tmp_name'], $file_path)) {
            return [
                'success' => false,
                'message' => 'Erro ao mover o arquivo para o diretório de uploads'
            ];
        }
        
        // Salvar documento no banco de dados
        $this->document->title = $data['title'];
        $this->document->author = $data['author'] ?? '';
        $this->document->file_path = str_replace(BASE_PATH, '', $file_path); // Salvar caminho relativo
        $this->document->file_type = $file['type'];
        $this->document->file_size = $file['size'];
        $this->document->category_id = $data['category_id'] ?? null;
        $this->document->status = $data['status'] ?? 'draft';
        $this->document->created_by = $user_id;
        
        if ($this->document->create()) {
            // Salvar palavras-chave
            if (!empty($data['keywords'])) {
                $keywords = explode(',', $data['keywords']);
                $this->document->setKeywords($keywords);
            }
            
            // Registrar log de auditoria
            logAudit($user_id, 'create', 'document', $this->document->id, 'Documento criado: ' . $data['title']);
            
            return [
                'success' => true,
                'message' => 'Documento enviado com sucesso',
                'document_id' => $this->document->id
            ];
        } else {
            // Remover arquivo se falhar ao salvar no banco
            unlink($file_path);
            
            return [
                'success' => false,
                'message' => 'Erro ao salvar o documento no banco de dados'
            ];
        }
    }
    
    /**
     * Obtém um documento pelo ID
     * @param int $id ID do documento
     * @return array Documento ou erro
     */
    public function getDocument($id) {
        if ($this->document->readOne($id)) {
            // Obter palavras-chave
            $keywords = $this->document->getKeywords();
            
            // Obter versões
            $versions = $this->document->getVersions();
            
            return [
                'success' => true,
                'document' => [
                    'id' => $this->document->id,
                    'title' => $this->document->title,
                    'author' => $this->document->author,
                    'file_path' => $this->document->file_path,
                    'file_type' => $this->document->file_type,
                    'file_size' => $this->document->file_size,
                    'category_id' => $this->document->category_id,
                    'status' => $this->document->status,
                    'created_at' => $this->document->created_at,
                    'updated_at' => $this->document->updated_at,
                    'created_by' => $this->document->created_by,
                    'keywords' => $keywords,
                    'versions' => $versions
                ]
            ];
        } else {
            return [
                'success' => false,
                'message' => 'Documento não encontrado'
            ];
        }
    }
    
    /**
     * Lista documentos com filtros e paginação
     * @param array $filters Filtros
     * @param int $page Página atual
     * @param int $per_page Itens por página
     * @return array Lista de documentos
     */
    public function listDocuments($filters = [], $page = 1, $per_page = 10) {
        $documents = $this->document->readAll($filters, $page, $per_page);
        $total = $this->document->countAll($filters);
        $total_pages = ceil($total / $per_page);
        
        return [
            'success' => true,
            'documents' => $documents,
            'pagination' => [
                'total' => $total,
                'per_page' => $per_page,
                'current_page' => $page,
                'total_pages' => $total_pages
            ]
        ];
    }
    
    /**
     * Atualiza um documento existente
     * @param int $id ID do documento
     * @param array $data Novos dados
     * @param int $user_id ID do usuário que está atualizando
     * @return array Resultado da atualização
     */
    public function updateDocument($id, $data, $user_id) {
        // Verificar se o documento existe
        if (!$this->document->readOne($id)) {
            return [
                'success' => false,
                'message' => 'Documento não encontrado'
            ];
        }
        
        // Verificar campos obrigatórios
        if (empty($data['title'])) {
            return [
                'success' => false,
                'message' => 'O título do documento é obrigatório'
            ];
        }
        
        // Atualizar dados
        $this->document->title = $data['title'];
        $this->document->author = $data['author'] ?? $this->document->author;
        $this->document->category_id = $data['category_id'] ?? $this->document->category_id;
        $this->document->status = $data['status'] ?? $this->document->status;
        
        if ($this->document->update()) {
            // Atualizar palavras-chave
            if (isset($data['keywords'])) {
                $keywords = explode(',', $data['keywords']);
                $this->document->setKeywords($keywords);
            }
            
            // Registrar log de auditoria
            logAudit($user_id, 'update', 'document', $id, 'Documento atualizado: ' . $data['title']);
            
            return [
                'success' => true,
                'message' => 'Documento atualizado com sucesso'
            ];
        } else {
            return [
                'success' => false,
                'message' => 'Erro ao atualizar o documento'
            ];
        }
    }
    
    /**
     * Atualiza o arquivo de um documento existente (nova versão)
     * @param int $id ID do documento
     * @param array $file Dados do arquivo
     * @param string $change_summary Resumo das alterações
     * @param int $user_id ID do usuário que está atualizando
     * @return array Resultado da atualização
     */
    public function updateFile($id, $file, $change_summary, $user_id) {
        // Verificar se o documento existe
        if (!$this->document->readOne($id)) {
            return [
                'success' => false,
                'message' => 'Documento não encontrado'
            ];
        }
        
        // Verificar se há arquivo
        if (empty($file['name'])) {
            return [
                'success' => false,
                'message' => 'Nenhum arquivo foi selecionado'
            ];
        }
        
        // Verificar se houve erro no upload
        if ($file['error'] !== UPLOAD_ERR_OK) {
            $error_message = $this->getUploadErrorMessage($file['error']);
            return [
                'success' => false,
                'message' => 'Erro no upload: ' . $error_message
            ];
        }
        
        // Verificar tipo de arquivo permitido
        $allowed_types = ['application/pdf', 'application/msword', 'application/vnd.openxmlformats-officedocument.wordprocessingml.document', 
                         'image/jpeg', 'image/png', 'image/gif', 'application/vnd.ms-excel', 
                         'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet', 'text/plain'];
        
        if (!in_array($file['type'], $allowed_types)) {
            return [
                'success' => false,
                'message' => 'Tipo de arquivo não permitido. Tipos permitidos: PDF, DOC, DOCX, XLS, XLSX, JPG, PNG, GIF, TXT'
            ];
        }
        
        // Verificar tamanho máximo (50MB)
        $max_size = 50 * 1024 * 1024; // 50MB em bytes
        if ($file['size'] > $max_size) {
            return [
                'success' => false,
                'message' => 'O arquivo excede o tamanho máximo permitido (50MB)'
            ];
        }
        
        // Criar diretório de uploads se não existir
        $upload_dir = UPLOAD_PATH . '/' . date('Y/m');
        if (!file_exists($upload_dir)) {
            mkdir($upload_dir, 0755, true);
        }
        
        // Gerar nome de arquivo único
        $file_extension = pathinfo($file['name'], PATHINFO_EXTENSION);
        $file_name = uniqid() . '_' . time() . '.' . $file_extension;
        $file_path = $upload_dir . '/' . $file_name;
        
        // Mover arquivo para o diretório de uploads
        if (!move_uploaded_file($file['tmp_name'], $file_path)) {
            return [
                'success' => false,
                'message' => 'Erro ao mover o arquivo para o diretório de uploads'
            ];
        }
        
        // Salvar versão anterior
        $old_file_path = $this->document->file_path;
        $old_file_size = $this->document->file_size;
        
        // Criar nova versão
        if ($this->document->createVersion($old_file_path, $old_file_size, $user_id, $change_summary)) {
            // Atualizar arquivo principal
            $relative_path = str_replace(BASE_PATH, '', $file_path);
            if ($this->document->updateFile($relative_path, $file['type'], $file['size'])) {
                // Registrar log de auditoria
                logAudit($user_id, 'update_file', 'document', $id, 'Arquivo do documento atualizado: ' . $this->document->title);
                
                return [
                    'success' => true,
                    'message' => 'Arquivo atualizado com sucesso e versão anterior preservada'
                ];
            } else {
                // Remover arquivo se falhar ao atualizar
                unlink($file_path);
                
                return [
                    'success' => false,
                    'message' => 'Erro ao atualizar o arquivo do documento'
                ];
            }
        } else {
            // Remover arquivo se falhar ao criar versão
            unlink($file_path);
            
            return [
                'success' => false,
                'message' => 'Erro ao criar versão do documento'
            ];
        }
    }
    
    /**
     * Exclui um documento
     * @param int $id ID do documento
     * @param int $user_id ID do usuário que está excluindo
     * @return array Resultado da exclusão
     */
    public function deleteDocument($id, $user_id) {
        // Verificar se o documento existe
        if (!$this->document->readOne($id)) {
            return [
                'success' => false,
                'message' => 'Documento não encontrado'
            ];
        }
        
        // Salvar informações para log
        $document_title = $this->document->title;
        $file_path = BASE_PATH . $this->document->file_path;
        
        // Excluir documento do banco de dados
        if ($this->document->delete()) {
            // Excluir arquivo físico
            if (file_exists($file_path)) {
                unlink($file_path);
            }
            
            // Registrar log de auditoria
            logAudit($user_id, 'delete', 'document', $id, 'Documento excluído: ' . $document_title);
            
            return [
                'success' => true,
                'message' => 'Documento excluído com sucesso'
            ];
        } else {
            return [
                'success' => false,
                'message' => 'Erro ao excluir o documento'
            ];
        }
    }
    
    /**
     * Obtém uma versão específica de um documento
     * @param int $document_id ID do documento
     * @param int $version_id ID da versão
     * @return array Versão ou erro
     */
    public function getVersion($document_id, $version_id) {
        $query = "SELECT * FROM document_version WHERE id = ? AND document_id = ?";
        $stmt = $this->db->prepare($query);
        $stmt->bindParam(1, $version_id);
        $stmt->bindParam(2, $document_id);
        $stmt->execute();
        
        if ($stmt->rowCount() > 0) {
            $version = $stmt->fetch(PDO::FETCH_ASSOC);
            return [
                'success' => true,
                'version' => $version
            ];
        } else {
            return [
                'success' => false,
                'message' => 'Versão não encontrada'
            ];
        }
    }
    
    /**
     * Obtém mensagem de erro de upload
     * @param int $error_code Código de erro
     * @return string Mensagem de erro
     */
    private function getUploadErrorMessage($error_code) {
        switch ($error_code) {
            case UPLOAD_ERR_INI_SIZE:
                return 'O arquivo excede o tamanho máximo permitido pelo PHP';
            case UPLOAD_ERR_FORM_SIZE:
                return 'O arquivo excede o tamanho máximo permitido pelo formulário';
            case UPLOAD_ERR_PARTIAL:
                return 'O arquivo foi enviado parcialmente';
            case UPLOAD_ERR_NO_FILE:
                return 'Nenhum arquivo foi enviado';
            case UPLOAD_ERR_NO_TMP_DIR:
                return 'Diretório temporário não encontrado';
            case UPLOAD_ERR_CANT_WRITE:
                return 'Falha ao gravar o arquivo no disco';
            case UPLOAD_ERR_EXTENSION:
                return 'Upload interrompido por uma extensão PHP';
            default:
                return 'Erro desconhecido no upload';
        }
    }
}
?>
