<?php
/**
 * Controlador de Pesquisa
 * Sistema de Gerenciamento Eletrônico de Documentos (GED)
 */

// Incluir arquivos necessários
require_once __DIR__ . '/../config/database.php';
require_once __DIR__ . '/../models/document.php';

class SearchController {
    private $db;
    
    /**
     * Construtor
     */
    public function __construct() {
        $database = new Database();
        $this->db = $database->getConnection();
    }
    
    /**
     * Realiza pesquisa avançada de documentos
     * @param array $filters Filtros de pesquisa
     * @param int $page Página atual
     * @param int $per_page Itens por página
     * @return array Resultados da pesquisa
     */
    public function advancedSearch($filters, $page = 1, $per_page = 10) {
        // Construir consulta base
        $query = "SELECT d.*, c.name as category_name, u.username as created_by_username
                  FROM document d
                  LEFT JOIN category c ON d.category_id = c.id
                  LEFT JOIN user u ON d.created_by = u.id
                  WHERE 1=1";
        
        $params = [];
        
        // Aplicar filtros
        if (!empty($filters['title'])) {
            $query .= " AND d.title LIKE ?";
            $params[] = "%" . $filters['title'] . "%";
        }
        
        if (!empty($filters['author'])) {
            $query .= " AND d.author LIKE ?";
            $params[] = "%" . $filters['author'] . "%";
        }
        
        if (!empty($filters['category_id'])) {
            $query .= " AND d.category_id = ?";
            $params[] = $filters['category_id'];
        }
        
        if (!empty($filters['status'])) {
            $query .= " AND d.status = ?";
            $params[] = $filters['status'];
        }
        
        if (!empty($filters['start_date'])) {
            $query .= " AND d.created_at >= ?";
            $params[] = $filters['start_date'] . ' 00:00:00';
        }
        
        if (!empty($filters['end_date'])) {
            $query .= " AND d.created_at <= ?";
            $params[] = $filters['end_date'] . ' 23:59:59';
        }
        
        // Pesquisa por palavras-chave
        if (!empty($filters['keyword'])) {
            $query .= " AND EXISTS (SELECT 1 FROM document_keyword dk WHERE dk.document_id = d.id AND dk.keyword LIKE ?)";
            $params[] = "%" . $filters['keyword'] . "%";
        }
        
        // Pesquisa por conteúdo (se implementado OCR)
        if (!empty($filters['content'])) {
            // Nota: Esta parte requer implementação de OCR ou indexação de conteúdo
            // Por enquanto, apenas pesquisamos no título e palavras-chave
            $query .= " AND (d.title LIKE ? OR EXISTS (SELECT 1 FROM document_keyword dk WHERE dk.document_id = d.id AND dk.keyword LIKE ?))";
            $params[] = "%" . $filters['content'] . "%";
            $params[] = "%" . $filters['content'] . "%";
        }
        
        // Contar total de resultados para paginação
        $count_query = str_replace("SELECT d.*, c.name as category_name, u.username as created_by_username", "SELECT COUNT(*) as total", $query);
        $stmt = $this->db->prepare($count_query);
        
        for ($i = 0; $i < count($params); $i++) {
            $stmt->bindParam($i + 1, $params[$i]);
        }
        
        $stmt->execute();
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        $total = $row['total'];
        
        // Adicionar ordenação e paginação
        $query .= " ORDER BY d.created_at DESC";
        
        // Calcular offset para paginação
        $offset = ($page - 1) * $per_page;
        $query .= " LIMIT ?, ?";
        
        $stmt = $this->db->prepare($query);
        
        // Vincular parâmetros de filtro
        for ($i = 0; $i < count($params); $i++) {
            $stmt->bindParam($i + 1, $params[$i]);
        }
        
        // Vincular parâmetros de paginação
        $stmt->bindParam(count($params) + 1, $offset, PDO::PARAM_INT);
        $stmt->bindParam(count($params) + 2, $per_page, PDO::PARAM_INT);
        
        $stmt->execute();
        
        $documents = [];
        while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
            // Obter palavras-chave para cada documento
            $keywords_query = "SELECT keyword FROM document_keyword WHERE document_id = ?";
            $keywords_stmt = $this->db->prepare($keywords_query);
            $keywords_stmt->bindParam(1, $row['id']);
            $keywords_stmt->execute();
            
            $keywords = [];
            while ($keyword_row = $keywords_stmt->fetch(PDO::FETCH_ASSOC)) {
                $keywords[] = $keyword_row['keyword'];
            }
            
            $row['keywords'] = $keywords;
            $documents[] = $row;
        }
        
        // Calcular total de páginas
        $total_pages = ceil($total / $per_page);
        
        return [
            'success' => true,
            'documents' => $documents,
            'pagination' => [
                'total' => $total,
                'per_page' => $per_page,
                'current_page' => $page,
                'total_pages' => $total_pages
            ]
        ];
    }
    
    /**
     * Pesquisa rápida por termo em título, autor ou palavras-chave
     * @param string $term Termo de pesquisa
     * @param int $page Página atual
     * @param int $per_page Itens por página
     * @return array Resultados da pesquisa
     */
    public function quickSearch($term, $page = 1, $per_page = 10) {
        if (empty($term)) {
            return [
                'success' => false,
                'message' => 'Termo de pesquisa não fornecido'
            ];
        }
        
        // Construir consulta
        $query = "SELECT d.*, c.name as category_name, u.username as created_by_username
                  FROM document d
                  LEFT JOIN category c ON d.category_id = c.id
                  LEFT JOIN user u ON d.created_by = u.id
                  WHERE d.title LIKE ? OR d.author LIKE ? OR EXISTS (
                      SELECT 1 FROM document_keyword dk WHERE dk.document_id = d.id AND dk.keyword LIKE ?
                  )";
        
        // Contar total de resultados para paginação
        $count_query = str_replace("SELECT d.*, c.name as category_name, u.username as created_by_username", "SELECT COUNT(*) as total", $query);
        $stmt = $this->db->prepare($count_query);
        
        $search_term = "%" . $term . "%";
        $stmt->bindParam(1, $search_term);
        $stmt->bindParam(2, $search_term);
        $stmt->bindParam(3, $search_term);
        
        $stmt->execute();
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        $total = $row['total'];
        
        // Adicionar ordenação e paginação
        $query .= " ORDER BY d.created_at DESC";
        
        // Calcular offset para paginação
        $offset = ($page - 1) * $per_page;
        $query .= " LIMIT ?, ?";
        
        $stmt = $this->db->prepare($query);
        
        // Vincular parâmetros de pesquisa
        $stmt->bindParam(1, $search_term);
        $stmt->bindParam(2, $search_term);
        $stmt->bindParam(3, $search_term);
        
        // Vincular parâmetros de paginação
        $stmt->bindParam(4, $offset, PDO::PARAM_INT);
        $stmt->bindParam(5, $per_page, PDO::PARAM_INT);
        
        $stmt->execute();
        
        $documents = [];
        while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
            // Obter palavras-chave para cada documento
            $keywords_query = "SELECT keyword FROM document_keyword WHERE document_id = ?";
            $keywords_stmt = $this->db->prepare($keywords_query);
            $keywords_stmt->bindParam(1, $row['id']);
            $keywords_stmt->execute();
            
            $keywords = [];
            while ($keyword_row = $keywords_stmt->fetch(PDO::FETCH_ASSOC)) {
                $keywords[] = $keyword_row['keyword'];
            }
            
            $row['keywords'] = $keywords;
            $documents[] = $row;
        }
        
        // Calcular total de páginas
        $total_pages = ceil($total / $per_page);
        
        return [
            'success' => true,
            'documents' => $documents,
            'pagination' => [
                'total' => $total,
                'per_page' => $per_page,
                'current_page' => $page,
                'total_pages' => $total_pages
            ]
        ];
    }
}
?>
