<?php
/**
 * Controlador de Workflow
 * Sistema de Gerenciamento Eletrônico de Documentos (GED)
 */

// Incluir arquivos necessários
require_once __DIR__ . '/../config/database.php';
require_once __DIR__ . '/../models/workflow.php';
require_once __DIR__ . '/../models/document.php';

class WorkflowController {
    private $db;
    private $workflow;
    
    /**
     * Construtor
     */
    public function __construct() {
        $database = new Database();
        $this->db = $database->getConnection();
        $this->workflow = new Workflow($this->db);
    }
    
    /**
     * Lista todos os workflows
     * @return array Lista de workflows
     */
    public function listWorkflows() {
        $stmt = $this->workflow->readAll();
        $workflows = [];
        
        while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
            $workflows[] = $row;
        }
        
        return [
            'success' => true,
            'workflows' => $workflows
        ];
    }
    
    /**
     * Obtém um workflow pelo ID, incluindo suas etapas
     * @param int $id ID do workflow
     * @return array Workflow ou erro
     */
    public function getWorkflow($id) {
        if ($this->workflow->readOne($id)) {
            // Obter etapas
            $steps = $this->workflow->getSteps();
            
            return [
                'success' => true,
                'workflow' => [
                    'id' => $this->workflow->id,
                    'name' => $this->workflow->name,
                    'description' => $this->workflow->description,
                    'created_at' => $this->workflow->created_at,
                    'created_by' => $this->workflow->created_by,
                    'steps' => $steps
                ]
            ];
        } else {
            return [
                'success' => false,
                'message' => 'Workflow não encontrado'
            ];
        }
    }
    
    /**
     * Cria um novo workflow
     * @param array $data Dados do workflow
     * @param int $user_id ID do usuário que está criando
     * @return array Resultado da criação
     */
    public function createWorkflow($data, $user_id) {
        // Verificar campos obrigatórios
        if (empty($data['name'])) {
            return [
                'success' => false,
                'message' => 'O nome do workflow é obrigatório'
            ];
        }
        
        // Criar workflow
        $this->workflow->name = $data['name'];
        $this->workflow->description = $data['description'] ?? '';
        $this->workflow->created_by = $user_id;
        
        if ($this->workflow->create()) {
            // Registrar log de auditoria
            logAudit($user_id, 'create', 'workflow', $this->workflow->id, 'Workflow criado: ' . $data['name']);
            
            // Adicionar etapas, se fornecidas
            if (!empty($data['steps']) && is_array($data['steps'])) {
                foreach ($data['steps'] as $step) {
                    $this->workflow->addStep(
                        $step['name'],
                        $step['description'] ?? '',
                        $step['approver_id'] ?? $user_id
                    );
                }
            }
            
            return [
                'success' => true,
                'message' => 'Workflow criado com sucesso',
                'workflow_id' => $this->workflow->id
            ];
        } else {
            return [
                'success' => false,
                'message' => 'Erro ao criar workflow'
            ];
        }
    }
    
    /**
     * Atualiza um workflow existente
     * @param int $id ID do workflow
     * @param array $data Novos dados
     * @param int $user_id ID do usuário que está atualizando
     * @return array Resultado da atualização
     */
    public function updateWorkflow($id, $data, $user_id) {
        // Verificar se o workflow existe
        if (!$this->workflow->readOne($id)) {
            return [
                'success' => false,
                'message' => 'Workflow não encontrado'
            ];
        }
        
        // Verificar campos obrigatórios
        if (empty($data['name'])) {
            return [
                'success' => false,
                'message' => 'O nome do workflow é obrigatório'
            ];
        }
        
        // Atualizar workflow
        $this->workflow->name = $data['name'];
        $this->workflow->description = $data['description'] ?? $this->workflow->description;
        
        if ($this->workflow->update()) {
            // Registrar log de auditoria
            logAudit($user_id, 'update', 'workflow', $id, 'Workflow atualizado: ' . $data['name']);
            
            return [
                'success' => true,
                'message' => 'Workflow atualizado com sucesso'
            ];
        } else {
            return [
                'success' => false,
                'message' => 'Erro ao atualizar workflow'
            ];
        }
    }
    
    /**
     * Exclui um workflow
     * @param int $id ID do workflow
     * @param int $user_id ID do usuário que está excluindo
     * @return array Resultado da exclusão
     */
    public function deleteWorkflow($id, $user_id) {
        // Verificar se o workflow existe
        if (!$this->workflow->readOne($id)) {
            return [
                'success' => false,
                'message' => 'Workflow não encontrado'
            ];
        }
        
        // Salvar nome para log
        $workflow_name = $this->workflow->name;
        
        // Tentar excluir
        if ($this->workflow->delete()) {
            // Registrar log de auditoria
            logAudit($user_id, 'delete', 'workflow', $id, 'Workflow excluído: ' . $workflow_name);
            
            return [
                'success' => true,
                'message' => 'Workflow excluído com sucesso'
            ];
        } else {
            return [
                'success' => false,
                'message' => 'Não é possível excluir este workflow porque existem documentos associados a ele'
            ];
        }
    }
    
    /**
     * Adiciona uma etapa ao workflow
     * @param int $workflow_id ID do workflow
     * @param array $data Dados da etapa
     * @param int $user_id ID do usuário que está adicionando
     * @return array Resultado da adição
     */
    public function addStep($workflow_id, $data, $user_id) {
        // Verificar se o workflow existe
        if (!$this->workflow->readOne($workflow_id)) {
            return [
                'success' => false,
                'message' => 'Workflow não encontrado'
            ];
        }
        
        // Verificar campos obrigatórios
        if (empty($data['name'])) {
            return [
                'success' => false,
                'message' => 'O nome da etapa é obrigatório'
            ];
        }
        
        // Adicionar etapa
        $step_id = $this->workflow->addStep(
            $data['name'],
            $data['description'] ?? '',
            $data['approver_id'] ?? $user_id
        );
        
        if ($step_id) {
            // Registrar log de auditoria
            logAudit($user_id, 'create', 'workflow_step', $step_id, 'Etapa adicionada ao workflow: ' . $this->workflow->name);
            
            return [
                'success' => true,
                'message' => 'Etapa adicionada com sucesso',
                'step_id' => $step_id
            ];
        } else {
            return [
                'success' => false,
                'message' => 'Erro ao adicionar etapa'
            ];
        }
    }
    
    /**
     * Atualiza uma etapa do workflow
     * @param int $workflow_id ID do workflow
     * @param int $step_id ID da etapa
     * @param array $data Novos dados
     * @param int $user_id ID do usuário que está atualizando
     * @return array Resultado da atualização
     */
    public function updateStep($workflow_id, $step_id, $data, $user_id) {
        // Verificar se o workflow existe
        if (!$this->workflow->readOne($workflow_id)) {
            return [
                'success' => false,
                'message' => 'Workflow não encontrado'
            ];
        }
        
        // Verificar campos obrigatórios
        if (empty($data['name'])) {
            return [
                'success' => false,
                'message' => 'O nome da etapa é obrigatório'
            ];
        }
        
        // Atualizar etapa
        if ($this->workflow->updateStep(
            $step_id,
            $data['name'],
            $data['description'] ?? '',
            $data['approver_id'] ?? $user_id
        )) {
            // Registrar log de auditoria
            logAudit($user_id, 'update', 'workflow_step', $step_id, 'Etapa atualizada: ' . $data['name']);
            
            return [
                'success' => true,
                'message' => 'Etapa atualizada com sucesso'
            ];
        } else {
            return [
                'success' => false,
                'message' => 'Erro ao atualizar etapa'
            ];
        }
    }
    
    /**
     * Remove uma etapa do workflow
     * @param int $workflow_id ID do workflow
     * @param int $step_id ID da etapa
     * @param int $user_id ID do usuário que está removendo
     * @return array Resultado da remoção
     */
    public function removeStep($workflow_id, $step_id, $user_id) {
        // Verificar se o workflow existe
        if (!$this->workflow->readOne($workflow_id)) {
            return [
                'success' => false,
                'message' => 'Workflow não encontrado'
            ];
        }
        
        // Remover etapa
        if ($this->workflow->removeStep($step_id)) {
            // Registrar log de auditoria
            logAudit($user_id, 'delete', 'workflow_step', $step_id, 'Etapa removida do workflow: ' . $this->workflow->name);
            
            return [
                'success' => true,
                'message' => 'Etapa removida com sucesso'
            ];
        } else {
            return [
                'success' => false,
                'message' => 'Erro ao remover etapa'
            ];
        }
    }
    
    /**
     * Reordena as etapas do workflow
     * @param int $workflow_id ID do workflow
     * @param array $step_order Array associativo de ID da etapa => nova ordem
     * @param int $user_id ID do usuário que está reordenando
     * @return array Resultado da reordenação
     */
    public function reorderSteps($workflow_id, $step_order, $user_id) {
        // Verificar se o workflow existe
        if (!$this->workflow->readOne($workflow_id)) {
            return [
                'success' => false,
                'message' => 'Workflow não encontrado'
            ];
        }
        
        // Reordenar etapas
        if ($this->workflow->reorderSteps($step_order)) {
            // Registrar log de auditoria
            logAudit($user_id, 'update', 'workflow', $workflow_id, 'Etapas reordenadas no workflow: ' . $this->workflow->name);
            
            return [
                'success' => true,
                'message' => 'Etapas reordenadas com sucesso'
            ];
        } else {
            return [
                'success' => false,
                'message' => 'Erro ao reordenar etapas'
            ];
        }
    }
    
    /**
     * Inicia um processo de aprovação para um documento
     * @param int $document_id ID do documento
     * @param int $workflow_id ID do workflow
     * @param int $user_id ID do usuário que está iniciando
     * @return array Resultado do início do processo
     */
    public function startApproval($document_id, $workflow_id, $user_id) {
        // Verificar se o documento existe
        $document = new Document($this->db);
        if (!$document->readOne($document_id)) {
            return [
                'success' => false,
                'message' => 'Documento não encontrado'
            ];
        }
        
        // Verificar se o workflow existe
        if (!$this->workflow->readOne($workflow_id)) {
            return [
                'success' => false,
                'message' => 'Workflow não encontrado'
            ];
        }
        
        // Verificar se o documento já está em um processo de aprovação
        $query = "SELECT id FROM document_approval WHERE document_id = ? AND status = 'pending'";
        $stmt = $this->db->prepare($query);
        $stmt->bindParam(1, $document_id);
        $stmt->execute();
        
        if ($stmt->rowCount() > 0) {
            return [
                'success' => false,
                'message' => 'Este documento já está em um processo de aprovação'
            ];
        }
        
        // Verificar se o workflow tem etapas
        $steps = $this->workflow->getSteps();
        if (empty($steps)) {
            return [
                'success' => false,
                'message' => 'Este workflow não possui etapas definidas'
            ];
        }
        
        // Iniciar processo de aprovação
        $query = "INSERT INTO document_approval (document_id, workflow_id, current_step, status)
                  VALUES (?, ?, 1, 'pending')";
        $stmt = $this->db->prepare($query);
        $stmt->bindParam(1, $document_id);
        $stmt->bindParam(2, $workflow_id);
        
        if ($stmt->execute()) {
            $approval_id = $this->db->lastInsertId();
            
            // Atualizar status do documento
            $document->status = 'in_review';
            $document->update();
            
            // Registrar log de auditoria
            logAudit($user_id, 'create', 'document_approval', $approval_id, 'Processo de aprovação iniciado para documento: ' . $document->title);
            
            return [
                'success' => true,
                'message' => 'Processo de aprovação iniciado com sucesso',
                'approval_id' => $approval_id
            ];
        } else {
            return [
                'success' => false,
                'message' => 'Erro ao iniciar processo de aprovação'
            ];
        }
    }
    
    /**
     * Aprova uma etapa do processo de aprovação
     * @param int $approval_id ID do processo de aprovação
     * @param string $comments Comentários opcionais
     * @param int $user_id ID do usuário que está aprovando
     * @return array Resultado da aprovação
     */
    public function approveStep($approval_id, $comments, $user_id) {
        // Obter informações do processo de aprovação
        $query = "SELECT da.*, d.title as document_title, d.id as document_id, w.id as workflow_id
                  FROM document_approval da
                  JOIN document d ON da.document_id = d.id
                  JOIN workflow w ON da.workflow_id = w.id
                  WHERE da.id = ?";
        $stmt = $this->db->prepare($query);
        $stmt->bindParam(1, $approval_id);
        $stmt->execute();
        
        if ($stmt->rowCount() == 0) {
            return [
                'success' => false,
                'message' => 'Processo de aprovação não encontrado'
            ];
        }
        
        $approval = $stmt->fetch(PDO::FETCH_ASSOC);
        
        // Verificar se o processo está pendente
        if ($approval['status'] !== 'pending') {
            return [
                'success' => false,
                'message' => 'Este processo de aprovação já foi concluído'
            ];
        }
        
        // Verificar se o usuário é o aprovador da etapa atual
        $this->workflow->readOne($approval['workflow_id']);
        $steps = $this->workflow->getSteps();
        
        $current_step = null;
        foreach ($steps as $step) {
            if ($step['step_order'] == $approval['current_step']) {
                $current_step = $step;
                break;
            }
        }
        
        if (!$current_step) {
            return [
                'success' => false,
                'message' => 'Etapa atual não encontrada'
            ];
        }
        
        // Verificar se o usuário é o aprovador ou um administrador
        if ($current_step['approver_id'] != $user_id && !hasPermission('workflow_manage')) {
            return [
                'success' => false,
                'message' => 'Você não tem permissão para aprovar esta etapa'
            ];
        }
        
        // Registrar histórico de aprovação
        $query = "INSERT INTO approval_history (document_approval_id, step_id, user_id, action, comments)
                  VALUES (?, ?, ?, 'approved', ?)";
        $stmt = $this->db->prepare($query);
        $stmt->bindParam(1, $approval_id);
        $stmt->bindParam(2, $current_step['id']);
        $stmt->bindParam(3, $user_id);
        $stmt->bindParam(4, $comments);
        $stmt->execute();
        
        // Verificar se é a última etapa
        $is_last_step = true;
        foreach ($steps as $step) {
            if ($step['step_order'] > $approval['current_step']) {
                $is_last_step = false;
                break;
            }
        }
        
        if ($is_last_step) {
            // Concluir processo de aprovação
            $query = "UPDATE document_approval
                      SET status = 'approved', completed_at = NOW()
                      WHERE id = ?";
            $stmt = $this->db->prepare($query);
            $stmt->bindParam(1, $approval_id);
            $stmt->execute();
            
            // Atualizar status do documento
            $document = new Document($this->db);
            $document->readOne($approval['document_id']);
            $document->status = 'approved';
            $document->update();
            
            // Registrar log de auditoria
            logAudit($user_id, 'update', 'document_approval', $approval_id, 'Documento aprovado: ' . $approval['document_title']);
            
            return [
                'success' => true,
                'message' => 'Documento aprovado com sucesso',
                'is_completed' => true
            ];
        } else {
            // Avançar para a próxima etapa
            $query = "UPDATE document_approval
                      SET current_step = current_step + 1
                      WHERE id = ?";
            $stmt = $this->db->prepare($query);
            $stmt->bindParam(1, $approval_id);
            $stmt->execute();
            
            // Registrar log de auditoria
            logAudit($user_id, 'update', 'document_approval', $approval_id, 'Etapa aprovada: ' . $current_step['name']);
            
            return [
                'success' => true,
                'message' => 'Etapa aprovada com sucesso',
                'is_completed' => false,
                'next_step' => $approval['current_step'] + 1
            ];
        }
    }
    
    /**
     * Rejeita um processo de aprovação
     * @param int $approval_id ID do processo de aprovação
     * @param string $comments Comentários obrigatórios
     * @param int $user_id ID do usuário que está rejeitando
     * @return array Resultado da rejeição
     */
    public function rejectApproval($approval_id, $comments, $user_id) {
        // Verificar comentários
        if (empty($comments)) {
            return [
                'success' => false,
                'message' => 'Comentários são obrigatórios para rejeição'
            ];
        }
        
        // Obter informações do processo de aprovação
        $query = "SELECT da.*, d.title as document_title, d.id as document_id, w.id as workflow_id
                  FROM document_approval da
                  JOIN document d ON da.document_id = d.id
                  JOIN workflow w ON da.workflow_id = w.id
                  WHERE da.id = ?";
        $stmt = $this->db->prepare($query);
        $stmt->bindParam(1, $approval_id);
        $stmt->execute();
        
        if ($stmt->rowCount() == 0) {
            return [
                'success' => false,
                'message' => 'Processo de aprovação não encontrado'
            ];
        }
        
        $approval = $stmt->fetch(PDO::FETCH_ASSOC);
        
        // Verificar se o processo está pendente
        if ($approval['status'] !== 'pending') {
            return [
                'success' => false,
                'message' => 'Este processo de aprovação já foi concluído'
            ];
        }
        
        // Verificar se o usuário é o aprovador da etapa atual
        $this->workflow->readOne($approval['workflow_id']);
        $steps = $this->workflow->getSteps();
        
        $current_step = null;
        foreach ($steps as $step) {
            if ($step['step_order'] == $approval['current_step']) {
                $current_step = $step;
                break;
            }
        }
        
        if (!$current_step) {
            return [
                'success' => false,
                'message' => 'Etapa atual não encontrada'
            ];
        }
        
        // Verificar se o usuário é o aprovador ou um administrador
        if ($current_step['approver_id'] != $user_id && !hasPermission('workflow_manage')) {
            return [
                'success' => false,
                'message' => 'Você não tem permissão para rejeitar esta etapa'
            ];
        }
        
        // Registrar histórico de rejeição
        $query = "INSERT INTO approval_history (document_approval_id, step_id, user_id, action, comments)
                  VALUES (?, ?, ?, 'rejected', ?)";
        $stmt = $this->db->prepare($query);
        $stmt->bindParam(1, $approval_id);
        $stmt->bindParam(2, $current_step['id']);
        $stmt->bindParam(3, $user_id);
        $stmt->bindParam(4, $comments);
        $stmt->execute();
        
        // Concluir processo de aprovação como rejeitado
        $query = "UPDATE document_approval
                  SET status = 'rejected', completed_at = NOW()
                  WHERE id = ?";
        $stmt = $this->db->prepare($query);
        $stmt->bindParam(1, $approval_id);
        $stmt->execute();
        
        // Atualizar status do documento
        $document = new Document($this->db);
        $document->readOne($approval['document_id']);
        $document->status = 'draft'; // Voltar para rascunho
        $document->update();
        
        // Registrar log de auditoria
        logAudit($user_id, 'update', 'document_approval', $approval_id, 'Documento rejeitado: ' . $approval['document_title']);
        
        return [
            'success' => true,
            'message' => 'Documento rejeitado com sucesso'
        ];
    }
    
    /**
     * Obtém o histórico de aprovação de um documento
     * @param int $document_id ID do documento
     * @return array Histórico de aprovação
     */
    public function getApprovalHistory($document_id) {
        $query = "SELECT da.*, w.name as workflow_name
                  FROM document_approval da
                  JOIN workflow w ON da.workflow_id = w.id
                  WHERE da.document_id = ?
                  ORDER BY da.started_at DESC";
        $stmt = $this->db->prepare($query);
        $stmt->bindParam(1, $document_id);
        $stmt->execute();
        
        $approvals = [];
        while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
            // Obter histórico de cada aprovação
            $history_query = "SELECT ah.*, ws.name as step_name, u.username
                              FROM approval_history ah
                              JOIN workflow_step ws ON ah.step_id = ws.id
                              JOIN user u ON ah.user_id = u.id
                              WHERE ah.document_approval_id = ?
                              ORDER BY ah.created_at ASC";
            $history_stmt = $this->db->prepare($history_query);
            $history_stmt->bindParam(1, $row['id']);
            $history_stmt->execute();
            
            $history = [];
            while ($history_row = $history_stmt->fetch(PDO::FETCH_ASSOC)) {
                $history[] = $history_row;
            }
            
            $row['history'] = $history;
            $approvals[] = $row;
        }
        
        return [
            'success' => true,
            'approvals' => $approvals
        ];
    }
    
    /**
     * Lista documentos pendentes de aprovação para um usuário
     * @param int $user_id ID do usuário
     * @param int $page Página atual
     * @param int $per_page Itens por página
     * @return array Lista de documentos pendentes
     */
    public function getPendingApprovals($user_id, $page = 1, $per_page = 10) {
        // Construir consulta
        $query = "SELECT da.*, d.title as document_title, d.file_path, d.file_type,
                         w.name as workflow_name, ws.name as step_name
                  FROM document_approval da
                  JOIN document d ON da.document_id = d.id
                  JOIN workflow w ON da.workflow_id = w.id
                  JOIN workflow_step ws ON ws.workflow_id = w.id AND ws.step_order = da.current_step
                  WHERE da.status = 'pending' AND ws.approver_id = ?";
        
        // Contar total de resultados para paginação
        $count_query = str_replace("SELECT da.*, d.title as document_title, d.file_path, d.file_type,
                         w.name as workflow_name, ws.name as step_name", "SELECT COUNT(*) as total", $query);
        $stmt = $this->db->prepare($count_query);
        $stmt->bindParam(1, $user_id);
        $stmt->execute();
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        $total = $row['total'];
        
        // Adicionar paginação
        $offset = ($page - 1) * $per_page;
        $query .= " ORDER BY da.started_at ASC LIMIT ?, ?";
        
        $stmt = $this->db->prepare($query);
        $stmt->bindParam(1, $user_id);
        $stmt->bindParam(2, $offset, PDO::PARAM_INT);
        $stmt->bindParam(3, $per_page, PDO::PARAM_INT);
        $stmt->execute();
        
        $approvals = [];
        while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
            $approvals[] = $row;
        }
        
        // Calcular total de páginas
        $total_pages = ceil($total / $per_page);
        
        return [
            'success' => true,
            'approvals' => $approvals,
            'pagination' => [
                'total' => $total,
                'per_page' => $per_page,
                'current_page' => $page,
                'total_pages' => $total_pages
            ]
        ];
    }
}
?>
