<?php
/**
 * Página de visualização de documento
 * Sistema de Gerenciamento Eletrônico de Documentos (GED)
 */

// Iniciar sessão e verificar autenticação
session_start();
if (!isset($_SESSION['user_id'])) {
    header('Location: login.php');
    exit;
}

// Verificar se o ID do documento foi fornecido
if (!isset($_GET['id']) || empty($_GET['id'])) {
    $_SESSION['flash_message'] = "ID do documento não fornecido.";
    $_SESSION['flash_type'] = "danger";
    header('Location: documents.php');
    exit;
}

$document_id = $_GET['id'];

// Incluir arquivos necessários
require_once 'config/database.php';
require_once 'models/document.php';
require_once 'models/document_version.php';
require_once 'models/workflow.php';
require_once 'models/audit_log.php';
require_once 'controllers/document_controller.php';

// Conectar ao banco de dados
$database = new Database();
$db = $database->getConnection();

// Instanciar controladores e modelos
$document_controller = new DocumentController();
$document = new Document($db);
$document_version = new DocumentVersion($db);
$workflow = new Workflow($db);
$audit_log = new AuditLog($db);

// Obter detalhes do documento
$result = $document_controller->getDocument($document_id);

if (!$result['success']) {
    $_SESSION['flash_message'] = $result['message'];
    $_SESSION['flash_type'] = "danger";
    header('Location: documents.php');
    exit;
}

$doc = $result['document'];

// Verificar permissão para visualizar o documento
$user_id = $_SESSION['user_id'];
$user_type = $_SESSION['user_type'];

if ($doc['author_id'] != $user_id && $user_type != 'admin' && !$document_controller->userCanAccessDocument($user_id, $document_id)) {
    $_SESSION['flash_message'] = "Você não tem permissão para visualizar este documento.";
    $_SESSION['flash_type'] = "danger";
    header('Location: documents.php');
    exit;
}

// Obter versões do documento
$versions = $document_version->getVersionsByDocumentId($document_id);

// Obter histórico de aprovações
$approval_history = $workflow->getDocumentApprovalHistory($document_id);

// Obter histórico de atividade
$activity_history = $audit_log->getDocumentHistory($document_id);

// Registrar visualização nos logs
logAudit($user_id, 'view', 'document', $document_id, "Visualizou o documento: " . $doc['title']);

// Definir título da página
$page_title = "Visualizar Documento";

// Incluir cabeçalho
include 'includes/header.php';
?>

<div class="row mb-3">
    <div class="col-md-8">
        <h1><i class="fas fa-file-alt me-2"></i> <?php echo htmlspecialchars($doc['title']); ?></h1>
    </div>
    <div class="col-md-4 text-md-end">
        <div class="btn-group">
            <a href="documents.php" class="btn btn-secondary">
                <i class="fas fa-arrow-left me-1"></i> Voltar
            </a>
            <a href="document_download.php?id=<?php echo $doc['id']; ?>" class="btn btn-success">
                <i class="fas fa-download me-1"></i> Download
            </a>
            <?php if ($doc['author_id'] == $user_id || $user_type == 'admin'): ?>
            <a href="document_edit.php?id=<?php echo $doc['id']; ?>" class="btn btn-primary">
                <i class="fas fa-edit me-1"></i> Editar
            </a>
            <?php endif; ?>
        </div>
    </div>
</div>

<div class="row">
    <!-- Detalhes do documento -->
    <div class="col-md-8">
        <div class="card mb-4">
            <div class="card-header">
                <ul class="nav nav-tabs card-header-tabs" role="tablist">
                    <li class="nav-item" role="presentation">
                        <button class="nav-link active" id="details-tab" data-bs-toggle="tab" data-bs-target="#details" type="button" role="tab" aria-controls="details" aria-selected="true">
                            <i class="fas fa-info-circle me-1"></i> Detalhes
                        </button>
                    </li>
                    <li class="nav-item" role="presentation">
                        <button class="nav-link" id="versions-tab" data-bs-toggle="tab" data-bs-target="#versions" type="button" role="tab" aria-controls="versions" aria-selected="false">
                            <i class="fas fa-code-branch me-1"></i> Versões <span class="badge bg-secondary"><?php echo count($versions); ?></span>
                        </button>
                    </li>
                    <li class="nav-item" role="presentation">
                        <button class="nav-link" id="approvals-tab" data-bs-toggle="tab" data-bs-target="#approvals" type="button" role="tab" aria-controls="approvals" aria-selected="false">
                            <i class="fas fa-check-circle me-1"></i> Aprovações
                        </button>
                    </li>
                    <li class="nav-item" role="presentation">
                        <button class="nav-link" id="history-tab" data-bs-toggle="tab" data-bs-target="#history" type="button" role="tab" aria-controls="history" aria-selected="false">
                            <i class="fas fa-history me-1"></i> Histórico
                        </button>
                    </li>
                </ul>
            </div>
            <div class="card-body">
                <div class="tab-content">
                    <!-- Aba de detalhes -->
                    <div class="tab-pane fade show active" id="details" role="tabpanel" aria-labelledby="details-tab">
                        <div class="row mb-3">
                            <div class="col-md-6">
                                <p><strong>Categoria:</strong> <?php echo htmlspecialchars($doc['category_name']); ?></p>
                                <p><strong>Autor:</strong> <?php echo htmlspecialchars($doc['author_name']); ?></p>
                                <p><strong>Data de Criação:</strong> <?php echo date('d/m/Y H:i', strtotime($doc['created_at'])); ?></p>
                                <p><strong>Última Atualização:</strong> <?php echo date('d/m/Y H:i', strtotime($doc['updated_at'])); ?></p>
                            </div>
                            <div class="col-md-6">
                                <p>
                                    <strong>Status:</strong>
                                    <?php if ($doc['status'] == 'approved'): ?>
                                    <span class="badge bg-success">Aprovado</span>
                                    <?php elseif ($doc['status'] == 'pending'): ?>
                                    <span class="badge bg-warning text-dark">Pendente</span>
                                    <?php elseif ($doc['status'] == 'rejected'): ?>
                                    <span class="badge bg-danger">Rejeitado</span>
                                    <?php else: ?>
                                    <span class="badge bg-secondary">Rascunho</span>
                                    <?php endif; ?>
                                </p>
                                <p><strong>Versão Atual:</strong> <?php echo $doc['version']; ?></p>
                                <p><strong>Tamanho:</strong> <?php echo formatFileSize($doc['file_size']); ?></p>
                                <p><strong>Tipo de Arquivo:</strong> <?php echo strtoupper($doc['file_type']); ?></p>
                            </div>
                        </div>
                        
                        <?php if (!empty($doc['description'])): ?>
                        <div class="mb-3">
                            <h5>Descrição</h5>
                            <p><?php echo nl2br(htmlspecialchars($doc['description'])); ?></p>
                        </div>
                        <?php endif; ?>
                        
                        <?php if (!empty($doc['keywords'])): ?>
                        <div class="mb-3">
                            <h5>Palavras-chave</h5>
                            <?php 
                            $keywords = explode(',', $doc['keywords']);
                            foreach ($keywords as $keyword): 
                            ?>
                            <span class="badge bg-info text-dark me-1"><?php echo htmlspecialchars(trim($keyword)); ?></span>
                            <?php endforeach; ?>
                        </div>
                        <?php endif; ?>
                        
                        <?php if ($doc['status'] == 'pending' && ($user_type == 'admin' || $document_controller->userCanApproveDocument($user_id, $document_id))): ?>
                        <div class="mt-4">
                            <h5>Ações de Aprovação</h5>
                            <div class="btn-group">
                                <a href="document_approve.php?id=<?php echo $doc['id']; ?>&action=approve" class="btn btn-success">
                                    <i class="fas fa-check me-1"></i> Aprovar
                                </a>
                                <a href="document_approve.php?id=<?php echo $doc['id']; ?>&action=reject" class="btn btn-danger">
                                    <i class="fas fa-times me-1"></i> Rejeitar
                                </a>
                            </div>
                        </div>
                        <?php endif; ?>
                    </div>
                    
                    <!-- Aba de versões -->
                    <div class="tab-pane fade" id="versions" role="tabpanel" aria-labelledby="versions-tab">
                        <?php if (count($versions) > 0): ?>
                        <div class="table-responsive">
                            <table class="table table-hover">
                                <thead>
                                    <tr>
                                        <th>Versão</th>
                                        <th>Criado por</th>
                                        <th>Data</th>
                                        <th>Comentário</th>
                                        <th>Ações</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($versions as $version): ?>
                                    <tr <?php echo ($version['version'] == $doc['version']) ? 'class="table-primary"' : ''; ?>>
                                        <td><?php echo $version['version']; ?> <?php echo ($version['version'] == $doc['version']) ? '<span class="badge bg-primary">Atual</span>' : ''; ?></td>
                                        <td><?php echo htmlspecialchars($version['created_by_name']); ?></td>
                                        <td><?php echo date('d/m/Y H:i', strtotime($version['created_at'])); ?></td>
                                        <td><?php echo htmlspecialchars($version['comment']); ?></td>
                                        <td>
                                            <div class="btn-group btn-group-sm">
                                                <a href="document_download.php?id=<?php echo $doc['id']; ?>&version=<?php echo $version['version']; ?>" class="btn btn-success" title="Download">
                                                    <i class="fas fa-download"></i>
                                                </a>
                                                <?php if ($version['version'] != $doc['version'] && ($doc['author_id'] == $user_id || $user_type == 'admin')): ?>
                                                <a href="document_restore.php?id=<?php echo $doc['id']; ?>&version=<?php echo $version['version']; ?>" class="btn btn-warning" title="Restaurar" onclick="return confirm('Tem certeza que deseja restaurar esta versão?');">
                                                    <i class="fas fa-undo"></i>
                                                </a>
                                                <?php endif; ?>
                                            </div>
                                        </td>
                                    </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                        <?php else: ?>
                        <p class="text-muted">Nenhuma versão encontrada.</p>
                        <?php endif; ?>
                    </div>
                    
                    <!-- Aba de aprovações -->
                    <div class="tab-pane fade" id="approvals" role="tabpanel" aria-labelledby="approvals-tab">
                        <?php if (count($approval_history) > 0): ?>
                        <div class="table-responsive">
                            <table class="table table-hover">
                                <thead>
                                    <tr>
                                        <th>Data</th>
                                        <th>Usuário</th>
                                        <th>Ação</th>
                                        <th>Comentário</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($approval_history as $approval): ?>
                                    <tr>
                                        <td><?php echo date('d/m/Y H:i', strtotime($approval['created_at'])); ?></td>
                                        <td><?php echo htmlspecialchars($approval['username']); ?></td>
                                        <td>
                                            <?php if ($approval['action'] == 'approve'): ?>
                                            <span class="badge bg-success">Aprovado</span>
                                            <?php elseif ($approval['action'] == 'reject'): ?>
                                            <span class="badge bg-danger">Rejeitado</span>
                                            <?php elseif ($approval['action'] == 'request'): ?>
                                            <span class="badge bg-info">Solicitado</span>
                                            <?php else: ?>
                                            <span class="badge bg-secondary"><?php echo htmlspecialchars($approval['action']); ?></span>
                                            <?php endif; ?>
                                        </td>
                                        <td><?php echo htmlspecialchars($approval['comment']); ?></td>
                                    </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                        <?php else: ?>
                        <p class="text-muted">Nenhum histórico de aprovação encontrado.</p>
                        <?php endif; ?>
                    </div>
                    
                    <!-- Aba de histórico -->
                    <div class="tab-pane fade" id="history" role="tabpanel" aria-labelledby="history-tab">
                        <?php if (count($activity_history) > 0): ?>
                        <div class="timeline">
                            <?php foreach ($activity_history as $activity): ?>
                            <div class="timeline-item">
                                <div class="timeline-marker"></div>
                                <div class="timeline-content">
                                    <h6 class="mb-0"><?php echo htmlspecialchars($activity['username']); ?></h6>
                                    <p class="mb-0"><?php echo htmlspecialchars($activity['details']); ?></p>
                                    <small class="text-muted"><?php echo date('d/m/Y H:i', strtotime($activity['created_at'])); ?></small>
                                </div>
                            </div>
                            <?php endforeach; ?>
                        </div>
                        <?php else: ?>
                        <p class="text-muted">Nenhum histórico de atividade encontrado.</p>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Visualização do documento -->
    <div class="col-md-4">
        <div class="card mb-4">
            <div class="card-header">
                <h5 class="mb-0"><i class="fas fa-eye me-2"></i> Visualização</h5>
            </div>
            <div class="card-body text-center">
                <?php
                $file_extension = pathinfo($doc['file_path'], PATHINFO_EXTENSION);
                $preview_supported = in_array(strtolower($file_extension), ['pdf', 'jpg', 'jpeg', 'png', 'gif']);
                
                if ($preview_supported):
                    if (in_array(strtolower($file_extension), ['jpg', 'jpeg', 'png', 'gif'])):
                ?>
                <img src="document_preview.php?id=<?php echo $doc['id']; ?>" class="img-fluid mb-3" alt="Visualização do documento">
                <?php elseif (strtolower($file_extension) == 'pdf'): ?>
                <div class="ratio ratio-4x3">
                    <iframe src="document_preview.php?id=<?php echo $doc['id']; ?>" allowfullscreen></iframe>
                </div>
                <?php endif; ?>
                <?php else: ?>
                <div class="document-icon mb-3">
                    <i class="fas fa-file-alt fa-5x text-primary"></i>
                </div>
                <p class="text-muted">Visualização não disponível para este tipo de arquivo.</p>
                <?php endif; ?>
                
                <a href="document_download.php?id=<?php echo $doc['id']; ?>" class="btn btn-primary">
                    <i class="fas fa-download me-1"></i> Download (<?php echo formatFileSize($doc['file_size']); ?>)
                </a>
            </div>
        </div>
        
        <!-- Documentos relacionados -->
        <?php
        $related_documents = $document_controller->getRelatedDocuments($document_id, 5);
        if (count($related_documents) > 0):
        ?>
        <div class="card">
            <div class="card-header">
                <h5 class="mb-0"><i class="fas fa-link me-2"></i> Documentos Relacionados</h5>
            </div>
            <div class="list-group list-group-flush">
                <?php foreach ($related_documents as $related): ?>
                <a href="document_view.php?id=<?php echo $related['id']; ?>" class="list-group-item list-group-item-action">
                    <div class="d-flex w-100 justify-content-between">
                        <h6 class="mb-1"><?php echo htmlspecialchars($related['title']); ?></h6>
                        <small>
                            <?php if ($related['status'] == 'approved'): ?>
                            <span class="badge bg-success">Aprovado</span>
                            <?php elseif ($related['status'] == 'pending'): ?>
                            <span class="badge bg-warning text-dark">Pendente</span>
                            <?php elseif ($related['status'] == 'rejected'): ?>
                            <span class="badge bg-danger">Rejeitado</span>
                            <?php else: ?>
                            <span class="badge bg-secondary">Rascunho</span>
                            <?php endif; ?>
                        </small>
                    </div>
                    <small class="text-muted">
                        <i class="fas fa-folder me-1"></i> <?php echo htmlspecialchars($related['category_name']); ?>
                    </small>
                </a>
                <?php endforeach; ?>
            </div>
        </div>
        <?php endif; ?>
    </div>
</div>

<!-- CSS para timeline -->
<style>
.timeline {
    position: relative;
    padding-left: 30px;
}
.timeline-item {
    position: relative;
    padding-bottom: 20px;
}
.timeline-marker {
    position: absolute;
    left: -30px;
    width: 15px;
    height: 15px;
    border-radius: 50%;
    background-color: var(--bs-primary);
}
.timeline-marker:before {
    content: '';
    position: absolute;
    left: 7px;
    height: calc(100% + 20px);
    width: 1px;
    background-color: var(--bs-border-color);
}
.timeline-item:last-child .timeline-marker:before {
    display: none;
}
</style>

<?php
/**
 * Função para formatar tamanho de arquivo
 * @param int $size Tamanho em bytes
 * @return string Tamanho formatado
 */
function formatFileSize($size) {
    $units = ['B', 'KB', 'MB', 'GB', 'TB'];
    $i = 0;
    while ($size >= 1024 && $i < count($units) - 1) {
        $size /= 1024;
        $i++;
    }
    return round($size, 2) . ' ' . $units[$i];
}

// Incluir rodapé
include 'includes/footer.php';
?>
