<?php
/**
 * Página inicial - Dashboard
 * Sistema de Gerenciamento Eletrônico de Documentos (GED)
 */

// Iniciar sessão e verificar autenticação
session_start();
if (!isset($_SESSION['user_id'])) {
    header('Location: login.php');
    exit;
}

// Incluir arquivos necessários
require_once 'config/database.php';
require_once 'models/document.php';
require_once 'models/category.php';
require_once 'models/workflow.php';
require_once 'models/audit_log.php';

// Conectar ao banco de dados
$database = new Database();
$db = $database->getConnection();

// Instanciar modelos
$document = new Document($db);
$category = new Category($db);
$workflow = new Workflow($db);
$audit_log = new AuditLog($db);

// Obter estatísticas para o dashboard
$user_id = $_SESSION['user_id'];
$user_type = $_SESSION['user_type'];

// Total de documentos
$total_documents = $document->countDocuments();

// Documentos por categoria
$documents_by_category = $category->getDocumentCountByCategory();

// Documentos recentes
$recent_documents = $document->getRecentDocuments(5);

// Documentos pendentes de aprovação para o usuário atual
$pending_approvals = $workflow->getPendingApprovalsForUser($user_id);

// Atividade recente (logs)
$recent_activity = $audit_log->readAll([], 1, 10)['logs'];

// Definir título da página
$page_title = "Dashboard";

// Incluir cabeçalho
include 'includes/header.php';
?>

<div class="row">
    <div class="col-12">
        <h1 class="mb-4">
            <i class="fas fa-tachometer-alt me-2"></i> Dashboard
        </h1>
    </div>
</div>

<!-- Cards de estatísticas -->
<div class="row mb-4">
    <div class="col-md-3 mb-3">
        <div class="card h-100 border-primary">
            <div class="card-body">
                <div class="d-flex justify-content-between align-items-center">
                    <div>
                        <h6 class="card-title text-muted">Total de Documentos</h6>
                        <h2 class="mb-0"><?php echo $total_documents; ?></h2>
                    </div>
                    <div class="bg-primary bg-opacity-10 p-3 rounded">
                        <i class="fas fa-file-alt fa-2x text-primary"></i>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <div class="col-md-3 mb-3">
        <div class="card h-100 border-success">
            <div class="card-body">
                <div class="d-flex justify-content-between align-items-center">
                    <div>
                        <h6 class="card-title text-muted">Documentos Aprovados</h6>
                        <h2 class="mb-0"><?php echo $document->countDocumentsByStatus('approved'); ?></h2>
                    </div>
                    <div class="bg-success bg-opacity-10 p-3 rounded">
                        <i class="fas fa-check-circle fa-2x text-success"></i>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <div class="col-md-3 mb-3">
        <div class="card h-100 border-warning">
            <div class="card-body">
                <div class="d-flex justify-content-between align-items-center">
                    <div>
                        <h6 class="card-title text-muted">Pendentes de Aprovação</h6>
                        <h2 class="mb-0"><?php echo $document->countDocumentsByStatus('pending'); ?></h2>
                    </div>
                    <div class="bg-warning bg-opacity-10 p-3 rounded">
                        <i class="fas fa-clock fa-2x text-warning"></i>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <div class="col-md-3 mb-3">
        <div class="card h-100 border-info">
            <div class="card-body">
                <div class="d-flex justify-content-between align-items-center">
                    <div>
                        <h6 class="card-title text-muted">Categorias</h6>
                        <h2 class="mb-0"><?php echo $category->countCategories(); ?></h2>
                    </div>
                    <div class="bg-info bg-opacity-10 p-3 rounded">
                        <i class="fas fa-tags fa-2x text-info"></i>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<div class="row">
    <!-- Documentos recentes -->
    <div class="col-md-6 mb-4">
        <div class="card h-100">
            <div class="card-header bg-transparent">
                <div class="d-flex justify-content-between align-items-center">
                    <h5 class="mb-0"><i class="fas fa-file-alt me-2"></i> Documentos Recentes</h5>
                    <a href="documents.php" class="btn btn-sm btn-outline-primary">Ver Todos</a>
                </div>
            </div>
            <div class="card-body">
                <?php if (count($recent_documents) > 0): ?>
                <div class="table-responsive">
                    <table class="table table-hover">
                        <thead>
                            <tr>
                                <th>Título</th>
                                <th>Categoria</th>
                                <th>Status</th>
                                <th>Data</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($recent_documents as $doc): ?>
                            <tr>
                                <td>
                                    <a href="document_view.php?id=<?php echo $doc['id']; ?>">
                                        <?php echo htmlspecialchars($doc['title']); ?>
                                    </a>
                                </td>
                                <td><?php echo htmlspecialchars($doc['category_name']); ?></td>
                                <td>
                                    <?php if ($doc['status'] == 'approved'): ?>
                                    <span class="badge bg-success">Aprovado</span>
                                    <?php elseif ($doc['status'] == 'pending'): ?>
                                    <span class="badge bg-warning text-dark">Pendente</span>
                                    <?php elseif ($doc['status'] == 'rejected'): ?>
                                    <span class="badge bg-danger">Rejeitado</span>
                                    <?php else: ?>
                                    <span class="badge bg-secondary">Rascunho</span>
                                    <?php endif; ?>
                                </td>
                                <td><?php echo date('d/m/Y', strtotime($doc['created_at'])); ?></td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
                <?php else: ?>
                <p class="text-muted text-center my-4">Nenhum documento encontrado.</p>
                <?php endif; ?>
            </div>
            <div class="card-footer bg-transparent">
                <a href="document_add.php" class="btn btn-primary">
                    <i class="fas fa-plus me-1"></i> Novo Documento
                </a>
            </div>
        </div>
    </div>
    
    <!-- Aprovações pendentes -->
    <div class="col-md-6 mb-4">
        <div class="card h-100">
            <div class="card-header bg-transparent">
                <div class="d-flex justify-content-between align-items-center">
                    <h5 class="mb-0"><i class="fas fa-tasks me-2"></i> Aprovações Pendentes</h5>
                    <a href="pending_approvals.php" class="btn btn-sm btn-outline-primary">Ver Todas</a>
                </div>
            </div>
            <div class="card-body">
                <?php if (count($pending_approvals) > 0): ?>
                <div class="table-responsive">
                    <table class="table table-hover">
                        <thead>
                            <tr>
                                <th>Documento</th>
                                <th>Solicitante</th>
                                <th>Data</th>
                                <th>Ações</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($pending_approvals as $approval): ?>
                            <tr>
                                <td>
                                    <a href="document_view.php?id=<?php echo $approval['document_id']; ?>">
                                        <?php echo htmlspecialchars($approval['document_title']); ?>
                                    </a>
                                </td>
                                <td><?php echo htmlspecialchars($approval['requester_name']); ?></td>
                                <td><?php echo date('d/m/Y', strtotime($approval['created_at'])); ?></td>
                                <td>
                                    <div class="btn-group btn-group-sm">
                                        <a href="document_approve.php?id=<?php echo $approval['id']; ?>&action=approve" class="btn btn-success" title="Aprovar">
                                            <i class="fas fa-check"></i>
                                        </a>
                                        <a href="document_approve.php?id=<?php echo $approval['id']; ?>&action=reject" class="btn btn-danger" title="Rejeitar">
                                            <i class="fas fa-times"></i>
                                        </a>
                                    </div>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
                <?php else: ?>
                <p class="text-muted text-center my-4">Nenhuma aprovação pendente.</p>
                <?php endif; ?>
            </div>
        </div>
    </div>
</div>

<div class="row">
    <!-- Gráfico de documentos por categoria -->
    <div class="col-md-6 mb-4">
        <div class="card h-100">
            <div class="card-header bg-transparent">
                <h5 class="mb-0"><i class="fas fa-chart-pie me-2"></i> Documentos por Categoria</h5>
            </div>
            <div class="card-body">
                <canvas id="categoryChart" height="250"></canvas>
            </div>
        </div>
    </div>
    
    <!-- Atividade recente -->
    <div class="col-md-6 mb-4">
        <div class="card h-100">
            <div class="card-header bg-transparent">
                <h5 class="mb-0"><i class="fas fa-history me-2"></i> Atividade Recente</h5>
            </div>
            <div class="card-body">
                <?php if (count($recent_activity) > 0): ?>
                <div class="timeline">
                    <?php foreach ($recent_activity as $activity): ?>
                    <div class="timeline-item">
                        <div class="timeline-marker"></div>
                        <div class="timeline-content">
                            <h6 class="mb-0"><?php echo htmlspecialchars($activity['username']); ?></h6>
                            <p class="mb-0"><?php echo htmlspecialchars($activity['details']); ?></p>
                            <small class="text-muted"><?php echo date('d/m/Y H:i', strtotime($activity['created_at'])); ?></small>
                        </div>
                    </div>
                    <?php endforeach; ?>
                </div>
                <?php else: ?>
                <p class="text-muted text-center my-4">Nenhuma atividade recente.</p>
                <?php endif; ?>
            </div>
        </div>
    </div>
</div>

<!-- Chart.js para gráficos -->
<script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
<script>
document.addEventListener('DOMContentLoaded', function() {
    // Dados para o gráfico de categorias
    const categoryData = <?php echo json_encode($documents_by_category); ?>;
    
    // Preparar dados para Chart.js
    const labels = categoryData.map(item => item.name);
    const data = categoryData.map(item => item.count);
    
    // Gerar cores aleatórias
    const backgroundColors = labels.map(() => {
        const r = Math.floor(Math.random() * 255);
        const g = Math.floor(Math.random() * 255);
        const b = Math.floor(Math.random() * 255);
        return `rgba(${r}, ${g}, ${b}, 0.7)`;
    });
    
    // Criar gráfico
    const ctx = document.getElementById('categoryChart').getContext('2d');
    new Chart(ctx, {
        type: 'pie',
        data: {
            labels: labels,
            datasets: [{
                data: data,
                backgroundColor: backgroundColors,
                borderWidth: 1
            }]
        },
        options: {
            responsive: true,
            maintainAspectRatio: false,
            plugins: {
                legend: {
                    position: 'right'
                }
            }
        }
    });
});
</script>

<!-- CSS para timeline -->
<style>
.timeline {
    position: relative;
    padding-left: 30px;
}
.timeline-item {
    position: relative;
    padding-bottom: 20px;
}
.timeline-marker {
    position: absolute;
    left: -30px;
    width: 15px;
    height: 15px;
    border-radius: 50%;
    background-color: var(--bs-primary);
}
.timeline-marker:before {
    content: '';
    position: absolute;
    left: 7px;
    height: calc(100% + 20px);
    width: 1px;
    background-color: var(--bs-border-color);
}
.timeline-item:last-child .timeline-marker:before {
    display: none;
}
</style>

<?php
// Incluir rodapé
include 'includes/footer.php';
?>
