<?php
/**
 * Modelo de Documento
 * Sistema de Gerenciamento Eletrônico de Documentos (GED)
 */

class Document {
    // Conexão com o banco e nome da tabela
    private $conn;
    private $table_name = "document";

    // Propriedades do objeto
    public $id;
    public $title;
    public $author;
    public $file_path;
    public $file_type;
    public $file_size;
    public $category_id;
    public $status;
    public $created_at;
    public $updated_at;
    public $created_by;

    /**
     * Construtor com $db como conexão com o banco de dados
     * @param PDO $db Conexão com o banco de dados
     */
    public function __construct($db) {
        $this->conn = $db;
    }

    /**
     * Cria um novo documento
     * @return bool Verdadeiro se criado com sucesso, falso caso contrário
     */
    public function create() {
        $query = "INSERT INTO " . $this->table_name . "
                  (title, author, file_path, file_type, file_size, category_id, status, created_by)
                  VALUES (?, ?, ?, ?, ?, ?, ?, ?)";
        
        $stmt = $this->conn->prepare($query);
        
        // Sanitizar
        $this->title = htmlspecialchars(strip_tags($this->title));
        $this->author = htmlspecialchars(strip_tags($this->author));
        $this->file_path = htmlspecialchars(strip_tags($this->file_path));
        $this->file_type = htmlspecialchars(strip_tags($this->file_type));
        $this->category_id = htmlspecialchars(strip_tags($this->category_id));
        $this->status = htmlspecialchars(strip_tags($this->status));
        
        // Vincular valores
        $stmt->bindParam(1, $this->title);
        $stmt->bindParam(2, $this->author);
        $stmt->bindParam(3, $this->file_path);
        $stmt->bindParam(4, $this->file_type);
        $stmt->bindParam(5, $this->file_size);
        $stmt->bindParam(6, $this->category_id);
        $stmt->bindParam(7, $this->status);
        $stmt->bindParam(8, $this->created_by);
        
        // Executar
        if ($stmt->execute()) {
            $this->id = $this->conn->lastInsertId();
            return true;
        }
        
        return false;
    }

    /**
     * Lê um único documento pelo ID
     * @param int $id ID do documento
     * @return bool Verdadeiro se encontrado, falso caso contrário
     */
    public function readOne($id) {
        $query = "SELECT d.*, c.name as category_name
                  FROM " . $this->table_name . " d
                  LEFT JOIN category c ON d.category_id = c.id
                  WHERE d.id = ? LIMIT 0,1";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(1, $id);
        $stmt->execute();

        if ($stmt->rowCount() > 0) {
            $row = $stmt->fetch(PDO::FETCH_ASSOC);
            
            $this->id = $row['id'];
            $this->title = $row['title'];
            $this->author = $row['author'];
            $this->file_path = $row['file_path'];
            $this->file_type = $row['file_type'];
            $this->file_size = $row['file_size'];
            $this->category_id = $row['category_id'];
            $this->status = $row['status'];
            $this->created_at = $row['created_at'];
            $this->updated_at = $row['updated_at'];
            $this->created_by = $row['created_by'];
            
            return true;
        }
        
        return false;
    }

    /**
     * Lê todos os documentos com filtros opcionais
     * @param array $filters Filtros opcionais (title, author, category_id, status, start_date, end_date)
     * @param int $page Número da página
     * @param int $per_page Itens por página
     * @return array Documentos encontrados
     */
    public function readAll($filters = [], $page = 1, $per_page = 10) {
        $query = "SELECT d.*, c.name as category_name, u.username as created_by_username
                  FROM " . $this->table_name . " d
                  LEFT JOIN category c ON d.category_id = c.id
                  LEFT JOIN user u ON d.created_by = u.id
                  WHERE 1=1";
        
        // Aplicar filtros
        if (!empty($filters['title'])) {
            $query .= " AND d.title LIKE :title";
        }
        
        if (!empty($filters['author'])) {
            $query .= " AND d.author LIKE :author";
        }
        
        if (!empty($filters['category_id'])) {
            $query .= " AND d.category_id = :category_id";
        }
        
        if (!empty($filters['status'])) {
            $query .= " AND d.status = :status";
        }
        
        if (!empty($filters['start_date'])) {
            $query .= " AND d.created_at >= :start_date";
        }
        
        if (!empty($filters['end_date'])) {
            $query .= " AND d.created_at <= :end_date";
        }
        
        // Ordenação e paginação
        $query .= " ORDER BY d.created_at DESC";
        
        // Calcular offset para paginação
        $offset = ($page - 1) * $per_page;
        $query .= " LIMIT :offset, :per_page";
        
        $stmt = $this->conn->prepare($query);
        
        // Vincular parâmetros de filtro
        if (!empty($filters['title'])) {
            $title_param = "%" . $filters['title'] . "%";
            $stmt->bindParam(':title', $title_param);
        }
        
        if (!empty($filters['author'])) {
            $author_param = "%" . $filters['author'] . "%";
            $stmt->bindParam(':author', $author_param);
        }
        
        if (!empty($filters['category_id'])) {
            $stmt->bindParam(':category_id', $filters['category_id']);
        }
        
        if (!empty($filters['status'])) {
            $stmt->bindParam(':status', $filters['status']);
        }
        
        if (!empty($filters['start_date'])) {
            $stmt->bindParam(':start_date', $filters['start_date']);
        }
        
        if (!empty($filters['end_date'])) {
            $stmt->bindParam(':end_date', $filters['end_date']);
        }
        
        // Vincular parâmetros de paginação
        $stmt->bindParam(':offset', $offset, PDO::PARAM_INT);
        $stmt->bindParam(':per_page', $per_page, PDO::PARAM_INT);
        
        $stmt->execute();
        
        $documents = [];
        while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
            $documents[] = $row;
        }
        
        return $documents;
    }

    /**
     * Conta o total de documentos com filtros opcionais
     * @param array $filters Filtros opcionais
     * @return int Total de documentos
     */
    public function countAll($filters = []) {
        $query = "SELECT COUNT(*) as total FROM " . $this->table_name . " d WHERE 1=1";
        
        // Aplicar filtros
        if (!empty($filters['title'])) {
            $query .= " AND d.title LIKE :title";
        }
        
        if (!empty($filters['author'])) {
            $query .= " AND d.author LIKE :author";
        }
        
        if (!empty($filters['category_id'])) {
            $query .= " AND d.category_id = :category_id";
        }
        
        if (!empty($filters['status'])) {
            $query .= " AND d.status = :status";
        }
        
        if (!empty($filters['start_date'])) {
            $query .= " AND d.created_at >= :start_date";
        }
        
        if (!empty($filters['end_date'])) {
            $query .= " AND d.created_at <= :end_date";
        }
        
        $stmt = $this->conn->prepare($query);
        
        // Vincular parâmetros de filtro
        if (!empty($filters['title'])) {
            $title_param = "%" . $filters['title'] . "%";
            $stmt->bindParam(':title', $title_param);
        }
        
        if (!empty($filters['author'])) {
            $author_param = "%" . $filters['author'] . "%";
            $stmt->bindParam(':author', $author_param);
        }
        
        if (!empty($filters['category_id'])) {
            $stmt->bindParam(':category_id', $filters['category_id']);
        }
        
        if (!empty($filters['status'])) {
            $stmt->bindParam(':status', $filters['status']);
        }
        
        if (!empty($filters['start_date'])) {
            $stmt->bindParam(':start_date', $filters['start_date']);
        }
        
        if (!empty($filters['end_date'])) {
            $stmt->bindParam(':end_date', $filters['end_date']);
        }
        
        $stmt->execute();
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        
        return (int)$row['total'];
    }

    /**
     * Atualiza um documento existente
     * @return bool Verdadeiro se atualizado com sucesso, falso caso contrário
     */
    public function update() {
        $query = "UPDATE " . $this->table_name . "
                  SET title = ?, author = ?, category_id = ?, status = ?
                  WHERE id = ?";
        
        $stmt = $this->conn->prepare($query);
        
        // Sanitizar
        $this->title = htmlspecialchars(strip_tags($this->title));
        $this->author = htmlspecialchars(strip_tags($this->author));
        $this->category_id = htmlspecialchars(strip_tags($this->category_id));
        $this->status = htmlspecialchars(strip_tags($this->status));
        $this->id = htmlspecialchars(strip_tags($this->id));
        
        // Vincular valores
        $stmt->bindParam(1, $this->title);
        $stmt->bindParam(2, $this->author);
        $stmt->bindParam(3, $this->category_id);
        $stmt->bindParam(4, $this->status);
        $stmt->bindParam(5, $this->id);
        
        // Executar
        if ($stmt->execute()) {
            return true;
        }
        
        return false;
    }

    /**
     * Atualiza o arquivo do documento
     * @param string $file_path Caminho do arquivo
     * @param string $file_type Tipo do arquivo
     * @param int $file_size Tamanho do arquivo
     * @return bool Verdadeiro se atualizado com sucesso, falso caso contrário
     */
    public function updateFile($file_path, $file_type, $file_size) {
        $query = "UPDATE " . $this->table_name . "
                  SET file_path = ?, file_type = ?, file_size = ?
                  WHERE id = ?";
        
        $stmt = $this->conn->prepare($query);
        
        // Sanitizar
        $file_path = htmlspecialchars(strip_tags($file_path));
        $file_type = htmlspecialchars(strip_tags($file_type));
        
        // Vincular valores
        $stmt->bindParam(1, $file_path);
        $stmt->bindParam(2, $file_type);
        $stmt->bindParam(3, $file_size);
        $stmt->bindParam(4, $this->id);
        
        // Executar
        if ($stmt->execute()) {
            $this->file_path = $file_path;
            $this->file_type = $file_type;
            $this->file_size = $file_size;
            return true;
        }
        
        return false;
    }

    /**
     * Exclui um documento
     * @return bool Verdadeiro se excluído com sucesso, falso caso contrário
     */
    public function delete() {
        $query = "DELETE FROM " . $this->table_name . " WHERE id = ?";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(1, $this->id);
        
        if ($stmt->execute()) {
            return true;
        }
        
        return false;
    }

    /**
     * Obtém as palavras-chave de um documento
     * @return array Lista de palavras-chave
     */
    public function getKeywords() {
        $query = "SELECT keyword FROM document_keyword WHERE document_id = ?";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(1, $this->id);
        $stmt->execute();
        
        $keywords = [];
        while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
            $keywords[] = $row['keyword'];
        }
        
        return $keywords;
    }

    /**
     * Define as palavras-chave de um documento
     * @param array $keywords Lista de palavras-chave
     * @return bool Verdadeiro se definido com sucesso, falso caso contrário
     */
    public function setKeywords($keywords) {
        // Primeiro, remover todas as palavras-chave existentes
        $query = "DELETE FROM document_keyword WHERE document_id = ?";
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(1, $this->id);
        $stmt->execute();
        
        // Depois, inserir as novas palavras-chave
        $query = "INSERT INTO document_keyword (document_id, keyword) VALUES (?, ?)";
        $stmt = $this->conn->prepare($query);
        
        foreach ($keywords as $keyword) {
            if (!empty(trim($keyword))) {
                $keyword = htmlspecialchars(strip_tags(trim($keyword)));
                $stmt->bindParam(1, $this->id);
                $stmt->bindParam(2, $keyword);
                $stmt->execute();
            }
        }
        
        return true;
    }

    /**
     * Obtém as versões de um documento
     * @return array Lista de versões
     */
    public function getVersions() {
        $query = "SELECT v.*, u.username as created_by_username
                  FROM document_version v
                  LEFT JOIN user u ON v.created_by = u.id
                  WHERE v.document_id = ?
                  ORDER BY v.version_number DESC";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(1, $this->id);
        $stmt->execute();
        
        $versions = [];
        while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
            $versions[] = $row;
        }
        
        return $versions;
    }

    /**
     * Cria uma nova versão do documento
     * @param string $file_path Caminho do arquivo
     * @param int $file_size Tamanho do arquivo
     * @param int $created_by ID do usuário que criou a versão
     * @param string $change_summary Resumo das alterações
     * @return bool Verdadeiro se criado com sucesso, falso caso contrário
     */
    public function createVersion($file_path, $file_size, $created_by, $change_summary = '') {
        // Obter o número da próxima versão
        $query = "SELECT MAX(version_number) as max_version FROM document_version WHERE document_id = ?";
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(1, $this->id);
        $stmt->execute();
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        $next_version = ($row['max_version'] ?? 0) + 1;
        
        // Inserir nova versão
        $query = "INSERT INTO document_version
                  (document_id, version_number, file_path, file_size, created_by, change_summary)
                  VALUES (?, ?, ?, ?, ?, ?)";
        
        $stmt = $this->conn->prepare($query);
        
        // Sanitizar
        $file_path = htmlspecialchars(strip_tags($file_path));
        $change_summary = htmlspecialchars(strip_tags($change_summary));
        
        // Vincular valores
        $stmt->bindParam(1, $this->id);
        $stmt->bindParam(2, $next_version);
        $stmt->bindParam(3, $file_path);
        $stmt->bindParam(4, $file_size);
        $stmt->bindParam(5, $created_by);
        $stmt->bindParam(6, $change_summary);
        
        return $stmt->execute();
    }
}
?>
