<?php
/**
 * Modelo de Usuário
 * Sistema de Gerenciamento Eletrônico de Documentos (GED)
 */

class User {
    // Conexão com o banco e nome da tabela
    private $conn;
    private $table_name = "user";

    // Propriedades do objeto
    public $id;
    public $username;
    public $email;
    public $password_hash;
    public $full_name;
    public $user_type;
    public $is_active;
    public $created_at;
    public $last_login;

    /**
     * Construtor com $db como conexão com o banco de dados
     * @param PDO $db Conexão com o banco de dados
     */
    public function __construct($db) {
        $this->conn = $db;
    }

    /**
     * Lê um único usuário pelo ID
     * @param int $id ID do usuário
     * @return bool Verdadeiro se encontrado, falso caso contrário
     */
    public function readOne($id) {
        $query = "SELECT * FROM " . $this->table_name . " WHERE id = ? LIMIT 0,1";
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(1, $id);
        $stmt->execute();

        if ($stmt->rowCount() > 0) {
            $row = $stmt->fetch(PDO::FETCH_ASSOC);
            
            $this->id = $row['id'];
            $this->username = $row['username'];
            $this->email = $row['email'];
            $this->password_hash = $row['password_hash'];
            $this->full_name = $row['full_name'];
            $this->user_type = $row['user_type'];
            $this->is_active = $row['is_active'];
            $this->created_at = $row['created_at'];
            $this->last_login = $row['last_login'];
            
            return true;
        }
        
        return false;
    }

    /**
     * Lê um usuário pelo nome de usuário
     * @param string $username Nome de usuário
     * @return bool Verdadeiro se encontrado, falso caso contrário
     */
    public function readByUsername($username) {
        $query = "SELECT * FROM " . $this->table_name . " WHERE username = ? LIMIT 0,1";
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(1, $username);
        $stmt->execute();

        if ($stmt->rowCount() > 0) {
            $row = $stmt->fetch(PDO::FETCH_ASSOC);
            
            $this->id = $row['id'];
            $this->username = $row['username'];
            $this->email = $row['email'];
            $this->password_hash = $row['password_hash'];
            $this->full_name = $row['full_name'];
            $this->user_type = $row['user_type'];
            $this->is_active = $row['is_active'];
            $this->created_at = $row['created_at'];
            $this->last_login = $row['last_login'];
            
            return true;
        }
        
        return false;
    }

    /**
     * Lê todos os usuários
     * @return PDOStatement Resultado da consulta
     */
    public function readAll() {
        $query = "SELECT * FROM " . $this->table_name . " ORDER BY created_at DESC";
        $stmt = $this->conn->prepare($query);
        $stmt->execute();
        
        return $stmt;
    }

    /**
     * Cria um novo usuário
     * @return bool Verdadeiro se criado com sucesso, falso caso contrário
     */
    public function create() {
        $query = "INSERT INTO " . $this->table_name . "
                  (username, email, password_hash, full_name, user_type, is_active)
                  VALUES (?, ?, ?, ?, ?, ?)";
        
        $stmt = $this->conn->prepare($query);
        
        // Sanitizar
        $this->username = htmlspecialchars(strip_tags($this->username));
        $this->email = htmlspecialchars(strip_tags($this->email));
        $this->full_name = htmlspecialchars(strip_tags($this->full_name));
        $this->user_type = htmlspecialchars(strip_tags($this->user_type));
        $this->is_active = $this->is_active ? 1 : 0;
        
        // Vincular valores
        $stmt->bindParam(1, $this->username);
        $stmt->bindParam(2, $this->email);
        $stmt->bindParam(3, $this->password_hash);
        $stmt->bindParam(4, $this->full_name);
        $stmt->bindParam(5, $this->user_type);
        $stmt->bindParam(6, $this->is_active);
        
        // Executar
        if ($stmt->execute()) {
            $this->id = $this->conn->lastInsertId();
            return true;
        }
        
        return false;
    }

    /**
     * Atualiza um usuário existente
     * @return bool Verdadeiro se atualizado com sucesso, falso caso contrário
     */
    public function update() {
        $query = "UPDATE " . $this->table_name . "
                  SET email = ?, full_name = ?, user_type = ?, is_active = ?
                  WHERE id = ?";
        
        $stmt = $this->conn->prepare($query);
        
        // Sanitizar
        $this->email = htmlspecialchars(strip_tags($this->email));
        $this->full_name = htmlspecialchars(strip_tags($this->full_name));
        $this->user_type = htmlspecialchars(strip_tags($this->user_type));
        $this->is_active = $this->is_active ? 1 : 0;
        $this->id = htmlspecialchars(strip_tags($this->id));
        
        // Vincular valores
        $stmt->bindParam(1, $this->email);
        $stmt->bindParam(2, $this->full_name);
        $stmt->bindParam(3, $this->user_type);
        $stmt->bindParam(4, $this->is_active);
        $stmt->bindParam(5, $this->id);
        
        // Executar
        if ($stmt->execute()) {
            return true;
        }
        
        return false;
    }

    /**
     * Atualiza a senha de um usuário
     * @param string $new_password Nova senha (não criptografada)
     * @return bool Verdadeiro se atualizado com sucesso, falso caso contrário
     */
    public function updatePassword($new_password) {
        $query = "UPDATE " . $this->table_name . "
                  SET password_hash = ?
                  WHERE id = ?";
        
        $stmt = $this->conn->prepare($query);
        
        // Criptografar a senha
        $password_hash = password_hash($new_password, PASSWORD_BCRYPT);
        
        // Vincular valores
        $stmt->bindParam(1, $password_hash);
        $stmt->bindParam(2, $this->id);
        
        // Executar
        if ($stmt->execute()) {
            $this->password_hash = $password_hash;
            return true;
        }
        
        return false;
    }

    /**
     * Atualiza o último login do usuário
     * @return bool Verdadeiro se atualizado com sucesso, falso caso contrário
     */
    public function updateLastLogin() {
        $query = "UPDATE " . $this->table_name . "
                  SET last_login = NOW()
                  WHERE id = ?";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(1, $this->id);
        
        if ($stmt->execute()) {
            $this->last_login = date('Y-m-d H:i:s');
            return true;
        }
        
        return false;
    }

    /**
     * Verifica se a senha fornecida corresponde à senha armazenada
     * @param string $password Senha a verificar
     * @return bool Verdadeiro se a senha corresponder, falso caso contrário
     */
    public function verifyPassword($password) {
        return password_verify($password, $this->password_hash);
    }

    /**
     * Verifica se o usuário tem uma permissão específica
     * @param string $permission_name Nome da permissão
     * @return bool Verdadeiro se o usuário tem a permissão, falso caso contrário
     */
    public function hasPermission($permission_name) {
        // Administradores têm todas as permissões
        if ($this->user_type === 'admin') {
            return true;
        }
        
        $query = "SELECT p.id FROM permission p
                  JOIN user_permission up ON p.id = up.permission_id
                  WHERE up.user_id = ? AND p.name = ?";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(1, $this->id);
        $stmt->bindParam(2, $permission_name);
        $stmt->execute();
        
        return ($stmt->rowCount() > 0);
    }

    /**
     * Obtém todas as permissões do usuário
     * @return array Lista de permissões
     */
    public function getPermissions() {
        $permissions = array();
        
        // Administradores têm todas as permissões
        if ($this->user_type === 'admin') {
            $query = "SELECT * FROM permission";
            $stmt = $this->conn->prepare($query);
            $stmt->execute();
            
            while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
                $permissions[] = $row;
            }
            
            return $permissions;
        }
        
        $query = "SELECT p.* FROM permission p
                  JOIN user_permission up ON p.id = up.permission_id
                  WHERE up.user_id = ?";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(1, $this->id);
        $stmt->execute();
        
        while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
            $permissions[] = $row;
        }
        
        return $permissions;
    }
}
?>
