<?php
/**
 * Modelo de Workflow
 * Sistema de Gerenciamento Eletrônico de Documentos (GED)
 */

class Workflow {
    // Conexão com o banco e nome da tabela
    private $conn;
    private $table_name = "workflow";

    // Propriedades do objeto
    public $id;
    public $name;
    public $description;
    public $created_at;
    public $created_by;

    /**
     * Construtor com $db como conexão com o banco de dados
     * @param PDO $db Conexão com o banco de dados
     */
    public function __construct($db) {
        $this->conn = $db;
    }

    /**
     * Lê todos os workflows
     * @return PDOStatement Resultado da consulta
     */
    public function readAll() {
        $query = "SELECT w.*, u.username as created_by_username
                  FROM " . $this->table_name . " w
                  LEFT JOIN user u ON w.created_by = u.id
                  ORDER BY w.name ASC";
        
        $stmt = $this->conn->prepare($query);
        $stmt->execute();
        
        return $stmt;
    }

    /**
     * Lê um único workflow pelo ID
     * @param int $id ID do workflow
     * @return bool Verdadeiro se encontrado, falso caso contrário
     */
    public function readOne($id) {
        $query = "SELECT w.*, u.username as created_by_username
                  FROM " . $this->table_name . " w
                  LEFT JOIN user u ON w.created_by = u.id
                  WHERE w.id = ? LIMIT 0,1";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(1, $id);
        $stmt->execute();

        if ($stmt->rowCount() > 0) {
            $row = $stmt->fetch(PDO::FETCH_ASSOC);
            
            $this->id = $row['id'];
            $this->name = $row['name'];
            $this->description = $row['description'];
            $this->created_at = $row['created_at'];
            $this->created_by = $row['created_by'];
            
            return true;
        }
        
        return false;
    }

    /**
     * Cria um novo workflow
     * @return bool Verdadeiro se criado com sucesso, falso caso contrário
     */
    public function create() {
        $query = "INSERT INTO " . $this->table_name . "
                  (name, description, created_by)
                  VALUES (?, ?, ?)";
        
        $stmt = $this->conn->prepare($query);
        
        // Sanitizar
        $this->name = htmlspecialchars(strip_tags($this->name));
        $this->description = htmlspecialchars(strip_tags($this->description));
        
        // Vincular valores
        $stmt->bindParam(1, $this->name);
        $stmt->bindParam(2, $this->description);
        $stmt->bindParam(3, $this->created_by);
        
        // Executar
        if ($stmt->execute()) {
            $this->id = $this->conn->lastInsertId();
            return true;
        }
        
        return false;
    }

    /**
     * Atualiza um workflow existente
     * @return bool Verdadeiro se atualizado com sucesso, falso caso contrário
     */
    public function update() {
        $query = "UPDATE " . $this->table_name . "
                  SET name = ?, description = ?
                  WHERE id = ?";
        
        $stmt = $this->conn->prepare($query);
        
        // Sanitizar
        $this->name = htmlspecialchars(strip_tags($this->name));
        $this->description = htmlspecialchars(strip_tags($this->description));
        $this->id = htmlspecialchars(strip_tags($this->id));
        
        // Vincular valores
        $stmt->bindParam(1, $this->name);
        $stmt->bindParam(2, $this->description);
        $stmt->bindParam(3, $this->id);
        
        // Executar
        if ($stmt->execute()) {
            return true;
        }
        
        return false;
    }

    /**
     * Exclui um workflow
     * @return bool Verdadeiro se excluído com sucesso, falso caso contrário
     */
    public function delete() {
        // Verificar se há documentos usando este workflow
        $query = "SELECT COUNT(*) as count FROM document_approval WHERE workflow_id = ?";
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(1, $this->id);
        $stmt->execute();
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($row['count'] > 0) {
            // Não permitir exclusão se houver documentos associados
            return false;
        }
        
        // Excluir etapas do workflow
        $query = "DELETE FROM workflow_step WHERE workflow_id = ?";
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(1, $this->id);
        $stmt->execute();
        
        // Excluir workflow
        $query = "DELETE FROM " . $this->table_name . " WHERE id = ?";
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(1, $this->id);
        
        if ($stmt->execute()) {
            return true;
        }
        
        return false;
    }

    /**
     * Obtém as etapas de um workflow
     * @return array Lista de etapas
     */
    public function getSteps() {
        $query = "SELECT ws.*, u.username as approver_username
                  FROM workflow_step ws
                  LEFT JOIN user u ON ws.approver_id = u.id
                  WHERE ws.workflow_id = ?
                  ORDER BY ws.step_order ASC";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(1, $this->id);
        $stmt->execute();
        
        $steps = [];
        while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
            $steps[] = $row;
        }
        
        return $steps;
    }

    /**
     * Adiciona uma etapa ao workflow
     * @param string $name Nome da etapa
     * @param string $description Descrição da etapa
     * @param int $approver_id ID do aprovador
     * @return int ID da etapa criada ou false em caso de erro
     */
    public function addStep($name, $description, $approver_id) {
        // Obter a próxima ordem
        $query = "SELECT MAX(step_order) as max_order FROM workflow_step WHERE workflow_id = ?";
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(1, $this->id);
        $stmt->execute();
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        $next_order = ($row['max_order'] ?? 0) + 1;
        
        // Inserir etapa
        $query = "INSERT INTO workflow_step
                  (workflow_id, step_order, name, description, approver_id)
                  VALUES (?, ?, ?, ?, ?)";
        
        $stmt = $this->conn->prepare($query);
        
        // Sanitizar
        $name = htmlspecialchars(strip_tags($name));
        $description = htmlspecialchars(strip_tags($description));
        
        // Vincular valores
        $stmt->bindParam(1, $this->id);
        $stmt->bindParam(2, $next_order);
        $stmt->bindParam(3, $name);
        $stmt->bindParam(4, $description);
        $stmt->bindParam(5, $approver_id);
        
        if ($stmt->execute()) {
            return $this->conn->lastInsertId();
        }
        
        return false;
    }

    /**
     * Atualiza uma etapa do workflow
     * @param int $step_id ID da etapa
     * @param string $name Nome da etapa
     * @param string $description Descrição da etapa
     * @param int $approver_id ID do aprovador
     * @return bool Verdadeiro se atualizado com sucesso, falso caso contrário
     */
    public function updateStep($step_id, $name, $description, $approver_id) {
        $query = "UPDATE workflow_step
                  SET name = ?, description = ?, approver_id = ?
                  WHERE id = ? AND workflow_id = ?";
        
        $stmt = $this->conn->prepare($query);
        
        // Sanitizar
        $name = htmlspecialchars(strip_tags($name));
        $description = htmlspecialchars(strip_tags($description));
        
        // Vincular valores
        $stmt->bindParam(1, $name);
        $stmt->bindParam(2, $description);
        $stmt->bindParam(3, $approver_id);
        $stmt->bindParam(4, $step_id);
        $stmt->bindParam(5, $this->id);
        
        return $stmt->execute();
    }

    /**
     * Remove uma etapa do workflow
     * @param int $step_id ID da etapa
     * @return bool Verdadeiro se removido com sucesso, falso caso contrário
     */
    public function removeStep($step_id) {
        // Verificar se a etapa pertence a este workflow
        $query = "SELECT step_order FROM workflow_step WHERE id = ? AND workflow_id = ?";
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(1, $step_id);
        $stmt->bindParam(2, $this->id);
        $stmt->execute();
        
        if ($stmt->rowCount() == 0) {
            return false;
        }
        
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        $removed_order = $row['step_order'];
        
        // Remover etapa
        $query = "DELETE FROM workflow_step WHERE id = ?";
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(1, $step_id);
        
        if (!$stmt->execute()) {
            return false;
        }
        
        // Reordenar etapas restantes
        $query = "UPDATE workflow_step SET step_order = step_order - 1 WHERE workflow_id = ? AND step_order > ?";
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(1, $this->id);
        $stmt->bindParam(2, $removed_order);
        
        return $stmt->execute();
    }

    /**
     * Reordena as etapas do workflow
     * @param array $step_order Array associativo de ID da etapa => nova ordem
     * @return bool Verdadeiro se reordenado com sucesso, falso caso contrário
     */
    public function reorderSteps($step_order) {
        $this->conn->beginTransaction();
        
        try {
            foreach ($step_order as $step_id => $order) {
                $query = "UPDATE workflow_step SET step_order = ? WHERE id = ? AND workflow_id = ?";
                $stmt = $this->conn->prepare($query);
                $stmt->bindParam(1, $order);
                $stmt->bindParam(2, $step_id);
                $stmt->bindParam(3, $this->id);
                $stmt->execute();
            }
            
            $this->conn->commit();
            return true;
        } catch (Exception $e) {
            $this->conn->rollBack();
            return false;
        }
    }
}
?>
