<?php
/**
 * Script de teste de integração para o Sistema GED
 * Verifica a integração entre os módulos PHP e o banco de dados MySQL
 */

// Iniciar sessão para testes de autenticação
session_start();

// Definir constantes para o ambiente de teste
define('TEST_MODE', true);
define('TEST_USER_ID', 1);

// Incluir arquivos necessários
require_once 'config/database.php';
require_once 'config/config.php';
require_once 'models/user.php';
require_once 'models/document.php';
require_once 'models/category.php';
require_once 'models/workflow.php';
require_once 'models/audit_log.php';
require_once 'controllers/auth_controller.php';
require_once 'controllers/document_controller.php';
require_once 'controllers/category_controller.php';
require_once 'controllers/workflow_controller.php';
require_once 'controllers/audit_log_controller.php';
require_once 'controllers/settings_controller.php';

// Função para exibir resultados de teste
function test_result($test_name, $success, $message = '') {
    echo "<div style='margin: 5px; padding: 10px; border: 1px solid " . ($success ? "#4CAF50" : "#F44336") . "; background-color: " . ($success ? "#E8F5E9" : "#FFEBEE") . ";'>";
    echo "<strong>" . ($success ? "✓ PASSOU" : "✗ FALHOU") . ":</strong> $test_name";
    if ($message) {
        echo "<br><em>$message</em>";
    }
    echo "</div>";
    
    return $success;
}

// Iniciar HTML
echo "<!DOCTYPE html>
<html lang='pt-BR'>
<head>
    <meta charset='UTF-8'>
    <meta name='viewport' content='width=device-width, initial-scale=1.0'>
    <title>Teste de Integração - Sistema GED</title>
    <style>
        body { font-family: Arial, sans-serif; margin: 20px; }
        h1, h2 { color: #333; }
        .summary { margin-top: 20px; padding: 10px; background-color: #f5f5f5; border-radius: 5px; }
        .success { color: #4CAF50; }
        .failure { color: #F44336; }
    </style>
</head>
<body>
    <h1>Teste de Integração - Sistema GED</h1>
    <p>Este script verifica a integração entre os módulos PHP e o banco de dados MySQL.</p>
";

// Inicializar contadores
$tests_total = 0;
$tests_passed = 0;

// Testar conexão com o banco de dados
echo "<h2>1. Teste de Conexão com o Banco de Dados</h2>";

try {
    $database = new Database();
    $db = $database->getConnection();
    
    $query = "SELECT 1";
    $stmt = $db->prepare($query);
    $stmt->execute();
    
    $success = test_result("Conexão com o banco de dados", true, "Conexão estabelecida com sucesso");
    $tests_total++;
    if ($success) $tests_passed++;
} catch (PDOException $e) {
    test_result("Conexão com o banco de dados", false, "Erro: " . $e->getMessage());
    $tests_total++;
}

// Testar autenticação
echo "<h2>2. Teste de Autenticação</h2>";

try {
    $auth_controller = new AuthController();
    
    // Testar criação de usuário (se não existir)
    $user_exists = false;
    $query = "SELECT COUNT(*) FROM user WHERE username = 'admin'";
    $stmt = $db->prepare($query);
    $stmt->execute();
    if ($stmt->fetchColumn() > 0) {
        $user_exists = true;
    }
    
    if (!$user_exists) {
        $result = $auth_controller->register([
            'username' => 'admin',
            'password' => password_hash('admin123', PASSWORD_DEFAULT),
            'email' => 'admin@example.com',
            'full_name' => 'Administrador',
            'user_type' => 'admin',
            'status' => 'active'
        ]);
        
        $success = test_result("Criação de usuário admin", $result['success'], $result['success'] ? "Usuário admin criado com sucesso" : $result['message']);
    } else {
        $success = test_result("Verificação de usuário admin", true, "Usuário admin já existe");
    }
    $tests_total++;
    if ($success) $tests_passed++;
    
    // Testar login
    $result = $auth_controller->login('admin', 'admin123');
    $success = test_result("Login de usuário", $result['success'], $result['success'] ? "Login realizado com sucesso" : $result['message']);
    $tests_total++;
    if ($success) $tests_passed++;
    
    // Testar verificação de permissões
    $user_model = new User();
    $has_permission = $user_model->hasPermission(1, 'view_document');
    $success = test_result("Verificação de permissões", $has_permission !== false, "Verificação de permissão 'view_document' para usuário admin");
    $tests_total++;
    if ($success) $tests_passed++;
    
} catch (Exception $e) {
    test_result("Teste de autenticação", false, "Erro: " . $e->getMessage());
    $tests_total++;
}

// Testar categorias
echo "<h2>3. Teste de Categorias</h2>";

try {
    $category_controller = new CategoryController();
    
    // Testar criação de categoria
    $category_name = "Categoria Teste " . date('YmdHis');
    $result = $category_controller->create([
        'name' => $category_name,
        'description' => 'Categoria criada para teste de integração',
        'color' => '#3498db',
        'created_by' => TEST_USER_ID
    ]);
    
    $success = test_result("Criação de categoria", $result['success'], $result['success'] ? "Categoria criada com ID: " . $result['category_id'] : $result['message']);
    $tests_total++;
    if ($success) $tests_passed++;
    
    if ($result['success']) {
        $category_id = $result['category_id'];
        
        // Testar obtenção de categoria
        $category = $category_controller->getById($category_id);
        $success = test_result("Obtenção de categoria por ID", $category !== false, "Categoria obtida: " . ($category ? $category['name'] : "Não encontrada"));
        $tests_total++;
        if ($success) $tests_passed++;
        
        // Testar atualização de categoria
        $update_result = $category_controller->update($category_id, [
            'name' => $category_name . " (Atualizada)",
            'description' => 'Categoria atualizada para teste de integração',
            'color' => '#2ecc71'
        ]);
        
        $success = test_result("Atualização de categoria", $update_result['success'], $update_result['success'] ? "Categoria atualizada com sucesso" : $update_result['message']);
        $tests_total++;
        if ($success) $tests_passed++;
    }
    
    // Testar listagem de categorias
    $categories = $category_controller->getAll();
    $success = test_result("Listagem de categorias", is_array($categories), "Total de categorias: " . count($categories));
    $tests_total++;
    if ($success) $tests_passed++;
    
} catch (Exception $e) {
    test_result("Teste de categorias", false, "Erro: " . $e->getMessage());
    $tests_total++;
}

// Testar documentos
echo "<h2>4. Teste de Documentos</h2>";

try {
    $document_controller = new DocumentController();
    
    // Testar criação de documento
    $document_title = "Documento Teste " . date('YmdHis');
    $result = $document_controller->create([
        'title' => $document_title,
        'description' => 'Documento criado para teste de integração',
        'category_id' => $category_id ?? 1,
        'file_name' => 'teste.pdf',
        'file_path' => 'uploads/teste.pdf',
        'file_size' => 1024,
        'file_type' => 'application/pdf',
        'created_by' => TEST_USER_ID,
        'status' => 'draft'
    ]);
    
    $success = test_result("Criação de documento", $result['success'], $result['success'] ? "Documento criado com ID: " . $result['document_id'] : $result['message']);
    $tests_total++;
    if ($success) $tests_passed++;
    
    if ($result['success']) {
        $document_id = $result['document_id'];
        
        // Testar obtenção de documento
        $document = $document_controller->getById($document_id);
        $success = test_result("Obtenção de documento por ID", $document !== false, "Documento obtido: " . ($document ? $document['title'] : "Não encontrado"));
        $tests_total++;
        if ($success) $tests_passed++;
        
        // Testar atualização de documento
        $update_result = $document_controller->update($document_id, [
            'title' => $document_title . " (Atualizado)",
            'description' => 'Documento atualizado para teste de integração',
            'status' => 'published'
        ]);
        
        $success = test_result("Atualização de documento", $update_result['success'], $update_result['success'] ? "Documento atualizado com sucesso" : $update_result['message']);
        $tests_total++;
        if ($success) $tests_passed++;
        
        // Testar adição de versão de documento
        $version_result = $document_controller->addVersion($document_id, [
            'version_number' => 2,
            'file_name' => 'teste_v2.pdf',
            'file_path' => 'uploads/teste_v2.pdf',
            'file_size' => 2048,
            'file_type' => 'application/pdf',
            'created_by' => TEST_USER_ID,
            'changes' => 'Versão atualizada para teste'
        ]);
        
        $success = test_result("Adição de versão de documento", $version_result['success'], $version_result['success'] ? "Versão adicionada com sucesso" : $version_result['message']);
        $tests_total++;
        if ($success) $tests_passed++;
    }
    
    // Testar pesquisa de documentos
    $search_result = $document_controller->search(['query' => 'Teste']);
    $success = test_result("Pesquisa de documentos", is_array($search_result), "Total de documentos encontrados: " . count($search_result));
    $tests_total++;
    if ($success) $tests_passed++;
    
} catch (Exception $e) {
    test_result("Teste de documentos", false, "Erro: " . $e->getMessage());
    $tests_total++;
}

// Testar workflow
echo "<h2>5. Teste de Workflow</h2>";

try {
    $workflow_controller = new WorkflowController();
    
    // Testar criação de workflow
    $workflow_name = "Workflow Teste " . date('YmdHis');
    $result = $workflow_controller->create([
        'name' => $workflow_name,
        'description' => 'Workflow criado para teste de integração',
        'created_by' => TEST_USER_ID,
        'steps' => [
            ['name' => 'Revisão', 'order' => 1],
            ['name' => 'Aprovação', 'order' => 2],
            ['name' => 'Publicação', 'order' => 3]
        ]
    ]);
    
    $success = test_result("Criação de workflow", $result['success'], $result['success'] ? "Workflow criado com ID: " . $result['workflow_id'] : $result['message']);
    $tests_total++;
    if ($success) $tests_passed++;
    
    if ($result['success'] && isset($document_id)) {
        $workflow_id = $result['workflow_id'];
        
        // Testar atribuição de workflow a documento
        $assign_result = $workflow_controller->assignToDocument($workflow_id, $document_id);
        $success = test_result("Atribuição de workflow a documento", $assign_result['success'], $assign_result['success'] ? "Workflow atribuído com sucesso" : $assign_result['message']);
        $tests_total++;
        if ($success) $tests_passed++;
        
        // Testar aprovação de etapa de workflow
        $approve_result = $workflow_controller->approveStep($document_id, 1, TEST_USER_ID, 'Aprovado em teste de integração');
        $success = test_result("Aprovação de etapa de workflow", $approve_result['success'], $approve_result['success'] ? "Etapa aprovada com sucesso" : $approve_result['message']);
        $tests_total++;
        if ($success) $tests_passed++;
    }
    
    // Testar listagem de workflows
    $workflows = $workflow_controller->getAll();
    $success = test_result("Listagem de workflows", is_array($workflows), "Total de workflows: " . count($workflows));
    $tests_total++;
    if ($success) $tests_passed++;
    
} catch (Exception $e) {
    test_result("Teste de workflow", false, "Erro: " . $e->getMessage());
    $tests_total++;
}

// Testar logs de auditoria
echo "<h2>6. Teste de Logs de Auditoria</h2>";

try {
    $audit_log_controller = new AuditLogController();
    
    // Testar registro de log
    $log_result = $audit_log_controller->log(
        TEST_USER_ID,
        'test',
        'integration',
        'Teste de integração do sistema',
        $_SERVER['REMOTE_ADDR'] ?? '127.0.0.1'
    );
    
    $success = test_result("Registro de log de auditoria", $log_result, "Log registrado com sucesso");
    $tests_total++;
    if ($success) $tests_passed++;
    
    // Testar listagem de logs
    $logs = $audit_log_controller->getAll(1, 10);
    $success = test_result("Listagem de logs de auditoria", is_array($logs), "Total de logs: " . count($logs));
    $tests_total++;
    if ($success) $tests_passed++;
    
} catch (Exception $e) {
    test_result("Teste de logs de auditoria", false, "Erro: " . $e->getMessage());
    $tests_total++;
}

// Testar configurações
echo "<h2>7. Teste de Configurações</h2>";

try {
    $settings_controller = new SettingsController();
    
    // Testar obtenção de configurações
    $settings = $settings_controller->getAllSettings();
    $success = test_result("Obtenção de configurações", is_array($settings), "Total de configurações: " . count($settings));
    $tests_total++;
    if ($success) $tests_passed++;
    
    // Testar atualização de configuração
    $update_result = $settings_controller->updateSetting('test_setting', 'valor_teste_' . date('YmdHis'));
    $success = test_result("Atualização de configuração", $update_result, "Configuração atualizada com sucesso");
    $tests_total++;
    if ($success) $tests_passed++;
    
    // Testar obtenção de configuração específica
    $setting_value = $settings_controller->getSetting('test_setting', 'default');
    $success = test_result("Obtenção de configuração específica", $setting_value !== 'default', "Valor obtido: " . $setting_value);
    $tests_total++;
    if ($success) $tests_passed++;
    
} catch (Exception $e) {
    test_result("Teste de configurações", false, "Erro: " . $e->getMessage());
    $tests_total++;
}

// Exibir resumo dos testes
echo "<div class='summary'>";
echo "<h2>Resumo dos Testes</h2>";
echo "<p>Total de testes: <strong>$tests_total</strong></p>";
echo "<p>Testes bem-sucedidos: <strong class='success'>$tests_passed</strong></p>";
echo "<p>Testes falhos: <strong class='failure'>" . ($tests_total - $tests_passed) . "</strong></p>";
echo "<p>Taxa de sucesso: <strong>" . ($tests_total > 0 ? round(($tests_passed / $tests_total) * 100, 2) : 0) . "%</strong></p>";
echo "</div>";

// Finalizar HTML
echo "</body></html>";
?>
