<?php
/**
 * Página de gerenciamento de workflow
 * Sistema de Gerenciamento Eletrônico de Documentos (GED)
 */

// Iniciar sessão e verificar autenticação
session_start();
if (!isset($_SESSION['user_id'])) {
    header('Location: login.php');
    exit;
}

// Verificar se o usuário tem permissão de administrador
if ($_SESSION['user_type'] != 'admin') {
    $_SESSION['flash_message'] = "Acesso negado. Você não tem permissão para acessar esta página.";
    $_SESSION['flash_type'] = "danger";
    header('Location: index.php');
    exit;
}

// Incluir arquivos necessários
require_once 'config/database.php';
require_once 'models/workflow.php';
require_once 'controllers/workflow_controller.php';

// Conectar ao banco de dados
$database = new Database();
$db = $database->getConnection();

// Instanciar controlador
$workflow_controller = new WorkflowController();

// Processar ações
if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    if (isset($_POST['action'])) {
        // Adicionar workflow
        if ($_POST['action'] == 'add') {
            $name = $_POST['name'] ?? '';
            $description = $_POST['description'] ?? '';
            $steps = $_POST['steps'] ?? [];
            
            if (!empty($name) && !empty($steps)) {
                $result = $workflow_controller->addWorkflow([
                    'name' => $name,
                    'description' => $description,
                    'steps' => $steps,
                    'created_by' => $_SESSION['user_id']
                ]);
                
                if ($result['success']) {
                    $_SESSION['flash_message'] = "Workflow adicionado com sucesso.";
                    $_SESSION['flash_type'] = "success";
                } else {
                    $_SESSION['flash_message'] = $result['message'];
                    $_SESSION['flash_type'] = "danger";
                }
            } else {
                $_SESSION['flash_message'] = "Nome e etapas do workflow são obrigatórios.";
                $_SESSION['flash_type'] = "danger";
            }
        }
        
        // Editar workflow
        else if ($_POST['action'] == 'edit') {
            $id = $_POST['id'] ?? '';
            $name = $_POST['name'] ?? '';
            $description = $_POST['description'] ?? '';
            $steps = $_POST['steps'] ?? [];
            
            if (!empty($id) && !empty($name) && !empty($steps)) {
                $result = $workflow_controller->updateWorkflow([
                    'id' => $id,
                    'name' => $name,
                    'description' => $description,
                    'steps' => $steps,
                    'updated_by' => $_SESSION['user_id']
                ]);
                
                if ($result['success']) {
                    $_SESSION['flash_message'] = "Workflow atualizado com sucesso.";
                    $_SESSION['flash_type'] = "success";
                } else {
                    $_SESSION['flash_message'] = $result['message'];
                    $_SESSION['flash_type'] = "danger";
                }
            } else {
                $_SESSION['flash_message'] = "ID, nome e etapas do workflow são obrigatórios.";
                $_SESSION['flash_type'] = "danger";
            }
        }
        
        // Excluir workflow
        else if ($_POST['action'] == 'delete') {
            $id = $_POST['id'] ?? '';
            
            if (!empty($id)) {
                $result = $workflow_controller->deleteWorkflow($id);
                
                if ($result['success']) {
                    $_SESSION['flash_message'] = "Workflow excluído com sucesso.";
                    $_SESSION['flash_type'] = "success";
                } else {
                    $_SESSION['flash_message'] = $result['message'];
                    $_SESSION['flash_type'] = "danger";
                }
            } else {
                $_SESSION['flash_message'] = "ID do workflow é obrigatório.";
                $_SESSION['flash_type'] = "danger";
            }
        }
        
        // Redirecionar para evitar reenvio do formulário
        header('Location: workflows.php');
        exit;
    }
}

// Obter todos os workflows
$workflow = new Workflow($db);
$workflows = $workflow->readAll();

// Definir título da página
$page_title = "Gerenciar Workflows";

// Incluir cabeçalho
include 'includes/header.php';
?>

<div class="row mb-4">
    <div class="col-md-6">
        <h1><i class="fas fa-project-diagram me-2"></i> Gerenciar Workflows</h1>
    </div>
    <div class="col-md-6 text-md-end">
        <button type="button" class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#addWorkflowModal">
            <i class="fas fa-plus me-1"></i> Novo Workflow
        </button>
    </div>
</div>

<!-- Lista de workflows -->
<div class="card">
    <div class="card-body">
        <?php if (count($workflows) > 0): ?>
        <div class="table-responsive">
            <table class="table table-hover datatable">
                <thead>
                    <tr>
                        <th>ID</th>
                        <th>Nome</th>
                        <th>Descrição</th>
                        <th>Etapas</th>
                        <th>Documentos</th>
                        <th>Criado por</th>
                        <th>Data de Criação</th>
                        <th>Ações</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($workflows as $wf): ?>
                    <tr>
                        <td><?php echo $wf['id']; ?></td>
                        <td><?php echo htmlspecialchars($wf['name']); ?></td>
                        <td><?php echo htmlspecialchars($wf['description'] ?? ''); ?></td>
                        <td>
                            <span class="badge bg-info"><?php echo count(json_decode($wf['steps'], true)); ?> etapas</span>
                        </td>
                        <td>
                            <a href="documents.php?workflow_id=<?php echo $wf['id']; ?>" class="badge bg-primary text-decoration-none">
                                <?php echo $wf['document_count']; ?> documentos
                            </a>
                        </td>
                        <td><?php echo htmlspecialchars($wf['created_by_name']); ?></td>
                        <td><?php echo date('d/m/Y', strtotime($wf['created_at'])); ?></td>
                        <td>
                            <div class="btn-group btn-group-sm">
                                <button type="button" class="btn btn-info view-workflow" 
                                        data-id="<?php echo $wf['id']; ?>"
                                        data-name="<?php echo htmlspecialchars($wf['name']); ?>"
                                        data-steps='<?php echo $wf['steps']; ?>'>
                                    <i class="fas fa-eye"></i>
                                </button>
                                <button type="button" class="btn btn-primary edit-workflow" 
                                        data-id="<?php echo $wf['id']; ?>"
                                        data-name="<?php echo htmlspecialchars($wf['name']); ?>"
                                        data-description="<?php echo htmlspecialchars($wf['description'] ?? ''); ?>"
                                        data-steps='<?php echo $wf['steps']; ?>'>
                                    <i class="fas fa-edit"></i>
                                </button>
                                <button type="button" class="btn btn-danger delete-workflow" 
                                        data-id="<?php echo $wf['id']; ?>"
                                        data-name="<?php echo htmlspecialchars($wf['name']); ?>"
                                        data-count="<?php echo $wf['document_count']; ?>">
                                    <i class="fas fa-trash"></i>
                                </button>
                            </div>
                        </td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
        <?php else: ?>
        <div class="alert alert-info">
            <i class="fas fa-info-circle me-2"></i> Nenhum workflow encontrado.
        </div>
        <?php endif; ?>
    </div>
</div>

<!-- Modal para adicionar workflow -->
<div class="modal fade" id="addWorkflowModal" tabindex="-1" aria-labelledby="addWorkflowModalLabel" aria-hidden="true">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <form method="POST" action="workflows.php" id="addWorkflowForm">
                <input type="hidden" name="action" value="add">
                
                <div class="modal-header">
                    <h5 class="modal-title" id="addWorkflowModalLabel">Novo Workflow</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Fechar"></button>
                </div>
                
                <div class="modal-body">
                    <div class="mb-3">
                        <label for="add-name" class="form-label">Nome <span class="text-danger">*</span></label>
                        <input type="text" class="form-control" id="add-name" name="name" required>
                    </div>
                    
                    <div class="mb-3">
                        <label for="add-description" class="form-label">Descrição</label>
                        <textarea class="form-control" id="add-description" name="description" rows="2"></textarea>
                    </div>
                    
                    <div class="mb-3">
                        <label class="form-label">Etapas do Workflow <span class="text-danger">*</span></label>
                        <p class="text-muted small">Defina as etapas que os documentos devem passar neste workflow.</p>
                        
                        <div id="add-steps-container">
                            <!-- As etapas serão adicionadas aqui dinamicamente -->
                        </div>
                        
                        <button type="button" class="btn btn-outline-secondary mt-2" id="add-step-btn">
                            <i class="fas fa-plus me-1"></i> Adicionar Etapa
                        </button>
                    </div>
                </div>
                
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancelar</button>
                    <button type="submit" class="btn btn-primary">Adicionar</button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Modal para editar workflow -->
<div class="modal fade" id="editWorkflowModal" tabindex="-1" aria-labelledby="editWorkflowModalLabel" aria-hidden="true">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <form method="POST" action="workflows.php" id="editWorkflowForm">
                <input type="hidden" name="action" value="edit">
                <input type="hidden" name="id" id="edit-id">
                
                <div class="modal-header">
                    <h5 class="modal-title" id="editWorkflowModalLabel">Editar Workflow</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Fechar"></button>
                </div>
                
                <div class="modal-body">
                    <div class="mb-3">
                        <label for="edit-name" class="form-label">Nome <span class="text-danger">*</span></label>
                        <input type="text" class="form-control" id="edit-name" name="name" required>
                    </div>
                    
                    <div class="mb-3">
                        <label for="edit-description" class="form-label">Descrição</label>
                        <textarea class="form-control" id="edit-description" name="description" rows="2"></textarea>
                    </div>
                    
                    <div class="mb-3">
                        <label class="form-label">Etapas do Workflow <span class="text-danger">*</span></label>
                        <p class="text-muted small">Defina as etapas que os documentos devem passar neste workflow.</p>
                        
                        <div id="edit-steps-container">
                            <!-- As etapas serão adicionadas aqui dinamicamente -->
                        </div>
                        
                        <button type="button" class="btn btn-outline-secondary mt-2" id="edit-step-btn">
                            <i class="fas fa-plus me-1"></i> Adicionar Etapa
                        </button>
                    </div>
                </div>
                
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancelar</button>
                    <button type="submit" class="btn btn-primary">Salvar</button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Modal para visualizar workflow -->
<div class="modal fade" id="viewWorkflowModal" tabindex="-1" aria-labelledby="viewWorkflowModalLabel" aria-hidden="true">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="viewWorkflowModalLabel">Visualizar Workflow</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Fechar"></button>
            </div>
            
            <div class="modal-body">
                <h4 id="view-name" class="mb-3"></h4>
                
                <div class="workflow-diagram mb-4">
                    <div id="view-steps-diagram" class="d-flex justify-content-center">
                        <!-- O diagrama será gerado aqui -->
                    </div>
                </div>
                
                <div class="card">
                    <div class="card-header">
                        <h5 class="mb-0">Detalhes das Etapas</h5>
                    </div>
                    <div class="card-body p-0">
                        <div class="list-group list-group-flush" id="view-steps-list">
                            <!-- A lista de etapas será gerada aqui -->
                        </div>
                    </div>
                </div>
            </div>
            
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Fechar</button>
            </div>
        </div>
    </div>
</div>

<!-- Modal para excluir workflow -->
<div class="modal fade" id="deleteWorkflowModal" tabindex="-1" aria-labelledby="deleteWorkflowModalLabel" aria-hidden="true">
    <div class="modal-dialog">
        <div class="modal-content">
            <form method="POST" action="workflows.php">
                <input type="hidden" name="action" value="delete">
                <input type="hidden" name="id" id="delete-id">
                
                <div class="modal-header">
                    <h5 class="modal-title" id="deleteWorkflowModalLabel">Excluir Workflow</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Fechar"></button>
                </div>
                
                <div class="modal-body">
                    <p>Tem certeza que deseja excluir o workflow <strong id="delete-name"></strong>?</p>
                    <div id="delete-warning" class="alert alert-warning d-none">
                        <i class="fas fa-exclamation-triangle me-2"></i> Este workflow está sendo usado por <strong id="delete-count"></strong> documentos. Ao excluí-lo, os documentos ficarão sem workflow.
                    </div>
                </div>
                
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancelar</button>
                    <button type="submit" class="btn btn-danger">Excluir</button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Template para etapa de workflow -->
<template id="step-template">
    <div class="card mb-2 workflow-step">
        <div class="card-body p-3">
            <div class="d-flex align-items-center mb-2">
                <span class="step-number badge bg-primary me-2">1</span>
                <h6 class="mb-0 flex-grow-1">Etapa</h6>
                <button type="button" class="btn btn-sm btn-outline-danger remove-step">
                    <i class="fas fa-times"></i>
                </button>
            </div>
            <div class="row">
                <div class="col-md-6 mb-2">
                    <label class="form-label">Nome da Etapa <span class="text-danger">*</span></label>
                    <input type="text" class="form-control step-name" name="steps[0][name]" required>
                </div>
                <div class="col-md-6 mb-2">
                    <label class="form-label">Responsável <span class="text-danger">*</span></label>
                    <select class="form-select step-role" name="steps[0][role]" required>
                        <option value="admin">Administrador</option>
                        <option value="manager">Gerente</option>
                        <option value="supervisor">Supervisor</option>
                        <option value="user">Usuário</option>
                    </select>
                </div>
            </div>
            <div class="mb-2">
                <label class="form-label">Descrição</label>
                <textarea class="form-control step-description" name="steps[0][description]" rows="2"></textarea>
            </div>
        </div>
    </div>
</template>

<!-- CSS para diagrama de workflow -->
<style>
.workflow-diagram {
    overflow-x: auto;
}

.workflow-step-box {
    min-width: 150px;
    padding: 10px;
    border: 2px solid var(--bs-primary);
    border-radius: 5px;
    background-color: rgba(var(--bs-primary-rgb), 0.1);
    text-align: center;
    position: relative;
}

.workflow-step-arrow {
    position: relative;
    width: 50px;
    height: 2px;
    background-color: var(--bs-primary);
}

.workflow-step-arrow:after {
    content: '';
    position: absolute;
    right: 0;
    top: -4px;
    width: 0;
    height: 0;
    border-left: 8px solid var(--bs-primary);
    border-top: 5px solid transparent;
    border-bottom: 5px solid transparent;
}

.workflow-step-role {
    font-size: 0.8rem;
    color: var(--bs-secondary);
}
</style>

<!-- JavaScript para manipulação dos modais e etapas de workflow -->
<script>
document.addEventListener('DOMContentLoaded', function() {
    // Função para atualizar os números das etapas
    function updateStepNumbers(container) {
        const steps = container.querySelectorAll('.workflow-step');
        steps.forEach((step, index) => {
            step.querySelector('.step-number').textContent = index + 1;
            
            // Atualizar os nomes dos campos para manter a sequência correta
            const nameInput = step.querySelector('.step-name');
            const roleSelect = step.querySelector('.step-role');
            const descTextarea = step.querySelector('.step-description');
            
            nameInput.name = `steps[${index}][name]`;
            roleSelect.name = `steps[${index}][role]`;
            descTextarea.name = `steps[${index}][description]`;
        });
    }
    
    // Função para adicionar uma nova etapa
    function addStep(container, data = null) {
        const template = document.getElementById('step-template');
        const stepNode = template.content.cloneNode(true);
        const step = stepNode.querySelector('.workflow-step');
        
        // Preencher com dados se fornecidos
        if (data) {
            step.querySelector('.step-name').value = data.name;
            step.querySelector('.step-role').value = data.role;
            step.querySelector('.step-description').value = data.description || '';
        }
        
        // Adicionar evento para remover etapa
        step.querySelector('.remove-step').addEventListener('click', function() {
            step.remove();
            updateStepNumbers(container);
        });
        
        container.appendChild(step);
        updateStepNumbers(container);
    }
    
    // Adicionar etapa no formulário de adição
    document.getElementById('add-step-btn').addEventListener('click', function() {
        addStep(document.getElementById('add-steps-container'));
    });
    
    // Adicionar etapa no formulário de edição
    document.getElementById('edit-step-btn').addEventListener('click', function() {
        addStep(document.getElementById('edit-steps-container'));
    });
    
    // Adicionar pelo menos uma etapa ao abrir o modal de adição
    document.getElementById('addWorkflowModal').addEventListener('show.bs.modal', function() {
        const container = document.getElementById('add-steps-container');
        container.innerHTML = '';
        addStep(container);
    });
    
    // Manipular clique no botão de visualizar
    document.querySelectorAll('.view-workflow').forEach(function(button) {
        button.addEventListener('click', function() {
            const id = this.getAttribute('data-id');
            const name = this.getAttribute('data-name');
            const steps = JSON.parse(this.getAttribute('data-steps'));
            
            document.getElementById('view-name').textContent = name;
            
            // Gerar diagrama de workflow
            const diagramContainer = document.getElementById('view-steps-diagram');
            diagramContainer.innerHTML = '';
            
            const stepsContainer = document.createElement('div');
            stepsContainer.className = 'd-flex align-items-center';
            
            steps.forEach((step, index) => {
                // Adicionar caixa de etapa
                const stepBox = document.createElement('div');
                stepBox.className = 'workflow-step-box';
                stepBox.innerHTML = `
                    <div class="fw-bold">${step.name}</div>
                    <div class="workflow-step-role">${getRoleName(step.role)}</div>
                `;
                stepsContainer.appendChild(stepBox);
                
                // Adicionar seta se não for a última etapa
                if (index < steps.length - 1) {
                    const arrow = document.createElement('div');
                    arrow.className = 'workflow-step-arrow mx-2';
                    stepsContainer.appendChild(arrow);
                }
            });
            
            diagramContainer.appendChild(stepsContainer);
            
            // Gerar lista de etapas
            const stepsList = document.getElementById('view-steps-list');
            stepsList.innerHTML = '';
            
            steps.forEach((step, index) => {
                const stepItem = document.createElement('div');
                stepItem.className = 'list-group-item';
                stepItem.innerHTML = `
                    <div class="d-flex align-items-center">
                        <span class="badge bg-primary me-3">${index + 1}</span>
                        <div>
                            <h6 class="mb-1">${step.name}</h6>
                            <p class="mb-1"><strong>Responsável:</strong> ${getRoleName(step.role)}</p>
                            ${step.description ? `<p class="mb-0 text-muted">${step.description}</p>` : ''}
                        </div>
                    </div>
                `;
                stepsList.appendChild(stepItem);
            });
            
            const viewModal = new bootstrap.Modal(document.getElementById('viewWorkflowModal'));
            viewModal.show();
        });
    });
    
    // Manipular clique no botão de editar
    document.querySelectorAll('.edit-workflow').forEach(function(button) {
        button.addEventListener('click', function() {
            const id = this.getAttribute('data-id');
            const name = this.getAttribute('data-name');
            const description = this.getAttribute('data-description');
            const steps = JSON.parse(this.getAttribute('data-steps'));
            
            document.getElementById('edit-id').value = id;
            document.getElementById('edit-name').value = name;
            document.getElementById('edit-description').value = description;
            
            // Limpar e adicionar etapas
            const container = document.getElementById('edit-steps-container');
            container.innerHTML = '';
            
            steps.forEach(step => {
                addStep(container, step);
            });
            
            const editModal = new bootstrap.Modal(document.getElementById('editWorkflowModal'));
            editModal.show();
        });
    });
    
    // Manipular clique no botão de excluir
    document.querySelectorAll('.delete-workflow').forEach(function(button) {
        button.addEventListener('click', function() {
            const id = this.getAttribute('data-id');
            const name = this.getAttribute('data-name');
            const count = parseInt(this.getAttribute('data-count'));
            
            document.getElementById('delete-id').value = id;
            document.getElementById('delete-name').textContent = name;
            
            // Mostrar aviso se o workflow estiver sendo usado
            if (count > 0) {
                document.getElementById('delete-count').textContent = count;
                document.getElementById('delete-warning').classList.remove('d-none');
            } else {
                document.getElementById('delete-warning').classList.add('d-none');
            }
            
            const deleteModal = new bootstrap.Modal(document.getElementById('deleteWorkflowModal'));
            deleteModal.show();
        });
    });
    
    // Validar formulários antes de enviar
    document.getElementById('addWorkflowForm').addEventListener('submit', validateWorkflowForm);
    document.getElementById('editWorkflowForm').addEventListener('submit', validateWorkflowForm);
    
    function validateWorkflowForm(e) {
        const form = e.target;
        const steps = form.querySelectorAll('.workflow-step');
        
        if (steps.length === 0) {
            e.preventDefault();
            alert('O workflow deve ter pelo menos uma etapa.');
            return false;
        }
        
        return true;
    }
    
    // Função para obter nome do papel
    function getRoleName(role) {
        switch (role) {
            case 'admin': return 'Administrador';
            case 'manager': return 'Gerente';
            case 'supervisor': return 'Supervisor';
            case 'user': return 'Usuário';
            default: return role;
        }
    }
});
</script>

<?php
// Incluir rodapé
include 'includes/footer.php';
?>
