<?php
/**
 * Arquivo de configuração principal
 * Sistema de Gerenciamento Eletrônico de Documentos (GED)
 */

// Iniciar sessão se ainda não foi iniciada
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

// Carregar configurações de ambiente
require_once __DIR__ . '/environment.php';

// Definições de caminho
define("BASE_PATH", dirname(__DIR__));
define("INCLUDE_PATH", BASE_PATH . "/includes");
define("MODEL_PATH", BASE_PATH . "/models");
define("VIEW_PATH", BASE_PATH . "/views");
define("CONTROLLER_PATH", BASE_PATH . "/controllers");
define("ASSET_PATH", BASE_PATH . "/assets");

// Criar diretório de logs se não existir
if (!file_exists(BASE_PATH . "/logs")) {
    mkdir(BASE_PATH . "/logs", 0755, true);
}

// Configurações de log
ini_set("error_log", BASE_PATH . "/logs/error.log");

// Função para carregar automaticamente classes
spl_autoload_register(function ($class_name) {
    // Verificar em models
    if (file_exists(MODEL_PATH . "/" . strtolower($class_name) . ".php")) {
        require_once MODEL_PATH . "/" . strtolower($class_name) . ".php";
        return;
    }
    
    // Verificar em controllers
    if (file_exists(CONTROLLER_PATH . "/" . strtolower($class_name) . ".php")) {
        require_once CONTROLLER_PATH . "/" . strtolower($class_name) . ".php";
        return;
    }
    
    // Verificar em includes
    if (file_exists(INCLUDE_PATH . "/" . strtolower($class_name) . ".php")) {
        require_once INCLUDE_PATH . "/" . strtolower($class_name) . ".php";
        return;
    }
});

// Função para sanitizar entrada
function sanitize($input) {
    if (is_array($input)) {
        foreach ($input as $key => $value) {
            $input[$key] = sanitize($value);
        }
    } else {
        $input = htmlspecialchars(trim($input), ENT_QUOTES, "UTF-8");
    }
    return $input;
}

// Função para redirecionar
function redirect($url) {
    header("Location: $url");
    exit;
}

// Função para verificar se o usuário está autenticado
function isAuthenticated() {
    return isset($_SESSION["user_id"]);
}

// Função para verificar permissão
function hasPermission($permission) {
    if (!isAuthenticated()) {
        return false;
    }
    
    // Administradores têm todas as permissões
    if ($_SESSION["user_type"] === "admin") {
        return true;
    }
    
    return in_array($permission, $_SESSION["permissions"] ?? []);
}

// Função para exigir autenticação
function requireAuth() {
    if (!isAuthenticated()) {
        $_SESSION["redirect_after_login"] = $_SERVER["REQUEST_URI"];
        redirect("/login.php");
    }
}

// Função para exigir permissão
function requirePermission($permission) {
    requireAuth();
    
    if (!hasPermission($permission)) {
        redirect("/access-denied.php");
    }
}

// Função para gerar token CSRF
function generateCSRFToken() {
    if (empty($_SESSION["csrf_token"])) {
        $_SESSION["csrf_token"] = bin2hex(random_bytes(32));
    }
    return $_SESSION["csrf_token"];
}

// Função para verificar token CSRF
function verifyCSRFToken($token) {
    return isset($_SESSION["csrf_token"]) && hash_equals($_SESSION["csrf_token"], $token);
}
?>

