<?php
/**
 * Página de aprovação de documentos
 * Sistema de Gerenciamento Eletrônico de Documentos (GED)
 */

// Iniciar sessão e verificar autenticação
session_start();
if (!isset($_SESSION["user_id"])) {
    header("Location: login.php");
    exit;
}

// Incluir arquivos necessários
require_once "config/database.php";
require_once "models/document.php";
require_once "models/workflow.php";
require_once "models/workflow_step.php";
require_once "models/workflow_approval.php";
require_once "controllers/document_controller.php";
require_once "controllers/workflow_controller.php";

// Conectar ao banco de dados
$database = new Database();
$db = $database->getConnection();

// Instanciar controladores
$document_controller = new DocumentController();
$workflow_controller = new WorkflowController();

// Processar ações de aprovação/rejeição
if ($_SERVER["REQUEST_METHOD"] == "POST") {
    if (isset($_POST["action"])) {
        $document_id = $_POST["document_id"] ?? 0;
        $workflow_step_id = $_POST["workflow_step_id"] ?? 0;
        $comments = $_POST["comments"] ?? "";
        
        if ($document_id && $workflow_step_id) {
            if ($_POST["action"] == "approve") {
                $result = $workflow_controller->approveDocumentStep($document_id, $workflow_step_id, $comments, $_SESSION["user_id"]);
                
                if ($result["success"]) {
                    $_SESSION["flash_message"] = "Documento aprovado com sucesso.";
                    $_SESSION["flash_type"] = "success";
                } else {
                    $_SESSION["flash_message"] = $result["message"];
                    $_SESSION["flash_type"] = "danger";
                }
            } 
            else if ($_POST["action"] == "reject") {
                $result = $workflow_controller->rejectDocumentStep($document_id, $workflow_step_id, $comments, $_SESSION["user_id"]);
                
                if ($result["success"]) {
                    $_SESSION["flash_message"] = "Documento rejeitado com sucesso.";
                    $_SESSION["flash_type"] = "success";
                } else {
                    $_SESSION["flash_message"] = $result["message"];
                    $_SESSION["flash_type"] = "danger";
                }
            }
        } else {
            $_SESSION["flash_message"] = "Parâmetros inválidos.";
            $_SESSION["flash_type"] = "danger";
        }
        
        // Redirecionar para evitar reenvio do formulário
        header("Location: document_approval.php");
        exit;
    }
}

// Obter documentos pendentes de aprovação para o usuário atual
$pending_documents = $workflow_controller->getPendingApprovalsForUser($_SESSION["user_id"], $_SESSION["user_type"]);

// Definir título da página
$page_title = "Aprovação de Documentos";

// Incluir cabeçalho
include "includes/header.php";
?>

<div class="row mb-4">
    <div class="col-md-6">
        <h1><i class="fas fa-check-double me-2"></i> Aprovação de Documentos</h1>
    </div>
    <div class="col-md-6 text-md-end">
        <a href="documents.php" class="btn btn-secondary">
            <i class="fas fa-arrow-left me-1"></i> Voltar para Documentos
        </a>
    </div>
</div>

<!-- Documentos pendentes de aprovação -->
<div class="card">
    <div class="card-header">
        <h5 class="mb-0">Documentos Pendentes de Aprovação</h5>
    </div>
    <div class="card-body">
        <?php if (count($pending_documents) > 0): ?>
        <div class="table-responsive">
            <table class="table table-hover datatable">
                <thead>
                    <tr>
                        <th>ID</th>
                        <th>Título</th>
                        <th>Categoria</th>
                        <th>Workflow</th>
                        <th>Etapa Atual</th>
                        <th>Enviado por</th>
                        <th>Data de Envio</th>
                        <th>Ações</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($pending_documents as $doc): ?>
                    <tr>
                        <td><?php echo $doc["document_id"]; ?></td>
                        <td>
                            <a href="document_view.php?id=<?php echo $doc["document_id"]; ?>" class="text-decoration-none">
                                <?php echo htmlspecialchars($doc["document_title"]); ?>
                            </a>
                        </td>
                        <td><?php echo htmlspecialchars($doc["category_name"]); ?></td>
                        <td><?php echo htmlspecialchars($doc["workflow_name"]); ?></td>
                        <td>
                            <span class="badge bg-warning text-dark">
                                <?php echo htmlspecialchars($doc["current_step_name"]); ?>
                            </span>
                        </td>
                        <td><?php echo htmlspecialchars($doc["requester_name"]); ?></td>
                        <td><?php echo date("d/m/Y H:i", strtotime($doc["created_at"])); ?></td>
                        <td>
                            <div class="btn-group btn-group-sm">
                                <a href="document_view.php?id=<?php echo $doc["document_id"]; ?>" class="btn btn-info" title="Visualizar">
                                    <i class="fas fa-eye"></i>
                                </a>
                                <button type="button" class="btn btn-success approve-document" 
                                        data-id="<?php echo $doc["document_id"]; ?>"
                                        data-title="<?php echo htmlspecialchars($doc["document_title"]); ?>"
                                        data-step-id="<?php echo $doc["current_step_id"]; ?>"
                                        data-step-name="<?php echo htmlspecialchars($doc["current_step_name"]); ?>"
                                        title="Aprovar">
                                    <i class="fas fa-check"></i>
                                </button>
                                <button type="button" class="btn btn-danger reject-document" 
                                        data-id="<?php echo $doc["document_id"]; ?>"
                                        data-title="<?php echo htmlspecialchars($doc["document_title"]); ?>"
                                        data-step-id="<?php echo $doc["current_step_id"]; ?>"
                                        data-step-name="<?php echo htmlspecialchars($doc["current_step_name"]); ?>"
                                        title="Rejeitar">
                                    <i class="fas fa-times"></i>
                                </button>
                            </div>
                        </td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
        <?php else: ?>
        <div class="alert alert-info">
            <i class="fas fa-info-circle me-2"></i> Não há documentos pendentes de aprovação.
        </div>
        <?php endif; ?>
    </div>
</div>

<!-- Modal para aprovar documento -->
<div class="modal fade" id="approveModal" tabindex="-1" aria-labelledby="approveModalLabel" aria-hidden="true">
    <div class="modal-dialog">
        <div class="modal-content">
            <form method="POST" action="document_approval.php">
                <input type="hidden" name="action" value="approve">
                <input type="hidden" name="document_id" id="approve-document-id">
                <input type="hidden" name="workflow_step_id" id="approve-step-id">
                
                <div class="modal-header">
                    <h5 class="modal-title" id="approveModalLabel">Aprovar Documento</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Fechar"></button>
                </div>
                
                <div class="modal-body">
                    <p>Você está aprovando o documento <strong id="approve-document-title"></strong> na etapa <strong id="approve-step-name"></strong>.</p>
                    
                    <div class="mb-3">
                        <label for="approve-comments" class="form-label">Comentários (opcional)</label>
                        <textarea class="form-control" id="approve-comments" name="comments" rows="3" placeholder="Adicione comentários sobre esta aprovação..."></textarea>
                    </div>
                </div>
                
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancelar</button>
                    <button type="submit" class="btn btn-success">
                        <i class="fas fa-check me-1"></i> Aprovar
                    </button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Modal para rejeitar documento -->
<div class="modal fade" id="rejectModal" tabindex="-1" aria-labelledby="rejectModalLabel" aria-hidden="true">
    <div class="modal-dialog">
        <div class="modal-content">
            <form method="POST" action="document_approval.php">
                <input type="hidden" name="action" value="reject">
                <input type="hidden" name="document_id" id="reject-document-id">
                <input type="hidden" name="workflow_step_id" id="reject-step-id">
                
                <div class="modal-header">
                    <h5 class="modal-title" id="rejectModalLabel">Rejeitar Documento</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Fechar"></button>
                </div>
                
                <div class="modal-body">
                    <p>Você está rejeitando o documento <strong id="reject-document-title"></strong> na etapa <strong id="reject-step-name"></strong>.</p>
                    
                    <div class="mb-3">
                        <label for="reject-comments" class="form-label">Motivo da Rejeição <span class="text-danger">*</span></label>
                        <textarea class="form-control" id="reject-comments" name="comments" rows="3" placeholder="Explique o motivo da rejeição..." required></textarea>
                    </div>
                </div>
                
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancelar</button>
                    <button type="submit" class="btn btn-danger">
                        <i class="fas fa-times me-1"></i> Rejeitar
                    </button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- JavaScript para manipulação dos modais -->
<script>
document.addEventListener("DOMContentLoaded", function() {
    // Manipular clique no botão de aprovar
    document.querySelectorAll(".approve-document").forEach(function(button) {
        button.addEventListener("click", function() {
            const id = this.getAttribute("data-id");
            const title = this.getAttribute("data-title");
            const stepId = this.getAttribute("data-step-id");
            const stepName = this.getAttribute("data-step-name");
            
            document.getElementById("approve-document-id").value = id;
            document.getElementById("approve-document-title").textContent = title;
            document.getElementById("approve-step-id").value = stepId;
            document.getElementById("approve-step-name").textContent = stepName;
            
            const approveModal = new bootstrap.Modal(document.getElementById("approveModal"));
            approveModal.show();
        });
    });
    
    // Manipular clique no botão de rejeitar
    document.querySelectorAll(".reject-document").forEach(function(button) {
        button.addEventListener("click", function() {
            const id = this.getAttribute("data-id");
            const title = this.getAttribute("data-title");
            const stepId = this.getAttribute("data-step-id");
            const stepName = this.getAttribute("data-step-name");
            
            document.getElementById("reject-document-id").value = id;
            document.getElementById("reject-document-title").textContent = title;
            document.getElementById("reject-step-id").value = stepId;
            document.getElementById("reject-step-name").textContent = stepName;
            
            const rejectModal = new bootstrap.Modal(document.getElementById("rejectModal"));
            rejectModal.show();
        });
    });
});
</script>

<?php
// Incluir rodapé
include "includes/footer.php";
?>


