<?php
/**
 * Modelo de Workflow
 * Sistema de Gerenciamento Eletrônico de Documentos (GED)
 */

require_once __DIR__ . '/workflow_step.php';

class Workflow {
    // Conexão com o banco e nome da tabela
    private $conn;
    private $table_name = "workflow";
    private $workflow_step;

    // Propriedades do objeto
    public $id;
    public $name;
    public $description;
    public $created_at;
    public $created_by;

    /**
     * Construtor com $db como conexão com o banco de dados
     * @param PDO $db Conexão com o banco de dados
     */
    public function __construct($db) {
        $this->conn = $db;
        $this->workflow_step = new WorkflowStep($db);
    }

    /**
     * Lê todos os workflows
     * @return PDOStatement Resultado da consulta
     */
    public function readAll() {
        $query = "SELECT w.*, u.full_name as created_by_username, COUNT(d.id) as document_count
                  FROM " . $this->table_name . " w
                  LEFT JOIN user u ON w.created_by = u.id
                  LEFT JOIN document d ON d.workflow_id = w.id
                  GROUP BY w.id
                  ORDER BY w.name ASC";
        
        $stmt = $this->conn->prepare($query);
        $stmt->execute();
        
        return $stmt;
    }

    /**
     * Lê um único workflow pelo ID
     * @param int $id ID do workflow
     * @return bool Verdadeiro se encontrado, falso caso contrário
     */
    public function readOne($id) {
        $query = "SELECT w.*, u.full_name as created_by_username
                  FROM " . $this->table_name . " w
                  LEFT JOIN user u ON w.created_by = u.id
                  WHERE w.id = ? LIMIT 0,1";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(1, $id);
        $stmt->execute();

        if ($stmt->rowCount() > 0) {
            $row = $stmt->fetch(PDO::FETCH_ASSOC);
            
            $this->id = $row['id'];
            $this->name = $row['name'];
            $this->description = $row['description'];
            $this->created_at = $row['created_at'];
            $this->created_by = $row['created_by'];
            
            return true;
        }
        
        return false;
    }

    /**
     * Cria um novo workflow
     * @return bool Verdadeiro se criado com sucesso, falso caso contrário
     */
    public function create() {
        $query = "INSERT INTO " . $this->table_name . "
                  SET name=:name, description=:description, created_by=:created_by";
        
        $stmt = $this->conn->prepare($query);
        
        // Sanitizar
        $this->name = htmlspecialchars(strip_tags($this->name));
        $this->description = htmlspecialchars(strip_tags($this->description));
        $this->created_by = htmlspecialchars(strip_tags($this->created_by));
        
        // Vincular valores
        $stmt->bindParam(":name", $this->name);
        $stmt->bindParam(":description", $this->description);
        $stmt->bindParam(":created_by", $this->created_by);
        
        // Executar
        if ($stmt->execute()) {
            $this->id = $this->conn->lastInsertId();
            return true;
        }
        
        return false;
    }

    /**
     * Atualiza um workflow existente
     * @return bool Verdadeiro se atualizado com sucesso, falso caso contrário
     */
    public function update() {
        $query = "UPDATE " . $this->table_name . "
                  SET name = :name, description = :description
                  WHERE id = :id";
        
        $stmt = $this->conn->prepare($query);
        
        // Sanitizar
        $this->name = htmlspecialchars(strip_tags($this->name));
        $this->description = htmlspecialchars(strip_tags($this->description));
        $this->id = htmlspecialchars(strip_tags($this->id));
        
        // Vincular valores
        $stmt->bindParam(":name", $this->name);
        $stmt->bindParam(":description", $this->description);
        $stmt->bindParam(":id", $this->id);
        
        // Executar
        if ($stmt->execute()) {
            return true;
        }
        
        return false;
    }

    /**
     * Exclui um workflow
     * @return bool Verdadeiro se excluído com sucesso, falso caso contrário
     */
    public function delete() {
        // Verificar se há documentos usando este workflow
        $query = "SELECT COUNT(*) as count FROM document WHERE workflow_id = ?";
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(1, $this->id);
        $stmt->execute();
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($row['count'] > 0) {
            // Não permitir exclusão se houver documentos associados
            return false;
        }
        
        // Excluir etapas do workflow
        $this->workflow_step->workflow_id = $this->id;
        $this->workflow_step->deleteByWorkflowId();
        
        // Excluir workflow
        $query = "DELETE FROM " . $this->table_name . " WHERE id = ?";
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(1, $this->id);
        
        if ($stmt->execute()) {
            return true;
        }
        
        return false;
    }

    /**
     * Obtém as etapas de um workflow
     * @return array Lista de etapas
     */
    public function getSteps() {
        $this->workflow_step->workflow_id = $this->id;
        $stmt = $this->workflow_step->readByWorkflowId();
        
        $steps = [];
        while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
            $steps[] = $row;
        }
        
        return $steps;
    }

    /**
     * Adiciona uma etapa ao workflow
     * @param array $data Dados da etapa (name, description, assigned_to_user_type, assigned_to_user_id)
     * @return bool Verdadeiro se criado com sucesso, falso caso contrário
     */
    public function addStep($data) {
        $this->workflow_step->workflow_id = $this->id;
        $this->workflow_step->step_order = $this->workflow_step->getLatestStepOrder($this->id) + 1;
        $this->workflow_step->name = $data['name'];
        $this->workflow_step->description = $data['description'] ?? '';
        $this->workflow_step->assigned_to_user_type = $data['assigned_to_user_type'] ?? 'any';
        $this->workflow_step->assigned_to_user_id = $data['assigned_to_user_id'] ?? null;

        return $this->workflow_step->create();
    }

    /**
     * Atualiza uma etapa do workflow
     * @param int $step_id ID da etapa
     * @param array $data Novos dados da etapa (name, description, assigned_to_user_type, assigned_to_user_id)
     * @return bool Verdadeiro se atualizado com sucesso, falso caso contrário
     */
    public function updateStep($step_id, $data) {
        $this->workflow_step->id = $step_id;
        if (!$this->workflow_step->readOne()) {
            return false; // Etapa não encontrada
        }

        $this->workflow_step->name = $data['name'];
        $this->workflow_step->description = $data['description'] ?? $this->workflow_step->description;
        $this->workflow_step->assigned_to_user_type = $data['assigned_to_user_type'] ?? $this->workflow_step->assigned_to_user_type;
        $this->workflow_step->assigned_to_user_id = $data['assigned_to_user_id'] ?? $this->workflow_step->assigned_to_user_id;

        return $this->workflow_step->update();
    }

    /**
     * Remove uma etapa do workflow
     * @param int $step_id ID da etapa
     * @return bool Verdadeiro se removido com sucesso, falso caso contrário
     */
    public function removeStep($step_id) {
        $this->workflow_step->id = $step_id;
        if (!$this->workflow_step->readOne()) {
            return false; // Etapa não encontrada
        }
        
        // Reordenar etapas após a remoção
        $removed_order = $this->workflow_step->step_order;
        $workflow_id = $this->workflow_step->workflow_id;

        if ($this->workflow_step->delete()) {
            $this->workflow_step->reorderStepsAfterDelete($workflow_id, $removed_order);
            return true;
        }
        return false;
    }

    /**
     * Reordena as etapas do workflow
     * @param array $step_ids Array de IDs das etapas na nova ordem
     * @return bool Verdadeiro se reordenado com sucesso, falso caso contrário
     */
    public function reorderSteps($step_ids) {
        $this->conn->beginTransaction();
        try {
            foreach ($step_ids as $order => $step_id) {
                $this->workflow_step->id = $step_id;
                if ($this->workflow_step->readOne()) {
                    $this->workflow_step->step_order = $order + 1; // Ordem baseada em 1
                    $this->workflow_step->updateOrder(); // Novo método para atualizar apenas a ordem
                }
            }
            $this->conn->commit();
            return true;
        } catch (PDOException $e) {
            $this->conn->rollBack();
            error_log("Erro ao reordenar etapas do workflow: " . $e->getMessage());
            return false;
        }
    }

    /**
     * Obtém aprovações pendentes para um usuário.
     * @param int $user_id ID do usuário.
     * @return array Lista de documentos pendentes de aprovação.
     */
    public function getPendingApprovalsForUser($user_id) {
        $query = "SELECT
                    da.id as approval_id, da.document_id, da.workflow_id, da.current_step_id, da.status as approval_status,
                    d.title as document_title, d.created_by as document_created_by,
                    u.full_name as requester_name,
                    ws.name as current_step_name, ws.assigned_to_user_type, ws.assigned_to_user_id
                  FROM
                    workflow_approvals da
                    JOIN document d ON da.document_id = d.id
                    JOIN user u ON d.created_by = u.id
                    JOIN workflow_steps ws ON da.current_step_id = ws.id
                  WHERE
                    da.status = 'pending' AND (
                        (ws.assigned_to_user_type = 'user' AND ws.assigned_to_user_id = :user_id) OR
                        (ws.assigned_to_user_type = 'admin' AND (SELECT user_type FROM user WHERE id = :user_id) = 'admin') OR
                        (ws.assigned_to_user_type = 'any')
                    )
                  ORDER BY da.created_at DESC";

        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':user_id', $user_id, PDO::PARAM_INT);
        $stmt->execute();

        $pending_approvals = [];
        while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
            $pending_approvals[] = $row;
        }
        return $pending_approvals;
    }
}
?>

