<?php
/**
 * Script de teste de integração para o Sistema GED
 * Verifica a integração entre os módulos PHP e o banco de dados MySQL
 */

// Iniciar sessão para testes de autenticação ANTES de qualquer saída
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

// Incluir arquivos necessários
require_once 'config/config.php'; // Incluir config.php primeiro para iniciar a sessão e definir constantes
require_once 'config/database.php';

// Definir constantes para o ambiente de teste APÓS incluir config.php
define('TEST_MODE', true);
define('TEST_USER_ID', 1);

require_once 'models/user.php';
require_once 'models/document.php';
require_once 'models/category.php';
require_once 'models/workflow.php';
require_once 'models/audit_log.php';
require_once 'models/permission.php';
require_once 'models/user_permission.php';
require_once 'models/workflow_step.php';
require_once 'models/workflow_approval.php';

require_once 'controllers/auth_controller.php';
require_once 'controllers/document_controller.php';
require_once 'controllers/category_controller.php';
require_once 'controllers/workflow_controller.php';
require_once 'controllers/audit_log_controller.php';
require_once 'controllers/settings_controller.php';
require_once 'controllers/search_controller.php';
require_once 'controllers/user_controller.php';
require_once 'controllers/permission_controller.php';

// Função para exibir resultados de teste
function test_result($test_name, $success, $message = '') {
    echo "<div style='margin: 5px; padding: 10px; border: 1px solid " . ($success ? "#4CAF50" : "#F44336") . "; background-color: " . ($success ? "#E8F5E9" : "#FFEBEE") . ";'>";
    echo "<strong>" . ($success ? "✓ PASSOU" : "✗ FALHOU") . ":</strong> $test_name";
    if ($message) {
        echo "<br><em>$message</em>";
    }
    echo "</div>";
    
    return $success;
}

// Iniciar HTML
echo "<!DOCTYPE html>\n<html lang='pt-BR'>\n<head>\n    <meta charset='UTF-8'>\n    <meta name='viewport' content='width=device-width, initial-scale=1.0'>\n    <title>Teste de Integração - Sistema GED</title>\n    <style>\n        body { font-family: Arial, sans-serif; margin: 20px; }\n        h1, h2 { color: #333; }\n        .summary { margin-top: 20px; padding: 10px; background-color: #f5f5f5; border-radius: 5px; }\n        .success { color: #4CAF50; }\n        .failure { color: #F44336; }\n    </style>\n</head>\n<body>\n    <h1>Teste de Integração - Sistema GED</h1>\n    <p>Este script verifica a integração entre os módulos PHP e o banco de dados MySQL.</p>\n";

// Inicializar contadores
$tests_total = 0;
$tests_passed = 0;

// Testar conexão com o banco de dados
echo "<h2>1. Teste de Conexão com o Banco de Dados</h2>";

try {
    $database = new Database();
    $db = $database->getConnection();
    
    $query = "SELECT 1";
    $stmt = $db->prepare($query);
    $stmt->execute();
    
    $success = test_result("Conexão com o banco de dados", true, "Conexão estabelecida com sucesso");
    $tests_total++;
    if ($success) $tests_passed++;
} catch (PDOException $e) {
    test_result("Conexão com o banco de dados", false, "Erro: " . $e->getMessage());
    $tests_total++;
}

// Testar autenticação e usuários
echo "<h2>2. Teste de Autenticação e Usuários</h2>";

try {
    $auth_controller = new AuthController();
    $user_controller = new UserController();
    $permission_controller = new PermissionController();

    // Criar permissões básicas se não existirem
    $permissions_to_create = [
        'view_document' => 'Permissão para visualizar documentos',
        'upload_document' => 'Permissão para fazer upload de documentos',
        'edit_document' => 'Permissão para editar documentos',
        'delete_document' => 'Permissão para excluir documentos',
        'manage_users' => 'Permissão para gerenciar usuários',
        'manage_permissions' => 'Permissão para gerenciar permissões',
        'manage_categories' => 'Permissão para gerenciar categorias',
        'manage_workflows' => 'Permissão para gerenciar workflows',
        'view_audit_logs' => 'Permissão para visualizar logs de auditoria'
    ];

    foreach ($permissions_to_create as $name => $description) {
        $perm_exists_query = "SELECT COUNT(*) FROM permission WHERE name = ?";
        $stmt = $db->prepare($perm_exists_query);
        $stmt->bindParam(1, $name);
        $stmt->execute();
        $result = ['success' => false, 'message' => '']; // Inicializa $result
        if ($stmt->fetchColumn() == 0) {
            // Passa um user_id fictício para o teste, já que o login ainda não ocorreu
            $result = $permission_controller->createPermission(["name" => $name, "description" => $description], TEST_USER_ID);
            test_result("Criação de permissão: " . $name, $result['success'], $result['success'] ? "Permissão criada" : $result['message']);
        } else {
            $result = ['success' => true, 'message' => 'Permissão já existe']; // Define $result para o caso de sucesso
            test_result("Verificação de permissão: " . $name, true, "Permissão já existe");
        }
        $tests_total++;
        if ($result['success']) $tests_passed++;
    }
    
    // Testar criação de usuário admin (se não existir)
    $admin_user_exists = false;
    $query = "SELECT COUNT(*) FROM user WHERE username = 'admin'";
    $stmt = $db->prepare($query);
    $stmt->execute();
    if ($stmt->fetchColumn() > 0) {
        $admin_user_exists = true;
    }
    
    if (!$admin_user_exists) {
        $result = $user_controller->createUser([
            'username' => 'admin',
            'password' => 'admin123',
            'email' => 'admin@example.com',
            'full_name' => 'Administrador do Sistema',
            'user_type' => 'admin',
            'is_active' => 1,
            'permissions' => array_keys($permissions_to_create) // Atribuir todas as permissões ao admin
        ]);
        
        $success = test_result("Criação de usuário admin", $result['success'], $result['success'] ? "Usuário admin criado com sucesso" : $result['message']);
    } else {
        $success = test_result("Verificação de usuário admin", true, "Usuário admin já existe");
    }
    $tests_total++;
    if ($success) $tests_passed++;
    
    // Testar login
    $result = $auth_controller->login('admin', 'admin123');
    $success = test_result("Login de usuário", $result['success'], $result['success'] ? "Login realizado com sucesso" : $result['message']);
    $tests_total++;
    if ($success) $tests_passed++;
    
    // Simular sessão para testes subsequentes
    if ($result['success']) {
        $_SESSION['user_id'] = $result['user']['id'];
        $_SESSION['user_type'] = $result['user']['user_type'];
    }

    // Testar verificação de permissões para o admin
    $user_model = new User($db);
    $user_model->readOne(TEST_USER_ID);
    $has_permission = $user_model->hasPermission('view_document');
    $success = test_result("Verificação de permissões (admin)", $has_permission, "Verificação de permissão 'view_document' para usuário admin");
    $tests_total++;
    if ($success) $tests_passed++;

    // Testar atualização de usuário
    $update_user_data = [
        'id' => TEST_USER_ID,
        'email' => 'admin_updated@example.com',
        'full_name' => 'Administrador Atualizado',
        'user_type' => 'admin',
        'is_active' => 1,
        'permissions' => ['view_document', 'upload_document'] // Reduzir permissões para teste
    ];
    $result = $user_controller->updateUser($update_user_data);
    $success = test_result("Atualização de usuário", $result['success'], $result['success'] ? "Usuário atualizado com sucesso" : $result['message']);
    $tests_total++;
    if ($success) $tests_passed++;

    // Testar obtenção de todos os usuários
    $all_users = $user_controller->getAllUsers();
    $success = test_result("Obtenção de todos os usuários", is_array($all_users) && count($all_users) > 0, "Total de usuários: " . count($all_users));
    $tests_total++;
    if ($success) $tests_passed++;
    
} catch (Exception $e) {
    test_result("Teste de autenticação e usuários", false, "Erro: " . $e->getMessage());
    $tests_total++;
}

// Testar categorias
echo "<h2>3. Teste de Categorias</h2>";

try {
    $category_controller = new CategoryController();
    
    // Testar criação de categoria
    $category_name = "Categoria Teste " . date('YmdHis');
    $result = $category_controller->createCategory([
        'name' => $category_name,
        'description' => 'Categoria criada para teste de integração',
        'color' => '#3498db'
    ], TEST_USER_ID);
    
    $success = test_result("Criação de categoria", $result['success'], $result['success'] ? "Categoria criada com ID: " . $result['category_id'] : $result['message']);
    $tests_total++;
    if ($success) $tests_passed++;
    
    if ($result['success']) {
        $category_id = $result['category_id'];
        
        // Testar obtenção de categoria
        $category_result = $category_controller->getCategory($category_id);
        $success = test_result("Obtenção de categoria por ID", $category_result['success'], "Categoria obtida: " . ($category_result['success'] ? $category_result['category']['name'] : "Não encontrada"));
        $tests_total++;
        if ($success) $tests_passed++;
        
        // Testar atualização de categoria
        $update_result = $category_controller->updateCategory($category_id, [
            'name' => $category_name . " (Atualizada)",
            'description' => 'Categoria atualizada para teste de integração',
            'color' => '#2ecc71'
        ], TEST_USER_ID);
        
        $success = test_result("Atualização de categoria", $update_result['success'], $update_result['success'] ? "Categoria atualizada com sucesso" : $update_result['message']);
        $tests_total++;
        if ($success) $tests_passed++;

        // Testar exclusão de categoria
        $delete_result = $category_controller->deleteCategory($category_id, TEST_USER_ID);
        $success = test_result("Exclusão de categoria", $delete_result['success'], $delete_result['success'] ? "Categoria excluída com sucesso" : $delete_result['message']);
        $tests_total++;
        if ($success) $tests_passed++;
    }
    
    // Testar listagem de categorias
    $categories_result = $category_controller->listCategories();
    $success = test_result("Listagem de categorias", $categories_result['success'], "Total de categorias: " . count($categories_result['categories']));
    $tests_total++;
    if ($success) $tests_passed++;
    
} catch (Exception $e) {
    test_result("Teste de categorias", false, "Erro: " . $e->getMessage());
    $tests_total++;
}

// Testar documentos
echo "<h2>4. Teste de Documentos</h2>";

try {
    $document_controller = new DocumentController();
    
    // Criar uma categoria para o documento de teste, se não houver uma existente
    $category_model = new Category($db);
    $stmt_cat = $category_model->readAll();
    $existing_category = $stmt_cat->fetch(PDO::FETCH_ASSOC);
    if (!$existing_category) {
        $category_controller = new CategoryController();
        $cat_result = $category_controller->createCategory([
            'name' => 'Geral',
            'description' => 'Categoria geral para documentos',
            'color' => '#cccccc',
            'created_by' => TEST_USER_ID
        ]);
        $category_id = $cat_result['category_id'];
    } else {
        $category_id = $existing_category['id'];
    }

    // Simular um arquivo para upload
    $dummy_file_path = '/tmp/dummy_document.txt';
    file_put_contents($dummy_file_path, 'Este é um documento de teste.');

    $dummy_file = [
        'name' => 'documento_teste.txt',
        'type' => 'text/plain',
        'tmp_name' => $dummy_file_path,
        'error' => UPLOAD_ERR_OK,
        'size' => filesize($dummy_file_path)
    ];

    // Testar upload de documento
    $document_title = "Documento Teste " . date('YmdHis');
    $result = $document_controller->uploadDocument([
        'title' => $document_title,
        'description' => 'Documento criado para teste de integração',
        'category_id' => $category_id,
        'keywords' => 'teste, integração, php',
        'author_id' => TEST_USER_ID,
        'file' => $dummy_file
    ]);
    
    $success = test_result("Upload de documento", $result['success'], $result['success'] ? "Documento enviado com ID: " . $result['document_id'] : $result['message']);
    $tests_total++;
    if ($success) $tests_passed++;
    
    if ($result['success']) {
        $document_id = $result['document_id'];
        
        // Testar obtenção de documento
        $document_data = $document_controller->getDocument($document_id);
        $success = test_result("Obtenção de documento por ID", $document_data['success'], "Documento obtido: " . ($document_data['success'] ? $document_data['document']['title'] : "Não encontrado"));
        $tests_total++;
        if ($success) $tests_passed++;
        
        // Testar atualização de documento
        $update_result = $document_controller->updateDocument([
            'id' => $document_id,
            'title' => $document_title . " (Atualizado)",
            'description' => 'Documento atualizado para teste de integração',
            'category_id' => $category_id,
            'keywords' => 'teste, integração, atualizado',
            'user_id' => TEST_USER_ID
        ]);
        
        $success = test_result("Atualização de documento", $update_result['success'], $update_result['success'] ? "Documento atualizado com sucesso" : $update_result['message']);
        $tests_total++;
        if ($success) $tests_passed++;
        
        // Testar upload de nova versão de documento
        $dummy_file_v2_path = '/tmp/dummy_document_v2.txt';
        file_put_contents($dummy_file_v2_path, 'Este é um documento de teste - Versão 2.');
        $dummy_file_v2 = [
            'name' => 'documento_teste_v2.txt',
            'type' => 'text/plain',
            'tmp_name' => $dummy_file_v2_path,
            'error' => UPLOAD_ERR_OK,
            'size' => filesize($dummy_file_v2_path)
        ];

        $version_result = $document_controller->uploadNewDocumentVersion($document_id, $dummy_file_v2, TEST_USER_ID, 'Atualização para versão 2');
        $success = test_result("Upload de nova versão de documento", $version_result['success'], $version_result['success'] ? "Nova versão adicionada com sucesso" : $version_result['message']);
        $tests_total++;
        if ($success) $tests_passed++;

        // Testar restauração de versão de documento (restaurar para a primeira versão)
        $restore_result = $document_controller->restoreDocumentVersion($document_id, 1, TEST_USER_ID);
        $success = test_result("Restauração de versão de documento", $restore_result['success'], $restore_result['success'] ? "Versão restaurada com sucesso" : $restore_result['message']);
        $tests_total++;
        if ($success) $tests_passed++;

        // Testar exclusão de documento
        $delete_result = $document_controller->deleteDocument($document_id, TEST_USER_ID);
        $success = test_result("Exclusão de documento", $delete_result['success'], $delete_result['success'] ? "Documento excluído com sucesso" : $delete_result['message']);
        $tests_total++;
        if ($success) $tests_passed++;
    }
    
    // Testar pesquisa de documentos
    $search_controller = new SearchController();
    $search_result = $search_controller->advancedSearch(['title' => 'teste']);
    $success = test_result("Pesquisa de documentos", is_array($search_result), "Pesquisa realizada com sucesso");
    $tests_total++;
    if ($success) $tests_passed++;
    
} catch (Exception $e) {
    test_result("Teste de documentos", false, "Erro: " . $e->getMessage());
    $tests_total++;
}

// Testar workflow
echo "<h2>5. Teste de Workflow</h2>";

try {
    $workflow_controller = new WorkflowController();
    
    // Testar criação de workflow
    $workflow_name = "Workflow Teste " . date('YmdHis');
    $result = $workflow_controller->createWorkflow([
        'name' => $workflow_name,
        'description' => 'Workflow criado para teste de integração',
        'steps' => [
            ['name' => 'Revisão Inicial', 'description' => 'Revisão inicial do documento', 'assigned_to_user_type' => 'admin', 'assigned_to_user_id' => null],
            ['name' => 'Aprovação Gerencial', 'description' => 'Aprovação pelo gerente', 'assigned_to_user_type' => 'user', 'assigned_to_user_id' => TEST_USER_ID],
            ['name' => 'Publicação Final', 'description' => 'Publicação do documento', 'assigned_to_user_type' => 'any', 'assigned_to_user_id' => null]
        ]
    ], TEST_USER_ID);
    
    $success = test_result("Criação de workflow", $result['success'], $result['success'] ? "Workflow criado com ID: " . $result['workflow_id'] : $result['message']);
    $tests_total++;
    if ($success) $tests_passed++;
    
    if ($result['success']) {
        $workflow_id = $result['workflow_id'];
        
        // Criar um documento para associar ao workflow
        $dummy_file_path_wf = '/tmp/dummy_document_workflow.txt';
        file_put_contents($dummy_file_path_wf, 'Este é um documento para teste de workflow.');
        $dummy_file_wf = [
            'name' => 'documento_workflow.txt',
            'type' => 'text/plain',
            'tmp_name' => $dummy_file_path_wf,
            'error' => UPLOAD_ERR_OK,
            'size' => filesize($dummy_file_path_wf)
        ];

        $document_controller = new DocumentController();
        $cat_model = new Category($db);
        $stmt_cat_wf = $cat_model->readAll();
        $existing_category_wf = $stmt_cat_wf->fetch(PDO::FETCH_ASSOC);
        if (!$existing_category_wf) {
            $category_controller_wf = new CategoryController();
            $cat_result_wf = $category_controller_wf->createCategory([
                'name' => 'Workflow Category',
                'description' => 'Categoria para documentos de workflow',
                'color' => '#aabbcc',
                'created_by' => TEST_USER_ID
            ]);
            $category_id_wf = $cat_result_wf['category_id'];
        } else {
            $category_id_wf = $existing_category_wf['id'];
        }

        $doc_wf_result = $document_controller->uploadDocument([
            'title' => 'Documento para Workflow ' . date('YmdHis'),
            'description' => 'Documento para teste de workflow',
            'category_id' => $category_id_wf,
            'keywords' => 'workflow, teste',
            'author_id' => TEST_USER_ID,
            'file' => $dummy_file_wf,
            'workflow_id' => $workflow_id
        ]);
        $success = test_result("Upload de documento para workflow", $doc_wf_result['success'], $doc_wf_result['success'] ? "Documento para workflow enviado com ID: " . $doc_wf_result['document_id'] : $doc_wf_result['message']);
        $tests_total++;
        if ($success) $tests_passed++;

        if ($doc_wf_result['success']) {
            $document_wf_id = $doc_wf_result['document_id'];

            // Iniciar workflow para o documento
            $start_wf_result = $workflow_controller->startWorkflowForDocument($document_wf_id, TEST_USER_ID);
            $success = test_result("Iniciar workflow para documento", $start_wf_result['success'], $start_wf_result['success'] ? "Workflow iniciado para documento" : $start_wf_result['message']);
            $tests_total++;
            if ($success) $tests_passed++;

            // Obter aprovações pendentes
            $pending_approvals = $workflow_controller->getPendingApprovalsForUser(TEST_USER_ID, 'admin'); // Assumindo que TEST_USER_ID é admin
            $success = test_result("Obter aprovações pendentes", is_array($pending_approvals) && count($pending_approvals) > 0, "Aprovações pendentes encontradas: " . count($pending_approvals));
            $tests_total++;
            if ($success) $tests_passed++;

            if ($success) {
                $first_approval = $pending_approvals[0];
                // Aprovar primeira etapa
                $approve_result = $workflow_controller->approveDocumentStep(
                    $first_approval['document_id'], 
                    $first_approval['current_step_id'], 
                    'Aprovado na revisão inicial.', 
                    TEST_USER_ID
                );
                $success = test_result("Aprovar primeira etapa do workflow", $approve_result['success'], $approve_result['success'] ? "Etapa aprovada com sucesso" : $approve_result['message']);
                $tests_total++;
                if ($success) $tests_passed++;

                // Tentar rejeitar (para testar a funcionalidade)
                $reject_result = $workflow_controller->rejectDocumentStep(
                    $first_approval['document_id'], 
                    $first_approval['current_step_id'], 
                    'Rejeitado para correção.', 
                    TEST_USER_ID
                );
                // Este deve falhar se a aprovação anterior foi bem-sucedida, então o teste é que ele não deve ser bem-sucedido novamente
                $success = test_result("Rejeitar etapa do workflow (após aprovação)", !$reject_result['success'], $reject_result['success'] ? "Erro: Rejeição bem-sucedida após aprovação" : "Rejeição falhou como esperado (ou já aprovado)");
                $tests_total++;
                if ($success) $tests_passed++;
            }
        }

        // Testar atualização de workflow
        $update_wf_result = $workflow_controller->updateWorkflow([
            'id' => $workflow_id,
            'name' => $workflow_name . " (Atualizado)",
            'description' => 'Workflow atualizado para teste de integração',
            'updated_by' => TEST_USER_ID,
            'steps' => [
                ['id' => $result['steps'][0]['id'], 'name' => 'Revisão Inicial (Atualizada)', 'description' => 'Revisão inicial do documento atualizada', 'assigned_to_user_type' => 'admin', 'assigned_to_user_id' => null],
                ['name' => 'Aprovação Final', 'description' => 'Aprovação final do documento', 'assigned_to_user_type' => 'any', 'assigned_to_user_id' => null]
            ]
        ]);
        $success = test_result("Atualização de workflow", $update_wf_result['success'], $update_wf_result['success'] ? "Workflow atualizado com sucesso" : $update_wf_result['message']);
        $tests_total++;
        if ($success) $tests_passed++;

        // Testar exclusão de workflow
        $delete_wf_result = $workflow_controller->deleteWorkflow($workflow_id);
        $success = test_result("Exclusão de workflow", $delete_wf_result['success'], $delete_wf_result['success'] ? "Workflow excluído com sucesso" : $delete_wf_result['message']);
        $tests_total++;
        if ($success) $tests_passed++;
    }

} catch (Exception $e) {
    test_result("Teste de workflow", false, "Erro: " . $e->getMessage());
    $tests_total++;
}

// Testar logs de auditoria
echo "<h2>6. Teste de Logs de Auditoria</h2>";

try {
    $audit_log_controller = new AuditLogController();
    
    // Um log já deve ter sido criado pelos testes anteriores (login, criação de usuário, etc.)
    $logs_result = $audit_log_controller->listLogs();
    $success = test_result("Obtenção de logs de auditoria", $logs_result['success'] && count($logs_result['logs']) > 0, "Total de logs: " . count($logs_result['logs']));
    $tests_total++;
    if ($success) $tests_passed++;
    
} catch (Exception $e) {
    test_result("Teste de logs de auditoria", false, "Erro: " . $e->getMessage());
    $tests_total++;
}

// Testar backup e restauração
echo "<h2>7. Teste de Backup e Restauração</h2>";

try {
    $backup_controller = new BackupController();

    // Testar criação de backup
    $backup_result = $backup_controller->createBackup();
    $success = test_result("Criação de backup", $backup_result['success'], $backup_result['success'] ? "Backup criado: " . $backup_result['message'] : $backup_result['message']);
    $tests_total++;
    if ($success) $tests_passed++;

    if ($backup_result['success']) {
        $backups = $backup_controller->listBackups();
        $latest_backup = $backups[0]; // Pegar o backup mais recente

        // Simular restauração (não executaremos a restauração completa para não corromper o ambiente de teste)
        // Apenas verificamos se a função de restauração pode ser chamada sem erros fatais
        // Para um teste de integração completo, seria necessário um ambiente de teste isolado para a restauração
        
        // test_result("Restauração de backup (simulado)", true, "Funcionalidade de restauração não testada completamente em ambiente compartilhado.");

        // Testar exclusão de backup
        $delete_backup_result = $backup_controller->deleteBackup($latest_backup['name']);
        $success = test_result("Exclusão de backup", $delete_backup_result['success'], $delete_backup_result['success'] ? "Backup excluído com sucesso" : $delete_backup_result['message']);
        $tests_total++;
        if ($success) $tests_passed++;
    }

} catch (Exception $e) {
    test_result("Teste de backup e restauração", false, "Erro: " . $e->getMessage());
    $tests_total++;
}

// Resumo dos testes
echo "<div class='summary'>";
echo "<h2>Resumo dos Testes</h2>";
echo "<p>Total de testes executados: <strong>" . $tests_total . "</strong></p>";
echo "<p>Testes que passaram: <strong class='success'>" . $tests_passed . "</strong></p>";
echo "<p>Testes que falharam: <strong class='failure'>" . ($tests_total - $tests_passed) . "</strong></p>";
echo "</div>";

echo "</body>\n</html>";

// Limpar arquivos temporários criados pelos testes
if (file_exists('/tmp/dummy_document.txt')) {
    unlink('/tmp/dummy_document.txt');
}
if (file_exists('/tmp/dummy_document_v2.txt')) {
    unlink('/tmp/dummy_document_v2.txt');
}
if (file_exists('/tmp/dummy_document_workflow.txt')) {
    unlink('/tmp/dummy_document_workflow.txt');
}

?>

