<?php
/**
 * Página de Backup e Restauração
 * Sistema de Gerenciamento Eletrônico de Documentos (GED)
 */

// Iniciar sessão e verificar autenticação
session_start();
if (!isset($_SESSION["user_id"])) {
    header("Location: login.php");
    exit;
}

// Verificar se o usuário tem permissão de administrador
if ($_SESSION["user_type"] != "admin") {
    $_SESSION["flash_message"] = "Acesso negado. Você não tem permissão para acessar esta funcionalidade.";
    $_SESSION["flash_type"] = "danger";
    header("Location: index.php");
    exit;
}

// Incluir arquivos necessários
require_once "config/database.php";
require_once "config/config.php";
require_once "controllers/backup_controller.php";

// Conectar ao banco de dados
$database = new Database();
$db = $database->getConnection();

// Instanciar controlador
$backup_controller = new BackupController();

$message = "";
$message_type = "";

// Processar ações de backup e restauração
if ($_SERVER["REQUEST_METHOD"] == "POST") {
    if (isset($_POST["action"])) {
        if ($_POST["action"] == "create_backup") {
            $result = $backup_controller->createBackup();
            $message = $result["message"];
            $message_type = $result["success"] ? "success" : "danger";
        } elseif ($_POST["action"] == "restore_backup") {
            if (isset($_FILES["backup_file"]) && $_FILES["backup_file"]["error"] == UPLOAD_ERR_OK) {
                $result = $backup_controller->restoreBackup($_FILES["backup_file"]["tmp_name"]);
                $message = $result["message"];
                $message_type = $result["success"] ? "success" : "danger";
            } else {
                $message = "Por favor, selecione um arquivo de backup para restaurar.";
                $message_type = "danger";
            }
        }
    }
}

// Obter lista de backups existentes
$backups = $backup_controller->listBackups();

// Definir título da página
$page_title = "Backup e Restauração";

// Incluir cabeçalho
include "includes/header.php";
?>

<div class="row">
    <div class="col-12">
        <h1 class="mb-4">
            <i class="fas fa-database me-2"></i> <?php echo $page_title; ?>
        </h1>
    </div>
</div>

<?php if ($message): ?>
<div class="alert alert-<?php echo $message_type; ?> alert-dismissible fade show" role="alert">
    <?php echo $message; ?>
    <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
</div>
<?php endif; ?>

<div class="row">
    <div class="col-md-6 mb-4">
        <div class="card h-100">
            <div class="card-header">
                <h5 class="mb-0">Criar Novo Backup</h5>
            </div>
            <div class="card-body">
                <p>Crie um backup completo do banco de dados e dos arquivos de documentos.</p>
                <form method="POST" action="backup.php">
                    <input type="hidden" name="action" value="create_backup">
                    <button type="submit" class="btn btn-primary">
                        <i class="fas fa-plus-circle me-1"></i> Gerar Backup Agora
                    </button>
                </form>
            </div>
        </div>
    </div>

    <div class="col-md-6 mb-4">
        <div class="card h-100">
            <div class="card-header">
                <h5 class="mb-0">Restaurar Backup</h5>
            </div>
            <div class="card-body">
                <p>Selecione um arquivo de backup (.zip) para restaurar o sistema.</p>
                <form method="POST" action="backup.php" enctype="multipart/form-data">
                    <input type="hidden" name="action" value="restore_backup">
                    <div class="mb-3">
                        <label for="backup_file" class="form-label">Arquivo de Backup (.zip)</label>
                        <input class="form-control" type="file" id="backup_file" name="backup_file" accept=".zip">
                    </div>
                    <button type="submit" class="btn btn-warning">
                        <i class="fas fa-upload me-1"></i> Restaurar Backup
                    </button>
                </form>
            </div>
        </div>
    </div>
</div>

<div class="card mb-4">
    <div class="card-header">
        <h5 class="mb-0">Backups Existentes</h5>
    </div>
    <div class="card-body">
        <?php if (!empty($backups)): ?>
        <div class="table-responsive">
            <table class="table table-hover">
                <thead>
                    <tr>
                        <th>Nome do Arquivo</th>
                        <th>Tamanho</th>
                        <th>Data de Criação</th>
                        <th>Ações</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($backups as $backup): ?>
                    <tr>
                        <td><?php echo htmlspecialchars($backup["name"]); ?></td>
                        <td><?php echo round($backup["size"] / (1024 * 1024), 2); ?> MB</td>
                        <td><?php echo date("d/m/Y H:i:s", $backup["time"]); ?></td>
                        <td>
                            <a href="download_backup.php?file=<?php echo urlencode($backup["name"]); ?>" class="btn btn-sm btn-info" title="Download">
                                <i class="fas fa-download"></i>
                            </a>
                            <a href="delete_backup.php?file=<?php echo urlencode($backup["name"]); ?>" class="btn btn-sm btn-danger" title="Excluir" onclick="return confirm(\'Tem certeza que deseja excluir este backup?\');">
                                <i class="fas fa-trash"></i>
                            </a>
                        </td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
        <?php else: ?>
        <div class="alert alert-info">
            <i class="fas fa-info-circle me-2"></i> Nenhum backup encontrado.
        </div>
        <?php endif; ?>
    </div>
</div>

<?php
// Incluir rodapé
include "includes/footer.php";
?>

