<?php
/**
 * Controlador de Workflow
 * Sistema de Gerenciamento Eletrônico de Documentos (GED)
 */

// Incluir arquivos necessários
require_once __DIR__ . "/../config/database.php";
require_once __DIR__ . "/../models/workflow.php";
require_once __DIR__ . "/../models/workflow_step.php";
require_once __DIR__ . "/../models/workflow_approval.php";
require_once __DIR__ . "/../models/document.php";
require_once __DIR__ . "/../models/audit_log.php";

class WorkflowController {
    private $db;
    private $workflow;
    private $workflow_step;
    private $workflow_approval;
    private $document;
    
    public function __construct() {
        $database = new Database();
        $this->db = $database->getConnection();
        $this->workflow = new Workflow($this->db);
        $this->workflow_step = new WorkflowStep($this->db);
        $this->workflow_approval = new WorkflowApproval($this->db);
        $this->document = new Document($this->db);
    }
    
    public function listWorkflows() {
        $stmt = $this->workflow->readAll();
        $workflows = [];
        while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
            $workflows[] = $row;
        }
        return ["success" => true, "workflows" => $workflows];
    }
    
    public function getWorkflow($id) {
        if ($this->workflow->readOne($id)) {
            $this->workflow_step->workflow_id = $id;
            $stmt = $this->workflow_step->readByWorkflowId();
            $steps = [];
            while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
                $steps[] = $row;
            }
            return ["success" => true, "workflow" => [
                "id" => $this->workflow->id,
                "name" => $this->workflow->name,
                "description" => $this->workflow->description,
                "created_at" => $this->workflow->created_at,
                "created_by" => $this->workflow->created_by,
                "steps" => $steps
            ]];
        } else {
            return ["success" => false, "message" => "Workflow não encontrado."];
        }
    }
    
    public function createWorkflow($data, $user_id) {
        if (empty($data["name"])) {
            return ["success" => false, "message" => "O nome do workflow é obrigatório."];
        }

        $this->workflow->name = $data["name"];
        $this->workflow->description = $data["description"] ?? "";
        $this->workflow->created_by = $user_id;

        if ($this->workflow->create()) {
            logAudit($user_id, "create", "workflow", $this->workflow->id, "Workflow criado: " . $data["name"]);
            if (!empty($data["steps"]) && is_array($data["steps"])) {
                foreach ($data["steps"] as $step) {
                    $this->workflow_step->workflow_id = $this->workflow->id;
                    $this->workflow_step->step_order = $step["step_order"];
                    $this->workflow_step->name = $step["name"];
                    $this->workflow_step->description = $step["description"] ?? "";
                    $this->workflow_step->assigned_to_user_type = $step["assigned_to_user_type"] ?? "any";
                    $this->workflow_step->assigned_to_user_id = $step["assigned_to_user_id"] ?? null;
                    $this->workflow_step->create();
                }
            }
            return ["success" => true, "message" => "Workflow criado com sucesso.", "workflow_id" => $this->workflow->id];
        } else {
            return ["success" => false, "message" => "Erro ao criar workflow."];
        }
    }
    
    public function updateWorkflow($id, $data, $user_id) {
        if (!$this->workflow->readOne($id)) {
            return ["success" => false, "message" => "Workflow não encontrado."];
        }
        if (empty($data["name"])) {
            return ["success" => false, "message" => "O nome do workflow é obrigatório."];
        }

        $this->workflow->name = $data["name"];
        $this->workflow->description = $data["description"] ?? $this->workflow->description;

        if ($this->workflow->update()) {
            logAudit($user_id, "update", "workflow", $id, "Workflow atualizado: " . $data["name"]);
            // Atualizar etapas
            $this->workflow_step->workflow_id = $id;
            $this->workflow_step->deleteByWorkflowId(); // Excluir todas as etapas existentes
            if (!empty($data["steps"]) && is_array($data["steps"])) {
                foreach ($data["steps"] as $step) {
                    $this->workflow_step->workflow_id = $id;
                    $this->workflow_step->step_order = $step["step_order"];
                    $this->workflow_step->name = $step["name"];
                    $this->workflow_step->description = $step["description"] ?? "";
                    $this->workflow_step->assigned_to_user_type = $step["assigned_to_user_type"] ?? "any";
                    $this->workflow_step->assigned_to_user_id = $step["assigned_to_user_id"] ?? null;
                    $this->workflow_step->create();
                }
            }
            return ["success" => true, "message" => "Workflow atualizado com sucesso."];
        } else {
            return ["success" => false, "message" => "Erro ao atualizar workflow."];
        }
    }
    
    public function deleteWorkflow($id, $user_id) {
        if (!$this->workflow->readOne($id)) {
            return ["success" => false, "message" => "Workflow não encontrado."];
        }
        $workflow_name = $this->workflow->name;

        if ($this->workflow->delete()) {
            logAudit($user_id, "delete", "workflow", $id, "Workflow excluído: " . $workflow_name);
            return ["success" => true, "message" => "Workflow excluído com sucesso."];
        } else {
            return ["success" => false, "message" => "Não é possível excluir este workflow porque existem documentos associados a ele."];
        }
    }

    public function startApprovalProcess($document_id, $workflow_id, $user_id) {
        if (!$this->document->readOne($document_id)) {
            return ["success" => false, "message" => "Documento não encontrado."];
        }
        if (!$this->workflow->readOne($workflow_id)) {
            return ["success" => false, "message" => "Workflow não encontrado."];
        }

        $this->workflow_step->workflow_id = $workflow_id;
        $steps_stmt = $this->workflow_step->readByWorkflowId();
        $steps = [];
        while($row = $steps_stmt->fetch(PDO::FETCH_ASSOC)) {
            $steps[] = $row;
        }

        if (empty($steps)) {
            return ["success" => false, "message" => "O workflow selecionado não possui etapas definidas."];
        }

        // Iniciar a primeira etapa de aprovação
        $first_step = $steps[0];
        $this->workflow_approval->document_id = $document_id;
        $this->workflow_approval->workflow_step_id = $first_step["id"];
        $this->workflow_approval->approver_id = $user_id; // O usuário que inicia é o primeiro aprovador
        $this->workflow_approval->status = "pending";
        $this->workflow_approval->comments = "Processo de aprovação iniciado.";

        if ($this->workflow_approval->create()) {
            // Atualizar o status do documento para 'em revisão' e associar o workflow
            $this->document->id = $document_id;
            $this->document->updateWorkflowStatus($document_id, $workflow_id, $first_step["id"]);
            logAudit($user_id, "start_approval", "document", $document_id, "Processo de aprovação iniciado para o documento: " . $this->document->title);
            return ["success" => true, "message" => "Processo de aprovação iniciado com sucesso."];
        } else {
            return ["success" => false, "message" => "Erro ao iniciar o processo de aprovação."];
        }
    }

    public function approveDocumentStep($document_id, $workflow_step_id, $comments, $user_id) {
        $this->workflow_approval->document_id = $document_id;
        $this->workflow_approval->workflow_step_id = $workflow_step_id;

        if (!$this->workflow_approval->readByDocumentAndStep()) {
            return ["success" => false, "message" => "Etapa de aprovação não encontrada para este documento."];
        }

        if ($this->workflow_approval->status !== "pending") {
            return ["success" => false, "message" => "Esta etapa já foi aprovada ou rejeitada."];
        }

        // Verificar permissões do usuário para aprovar esta etapa
        $this->workflow_step->id = $workflow_step_id;
        $this->workflow_step->readOne();

        $can_approve = false;
        if ($this->workflow_step->assigned_to_user_type == "any") {
            $can_approve = true;
        } elseif ($this->workflow_step->assigned_to_user_type == "user" && $this->workflow_step->assigned_to_user_id == $user_id) {
            $can_approve = true;
        } elseif ($this->workflow_step->assigned_to_user_type == "admin" && $_SESSION["user_type"] == "admin") {
            $can_approve = true;
        }

        if (!$can_approve) {
            return ["success" => false, "message" => "Você não tem permissão para aprovar esta etapa."];
        }

        $this->workflow_approval->status = "approved";
        $this->workflow_approval->comments = $comments;
        $this->workflow_approval->approver_id = $user_id;

        if ($this->workflow_approval->update()) {
            logAudit($user_id, "approve_step", "document", $document_id, "Etapa de workflow aprovada para o documento: " . $this->document->title . " (Etapa: " . $this->workflow_step->name . ")");

            // Verificar se há próxima etapa
            $this->workflow_step->workflow_id = $this->workflow_step->workflow_id;
            $steps_stmt = $this->workflow_step->readByWorkflowId();
            $steps = [];
            while($row = $steps_stmt->fetch(PDO::FETCH_ASSOC)) {
                $steps[] = $row;
            }

            $current_step_index = -1;
            foreach ($steps as $index => $step) {
                if ($step["id"] == $workflow_step_id) {
                    $current_step_index = $index;
                    break;
                }
            }

            if ($current_step_index !== -1 && isset($steps[$current_step_index + 1])) {
                $next_step = $steps[$current_step_index + 1];
                // Criar nova entrada de aprovação para a próxima etapa
                $new_approval = new WorkflowApproval($this->db);
                $new_approval->document_id = $document_id;
                $new_approval->workflow_step_id = $next_step["id"];
                $new_approval->approver_id = null; // Aprovador será definido quando a etapa for iniciada
                $new_approval->status = "pending";
                $new_approval->comments = "";
                $new_approval->create();

                // Atualizar documento para a próxima etapa
                $this->document->id = $document_id;
                $this->document->updateWorkflowStatus($document_id, $this->workflow_step->workflow_id, $next_step["id"]);
            } else {
                // Todas as etapas aprovadas, finalizar workflow
                $this->document->id = $document_id;
                $this->document->updateStatus("approved");
                $this->document->updateWorkflowStatus($document_id, null, null); // Remover associação com workflow
                logAudit($user_id, "workflow_complete", "document", $document_id, "Workflow de aprovação concluído para o documento: " . $this->document->title);
            }
            return ["success" => true, "message" => "Etapa aprovada com sucesso."];
        } else {
            return ["success" => false, "message" => "Erro ao aprovar etapa."];
        }
    }

    public function rejectDocumentStep($document_id, $workflow_step_id, $comments, $user_id) {
        $this->workflow_approval->document_id = $document_id;
        $this->workflow_approval->workflow_step_id = $workflow_step_id;

        if (!$this->workflow_approval->readByDocumentAndStep()) {
            return ["success" => false, "message" => "Etapa de aprovação não encontrada para este documento."];
        }

        if ($this->workflow_approval->status !== "pending") {
            return ["success" => false, "message" => "Esta etapa já foi aprovada ou rejeitada."];
        }

        // Verificar permissões do usuário para rejeitar esta etapa
        $this->workflow_step->id = $workflow_step_id;
        $this->workflow_step->readOne();

        $can_reject = false;
        if ($this->workflow_step->assigned_to_user_type == "any") {
            $can_reject = true;
        } elseif ($this->workflow_step->assigned_to_user_type == "user" && $this->workflow_step->assigned_to_user_id == $user_id) {
            $can_reject = true;
        } elseif ($this->workflow_step->assigned_to_user_type == "admin" && $_SESSION["user_type"] == "admin") {
            $can_reject = true;
        }

        if (!$can_reject) {
            return ["success" => false, "message" => "Você não tem permissão para rejeitar esta etapa."];
        }

        $this->workflow_approval->status = "rejected";
        $this->workflow_approval->comments = $comments;
        $this->workflow_approval->approver_id = $user_id;

        if ($this->workflow_approval->update()) {
            // Atualizar status do documento para 'rejeitado'
            $this->document->id = $document_id;
            $this->document->updateStatus("rejected");
            $this->document->updateWorkflowStatus($document_id, null, null); // Remover associação com workflow
            logAudit($user_id, "reject_step", "document", $document_id, "Etapa de workflow rejeitada para o documento: " . $this->document->title . " (Etapa: " . $this->workflow_step->name . ")");
            return ["success" => true, "message" => "Etapa rejeitada com sucesso. Documento marcado como rejeitado."];
        } else {
            return ["success" => false, "message" => "Erro ao rejeitar etapa."];
        }
    }

    public function getPendingApprovalsForUser($user_id, $user_type) {
        return $this->workflow->getPendingApprovalsForUser($user_id, $user_type);
    }
}
?>

