<?php
/**
 * Modelo de Documento
 * Sistema de Gerenciamento Eletrônico de Documentos (GED)
 */

require_once __DIR__ . '/document_keywords.php';

class Document {
    // Conexão com o banco e nome da tabela
    private $conn;
    private $table_name = "document";
    private $document_keywords;

    // Propriedades do objeto
    public $id;
    public $title;
    public $description;
    public $file_path;
    public $file_name;
    public $file_type;
    public $file_size;
    public $category_id;
    public $category_name;
    public $status;
    public $created_at;
    public $updated_at;
    public $created_by;
    public $author_name;
    public $workflow_id;
    public $current_step_id;

    /**
     * Construtor com $db como conexão com o banco de dados
     * @param PDO $db Conexão com o banco de dados
     */
    public function __construct($db) {
        $this->conn = $db;
        $this->document_keywords = new DocumentKeywords($db);
    }

    /**
     * Cria um novo documento
     * @return bool Verdadeiro se criado com sucesso, falso caso contrário
     */
    public function create() {
        $query = "INSERT INTO " . $this->table_name . "
                  SET
                    title=:title,
                    description=:description,
                    file_path=:file_path,
                    file_name=:file_name,
                    file_type=:file_type,
                    file_size=:file_size,
                    category_id=:category_id,
                    status=:status,
                    created_by=:created_by";
        
        $stmt = $this->conn->prepare($query);
        
        // Sanitizar
        $this->title = htmlspecialchars(strip_tags($this->title));
        $this->description = htmlspecialchars(strip_tags($this->description));
        $this->file_path = htmlspecialchars(strip_tags($this->file_path));
        $this->file_name = htmlspecialchars(strip_tags($this->file_name));
        $this->file_type = htmlspecialchars(strip_tags($this->file_type));
        $this->file_size = htmlspecialchars(strip_tags($this->file_size));
        $this->category_id = htmlspecialchars(strip_tags($this->category_id));
        $this->status = htmlspecialchars(strip_tags($this->status));
        $this->created_by = htmlspecialchars(strip_tags($this->created_by));
        
        // Vincular valores
        $stmt->bindParam(":title", $this->title);
        $stmt->bindParam(":description", $this->description);
        $stmt->bindParam(":file_path", $this->file_path);
        $stmt->bindParam(":file_name", $this->file_name);
        $stmt->bindParam(":file_type", $this->file_type);
        $stmt->bindParam(":file_size", $this->file_size);
        $stmt->bindParam(":category_id", $this->category_id);
        $stmt->bindParam(":status", $this->status);
        $stmt->bindParam(":created_by", $this->created_by);
        
        // Executar
        if ($stmt->execute()) {
            $this->id = $this->conn->lastInsertId();
            return true;
        }
        
        return false;
    }

    /**
     * Lê um único documento pelo ID
     * @param int $id ID do documento
     * @return bool Verdadeiro se encontrado, falso caso contrário
     */
    public function readOne($id) {
        $query = "SELECT d.*, c.name as category_name, u.full_name as author_name
                  FROM " . $this->table_name . " d
                  LEFT JOIN category c ON d.category_id = c.id
                  LEFT JOIN user u ON d.created_by = u.id
                  WHERE d.id = ? LIMIT 0,1";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(1, $id);
        $stmt->execute();

        if ($stmt->rowCount() > 0) {
            $row = $stmt->fetch(PDO::FETCH_ASSOC);
            
            $this->id = $row["id"];
            $this->title = $row["title"];
            $this->description = $row["description"];
            $this->file_path = $row["file_path"];
            $this->file_name = $row["file_name"];
            $this->file_type = $row["file_type"];
            $this->file_size = $row["file_size"];
            $this->category_id = $row["category_id"];
            $this->category_name = $row["category_name"];
            $this->status = $row["status"];
            $this->created_at = $row["created_at"];
            $this->updated_at = $row["updated_at"];
            $this->created_by = $row["created_by"];
            $this->author_name = $row["author_name"];
            $this->workflow_id = $row["workflow_id"];
            $this->current_step_id = $row["current_step_id"];
            
            return true;
        }
        
        return false;
    }

    /**
     * Lê todos os documentos com filtros opcionais
     * @param array $filters Filtros opcionais (title, description, category_id, status, created_by, start_date, end_date)
     * @param int $page Número da página
     * @param int $per_page Itens por página
     * @return array Documentos encontrados
     */
    public function readAll($filters = [], $page = 1, $per_page = 10) {
        $query = "SELECT d.*, c.name as category_name, u.full_name as author_name
                  FROM " . $this->table_name . " d
                  LEFT JOIN category c ON d.category_id = c.id
                  LEFT JOIN user u ON d.created_by = u.id
                  WHERE 1=1";
        
        // Aplicar filtros
        if (!empty($filters["query"])) {
            $query .= " AND (d.title LIKE :query OR d.description LIKE :query OR d.file_name LIKE :query OR EXISTS (SELECT 1 FROM document_keywords dk WHERE dk.document_id = d.id AND dk.keyword LIKE :query))";
        }
        
        if (!empty($filters["category_id"])) {
            $query .= " AND d.category_id = :category_id";
        }
        
        if (!empty($filters["status"])) {
            $query .= " AND d.status = :status";
        }

        if (!empty($filters["created_by"])) {
            $query .= " AND d.created_by = :created_by";
        }
        
        if (!empty($filters["start_date"])) {
            $query .= " AND d.created_at >= :start_date";
        }
        
        if (!empty($filters["end_date"])) {
            $query .= " AND d.created_at <= :end_date";
        }
        
        // Ordenação e paginação
        $query .= " ORDER BY d.created_at DESC";
        
        // Calcular offset para paginação
        $offset = ($page - 1) * $per_page;
        $query .= " LIMIT :offset, :per_page";
        
        $stmt = $this->conn->prepare($query);
        
        // Vincular parâmetros de filtro
        if (!empty($filters["query"])) {
            $query_param = "%" . $filters["query"] . "%";
            $stmt->bindParam(":query", $query_param);
        }
        
        if (!empty($filters["category_id"])) {
            $stmt->bindParam(":category_id", $filters["category_id"]);
        }
        
        if (!empty($filters["status"])) {
            $stmt->bindParam(":status", $filters["status"]);
        }

        if (!empty($filters["created_by"])) {
            $stmt->bindParam(":created_by", $filters["created_by"]);
        }
        
        if (!empty($filters["start_date"])) {
            $stmt->bindParam(":start_date", $filters["start_date"]);
        }
        
        if (!empty($filters["end_date"])) {
            $stmt->bindParam(":end_date", $filters["end_date"]);
        }
        
        // Vincular parâmetros de paginação
        $stmt->bindParam(":offset", $offset, PDO::PARAM_INT);
        $stmt->bindParam(":per_page", $per_page, PDO::PARAM_INT);
        
        $stmt->execute();
        
        $documents = [];
        while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
            $documents[] = $row;
        }
        
        return $documents;
    }

    /**
     * Conta o total de documentos com filtros opcionais
     * @param array $filters Filtros opcionais
     * @return int Total de documentos
     */
    public function countAll($filters = []) {
        $query = "SELECT COUNT(*) as total FROM " . $this->table_name . " d WHERE 1=1";
        
        // Aplicar filtros
        if (!empty($filters["query"])) {
            $query .= " AND (d.title LIKE :query OR d.description LIKE :query OR d.file_name LIKE :query OR EXISTS (SELECT 1 FROM document_keywords dk WHERE dk.document_id = d.id AND dk.keyword LIKE :query))";
        }
        
        if (!empty($filters["category_id"])) {
            $query .= " AND d.category_id = :category_id";
        }
        
        if (!empty($filters["status"])) {
            $query .= " AND d.status = :status";
        }

        if (!empty($filters["created_by"])) {
            $query .= " AND d.created_by = :created_by";
        }
        
        if (!empty($filters["start_date"])) {
            $query .= " AND d.created_at >= :start_date";
        }
        
        if (!empty($filters["end_date"])) {
            $query .= " AND d.created_at <= :end_date";
        }
        
        $stmt = $this->conn->prepare($query);
        
        // Vincular parâmetros de filtro
        if (!empty($filters["query"])) {
            $query_param = "%" . $filters["query"] . "%";
            $stmt->bindParam(":query", $query_param);
        }
        
        if (!empty($filters["category_id"])) {
            $stmt->bindParam(":category_id", $filters["category_id"]);
        }
        
        if (!empty($filters["status"])) {
            $stmt->bindParam(":status", $filters["status"]);
        }

        if (!empty($filters["created_by"])) {
            $stmt->bindParam(":created_by", $filters["created_by"]);
        }
        
        if (!empty($filters["start_date"])) {
            $stmt->bindParam(":start_date", $filters["start_date"]);
        }
        
        if (!empty($filters["end_date"])) {
            $stmt->bindParam(":end_date", $filters["end_date"]);
        }
        
        $stmt->execute();
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        
        return (int)$row["total"];
    }

    /**
     * Atualiza um documento existente
     * @return bool Verdadeiro se atualizado com sucesso, falso caso contrário
     */
    public function update() {
        $query = "UPDATE " . $this->table_name . "
                  SET
                    title=:title,
                    description=:description,
                    category_id=:category_id,
                    status=:status,
                    updated_at=NOW()
                  WHERE id = :id";
        
        $stmt = $this->conn->prepare($query);
        
        // Sanitizar
        $this->title = htmlspecialchars(strip_tags($this->title));
        $this->description = htmlspecialchars(strip_tags($this->description));
        $this->category_id = htmlspecialchars(strip_tags($this->category_id));
        $this->status = htmlspecialchars(strip_tags($this->status));
        $this->id = htmlspecialchars(strip_tags($this->id));
        
        // Vincular valores
        $stmt->bindParam(":title", $this->title);
        $stmt->bindParam(":description", $this->description);
        $stmt->bindParam(":category_id", $this->category_id);
        $stmt->bindParam(":status", $this->status);
        $stmt->bindParam(":id", $this->id);
        
        // Executar
        if ($stmt->execute()) {
            return true;
        }
        
        return false;
    }

    /**
     * Atualiza apenas os detalhes do arquivo de um documento (usado na restauração/nova versão)
     * @return bool Verdadeiro se atualizado com sucesso, falso caso contrário
     */
    public function updateFileDetails() {
        $query = "UPDATE " . $this->table_name . "
                  SET
                    file_path=:file_path,
                    file_name=:file_name,
                    file_type=:file_type,
                    file_size=:file_size,
                    updated_at=NOW()
                  WHERE id = :id";
        
        $stmt = $this->conn->prepare($query);
        
        // Sanitizar
        $this->file_path = htmlspecialchars(strip_tags($this->file_path));
        $this->file_name = htmlspecialchars(strip_tags($this->file_name));
        $this->file_type = htmlspecialchars(strip_tags($this->file_type));
        $this->file_size = htmlspecialchars(strip_tags($this->file_size));
        $this->id = htmlspecialchars(strip_tags($this->id));
        
        // Vincular valores
        $stmt->bindParam(":file_path", $this->file_path);
        $stmt->bindParam(":file_name", $this->file_name);
        $stmt->bindParam(":file_type", $this->file_type);
        $stmt->bindParam(":file_size", $this->file_size);
        $stmt->bindParam(":id", $this->id);
        
        // Executar
        if ($stmt->execute()) {
            return true;
        }
        
        return false;
    }

    /**
     * Atualiza o status de um documento
     * @return bool Verdadeiro se atualizado com sucesso, falso caso contrário
     */
    public function updateStatus() {
        $query = "UPDATE " . $this->table_name . "
                  SET status = :status, updated_at = NOW()
                  WHERE id = :id";
        
        $stmt = $this->conn->prepare($query);
        
        // Sanitizar
        $this->status = htmlspecialchars(strip_tags($this->status));
        $this->id = htmlspecialchars(strip_tags($this->id));
        
        // Vincular valores
        $stmt->bindParam(":status", $this->status);
        $stmt->bindParam(":id", $this->id);
        
        // Executar
        if ($stmt->execute()) {
            return true;
        }
        
        return false;
    }

    /**
     * Exclui um documento
     * @return bool Verdadeiro se excluído com sucesso, falso caso contrário
     */
    public function delete() {
        $query = "DELETE FROM " . $this->table_name . " WHERE id = ?";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(1, $this->id);
        
        if ($stmt->execute()) {
            // Excluir palavras-chave associadas
            $this->document_keywords->document_id = $this->id;
            $this->document_keywords->deleteByDocumentId();
            return true;
        }
        
        return false;
    }

    /**
     * Obtém as palavras-chave de um documento
     * @param int $document_id ID do documento
     * @return array Lista de palavras-chave
     */
    public function getKeywords($document_id) {
        $this->document_keywords->document_id = $document_id;
        $stmt = $this->document_keywords->readByDocumentId();
        
        $keywords = [];
        while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
            $keywords[] = $row["keyword"];
        }
        
        return $keywords;
    }

    /**
     * Define as palavras-chave de um documento
     * @param int $document_id ID do documento
     * @param array $keywords Lista de palavras-chave
     * @return bool Verdadeiro se definido com sucesso, falso caso contrário
     */
    public function setKeywords($document_id, $keywords) {
        // Primeiro, remover todas as palavras-chave existentes para este documento
        $this->document_keywords->document_id = $document_id;
        $this->document_keywords->deleteByDocumentId();
        
        // Depois, inserir as novas palavras-chave
        foreach ($keywords as $keyword) {
            if (!empty(trim($keyword))) {
                $this->document_keywords->document_id = $document_id;
                $this->document_keywords->keyword = trim($keyword);
                $this->document_keywords->create();
            }
        }
        
        return true;
    }

    /**
     * Conta o total de documentos.
     * @return int Total de documentos.
     */
    public function countDocuments() {
        $query = "SELECT COUNT(*) as total FROM " . $this->table_name;
        $stmt = $this->conn->prepare($query);
        $stmt->execute();
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        return (int)$row["total"];
    }

    /**
     * Conta o total de documentos por status.
     * @param string $status Status do documento.
     * @return int Total de documentos com o status especificado.
     */
    public function countDocumentsByStatus($status) {
        $query = "SELECT COUNT(*) as total FROM " . $this->table_name . " WHERE status = ?";
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(1, $status);
        $stmt->execute();
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        return (int)$row["total"];
    }

    /**
     * Obtém os documentos mais recentes.
     * @param int $limit Limite de documentos a serem retornados.
     * @return array Documentos mais recentes.
     */
    public function getRecentDocuments($limit) {
        $query = "SELECT d.*, c.name as category_name, u.full_name as author_name FROM " . $this->table_name . " d LEFT JOIN category c ON d.category_id = c.id LEFT JOIN user u ON d.created_by = u.id ORDER BY created_at DESC LIMIT 0, ?";
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(1, $limit, PDO::PARAM_INT);
        $stmt->execute();
        
        $documents = [];
        while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
            $documents[] = $row;
        }
        
        return $documents;
    }

    /**
     * Atualiza o workflow e a etapa atual de um documento.
     * @param int $document_id ID do documento.
     * @param int $workflow_id ID do workflow.
     * @param int $current_step_id ID da etapa atual do workflow.
     * @return bool Verdadeiro se atualizado com sucesso, falso caso contrário.
     */
    public function updateWorkflowStatus($document_id, $workflow_id, $current_step_id) {
        $query = "UPDATE " . $this->table_name . "
                  SET workflow_id = :workflow_id, current_step_id = :current_step_id, updated_at = NOW()
                  WHERE id = :id";
        
        $stmt = $this->conn->prepare($query);
        
        $stmt->bindParam(":workflow_id", $workflow_id);
        $stmt->bindParam(":current_step_id", $current_step_id);
        $stmt->bindParam(":id", $document_id);
        
        if ($stmt->execute()) {
            return true;
        }
        return false;
    }
}
?>

