<?php
/**
 * Página de gerenciamento de workflow
 * Sistema de Gerenciamento Eletrônico de Documentos (GED)
 */

// Iniciar sessão e verificar autenticação
session_start();
if (!isset($_SESSION["user_id"])) {
    header("Location: login.php");
    exit;
}

// Verificar se o usuário tem permissão de administrador
if ($_SESSION["user_type"] != "admin") {
    $_SESSION["flash_message"] = "Acesso negado. Você não tem permissão para acessar esta página.";
    $_SESSION["flash_type"] = "danger";
    header("Location: index.php");
    exit;
}

// Incluir arquivos necessários
require_once "config/database.php";
require_once "models/workflow.php";
require_once "models/user.php"; // Adicionado para obter usuários para atribuição de etapas
require_once "controllers/workflow_controller.php";

// Conectar ao banco de dados
$database = new Database();
$db = $database->getConnection();

// Instanciar controlador
$workflow_controller = new WorkflowController();
$user_model = new User($db); // Instanciar modelo de usuário

// Obter todos os usuários para o dropdown de atribuição de etapas
$all_users = $user_model->readAll();
$users_list = [];
while ($row = $all_users->fetch(PDO::FETCH_ASSOC)) {
    $users_list[] = $row;
}

// Processar ações
if ($_SERVER["REQUEST_METHOD"] == "POST") {
    if (isset($_POST["action"])) {
        // Adicionar workflow
        if ($_POST["action"] == "add") {
            $name = $_POST["name"] ?? "";
            $description = $_POST["description"] ?? "";
            $steps = json_decode($_POST["steps_json"] ?? "[]", true); // Decodificar JSON das etapas
            
            if (!empty($name) && !empty($steps)) {
                $result = $workflow_controller->createWorkflow([
                    "name" => $name,
                    "description" => $description,
                    "steps" => $steps,
                    "created_by" => $_SESSION["user_id"]
                ]);
                
                if ($result["success"]) {
                    $_SESSION["flash_message"] = "Workflow adicionado com sucesso.";
                    $_SESSION["flash_type"] = "success";
                } else {
                    $_SESSION["flash_message"] = $result["message"];
                    $_SESSION["flash_type"] = "danger";
                }
            } else {
                $_SESSION["flash_message"] = "Nome e etapas do workflow são obrigatórios.";
                $_SESSION["flash_type"] = "danger";
            }
        }
        
        // Editar workflow
        else if ($_POST["action"] == "edit") {
            $id = $_POST["id"] ?? "";
            $name = $_POST["name"] ?? "";
            $description = $_POST["description"] ?? "";
            $steps = json_decode($_POST["steps_json"] ?? "[]", true); // Decodificar JSON das etapas
            
            if (!empty($id) && !empty($name) && !empty($steps)) {
                $result = $workflow_controller->updateWorkflow([
                    "id" => $id,
                    "name" => $name,
                    "description" => $description,
                    "steps" => $steps,
                    "updated_by" => $_SESSION["user_id"]
                ]);
                
                if ($result["success"]) {
                    $_SESSION["flash_message"] = "Workflow atualizado com sucesso.";
                    $_SESSION["flash_type"] = "success";
                } else {
                    $_SESSION["flash_message"] = $result["message"];
                    $_SESSION["flash_type"] = "danger";
                }
            } else {
                $_SESSION["flash_message"] = "ID, nome e etapas do workflow são obrigatórios.";
                $_SESSION["flash_type"] = "danger";
            }
        }
        
        // Excluir workflow
        else if ($_POST["action"] == "delete") {
            $id = $_POST["id"] ?? "";
            
            if (!empty($id)) {
                $result = $workflow_controller->deleteWorkflow($id);
                
                if ($result["success"]) {
                    $_SESSION["flash_message"] = "Workflow excluído com sucesso.";
                    $_SESSION["flash_type"] = "success";
                } else {
                    $_SESSION["flash_message"] = $result["message"];
                    $_SESSION["flash_type"] = "danger";
                }
            } else {
                $_SESSION["flash_message"] = "ID do workflow é obrigatório.";
                $_SESSION["flash_type"] = "danger";
            }
        }
        
        // Redirecionar para evitar reenvio do formulário
        header("Location: workflows.php");
        exit;
    }
}

// Obter todos os workflows
$workflows_data = $workflow_controller->getAllWorkflows();

// Definir título da página
$page_title = "Gerenciar Workflows";

// Incluir cabeçalho
include "includes/header.php";
?>

<div class="row mb-4">
    <div class="col-md-6">
        <h1><i class="fas fa-project-diagram me-2"></i> Gerenciar Workflows</h1>
    </div>
    <div class="col-md-6 text-md-end">
        <button type="button" class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#addWorkflowModal">
            <i class="fas fa-plus me-1"></i> Novo Workflow
        </button>
    </div>
</div>

<!-- Lista de workflows -->
<div class="card">
    <div class="card-body">
        <?php if (count($workflows_data) > 0): ?>
        <div class="table-responsive">
            <table class="table table-hover datatable">
                <thead>
                    <tr>
                        <th>ID</th>
                        <th>Nome</th>
                        <th>Descrição</th>
                        <th>Etapas</th>
                        <th>Documentos</th>
                        <th>Criado por</th>
                        <th>Data de Criação</th>
                        <th>Ações</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($workflows_data as $wf): ?>
                    <tr>
                        <td><?php echo $wf["id"]; ?></td>
                        <td><?php echo htmlspecialchars($wf["name"]); ?></td>
                        <td><?php echo htmlspecialchars($wf["description"] ?? ""); ?></td>
                        <td>
                            <span class="badge bg-info"><?php echo count($wf["steps"]); ?> etapas</span>
                        </td>
                        <td>
                            <a href="documents.php?workflow_id=<?php echo $wf["id"]; ?>" class="badge bg-primary text-decoration-none">
                                <?php echo $wf["document_count"]; ?> documentos
                            </a>
                        </td>
                        <td><?php echo htmlspecialchars($wf["created_by_username"]); ?></td>
                        <td><?php echo date("d/m/Y", strtotime($wf["created_at"])); ?></td>
                        <td>
                            <div class="btn-group btn-group-sm">
                                <button type="button" class="btn btn-info view-workflow" 
                                        data-id="<?php echo $wf["id"]; ?>"
                                        data-name="<?php echo htmlspecialchars($wf["name"]); ?>"
                                        data-description="<?php echo htmlspecialchars($wf["description"] ?? ""); ?>"
                                        data-steps=\'<?php echo json_encode($wf["steps"]); ?>\'>
                                    <i class="fas fa-eye"></i>
                                </button>
                                <button type="button" class="btn btn-primary edit-workflow" 
                                        data-id="<?php echo $wf["id"]; ?>"
                                        data-name="<?php echo htmlspecialchars($wf["name"]); ?>"
                                        data-description="<?php echo htmlspecialchars($wf["description"] ?? ""); ?>"
                                        data-steps=\'<?php echo json_encode($wf["steps"]); ?>\'>
                                    <i class="fas fa-edit"></i>
                                </button>
                                <button type="button" class="btn btn-danger delete-workflow" 
                                        data-id="<?php echo $wf["id"]; ?>"
                                        data-name="<?php echo htmlspecialchars($wf["name"]); ?>"
                                        data-count="<?php echo $wf["document_count"]; ?>">
                                    <i class="fas fa-trash"></i>
                                </button>
                            </div>
                        </td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
        <?php else: ?>
        <div class="alert alert-info">
            <i class="fas fa-info-circle me-2"></i> Nenhum workflow encontrado.
        </div>
        <?php endif; ?>
    </div>
</div>

<!-- Modal para adicionar workflow -->
<div class="modal fade" id="addWorkflowModal" tabindex="-1" aria-labelledby="addWorkflowModalLabel" aria-hidden="true">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <form method="POST" action="workflows.php" id="addWorkflowForm">
                <input type="hidden" name="action" value="add">
                <input type="hidden" name="steps_json" id="add-steps-json">
                
                <div class="modal-header">
                    <h5 class="modal-title" id="addWorkflowModalLabel">Novo Workflow</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Fechar"></button>
                </div>
                
                <div class="modal-body">
                    <div class="mb-3">
                        <label for="add-name" class="form-label">Nome <span class="text-danger">*</span></label>
                        <input type="text" class="form-control" id="add-name" name="name" required>
                    </div>
                    
                    <div class="mb-3">
                        <label for="add-description" class="form-label">Descrição</label>
                        <textarea class="form-control" id="add-description" name="description" rows="2"></textarea>
                    </div>
                    
                    <div class="mb-3">
                        <label class="form-label">Etapas do Workflow <span class="text-danger">*</span></label>
                        <p class="text-muted small">Defina as etapas que os documentos devem passar neste workflow.</p>
                        
                        <div id="add-steps-container">
                            <!-- As etapas serão adicionadas aqui dinamicamente -->
                        </div>
                        
                        <button type="button" class="btn btn-outline-secondary mt-2" id="add-step-btn">
                            <i class="fas fa-plus me-1"></i> Adicionar Etapa
                        </button>
                    </div>
                </div>
                
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancelar</button>
                    <button type="submit" class="btn btn-primary">Adicionar</button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Modal para editar workflow -->
<div class="modal fade" id="editWorkflowModal" tabindex="-1" aria-labelledby="editWorkflowModalLabel" aria-hidden="true">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <form method="POST" action="workflows.php" id="editWorkflowForm">
                <input type="hidden" name="action" value="edit">
                <input type="hidden" name="id" id="edit-id">
                <input type="hidden" name="steps_json" id="edit-steps-json">
                
                <div class="modal-header">
                    <h5 class="modal-title" id="editWorkflowModalLabel">Editar Workflow</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Fechar"></button>
                </div>
                
                <div class="modal-body">
                    <div class="mb-3">
                        <label for="edit-name" class="form-label">Nome <span class="text-danger">*</span></label>
                        <input type="text" class="form-control" id="edit-name" name="name" required>
                    </div>
                    
                    <div class="mb-3">
                        <label for="edit-description" class="form-label">Descrição</label>
                        <textarea class="form-control" id="edit-description" name="description" rows="2"></textarea>
                    </div>
                    
                    <div class="mb-3">
                        <label class="form-label">Etapas do Workflow <span class="text-danger">*</span></label>
                        <p class="text-muted small">Defina as etapas que os documentos devem passar neste workflow.</p>
                        
                        <div id="edit-steps-container">
                            <!-- As etapas serão adicionadas aqui dinamicamente -->
                        </div>
                        
                        <button type="button" class="btn btn-outline-secondary mt-2" id="edit-step-btn">
                            <i class="fas fa-plus me-1"></i> Adicionar Etapa
                        </button>
                    </div>
                </div>
                
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancelar</button>
                    <button type="submit" class="btn btn-primary">Salvar</button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Modal para visualizar workflow -->
<div class="modal fade" id="viewWorkflowModal" tabindex="-1" aria-labelledby="viewWorkflowModalLabel" aria-hidden="true">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="viewWorkflowModalLabel">Visualizar Workflow</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Fechar"></button>
            </div>
            
            <div class="modal-body">
                <h4 id="view-name" class="mb-3"></h4>
                
                <div class="workflow-diagram mb-4">
                    <div id="view-steps-diagram" class="d-flex justify-content-center">
                        <!-- O diagrama será gerado aqui -->
                    </div>
                </div>
                
                <div class="card">
                    <div class="card-header">
                        <h5 class="mb-0">Detalhes das Etapas</h5>
                    </div>
                    <div class="card-body p-0">
                        <div class="list-group list-group-flush" id="view-steps-list">
                            <!-- A lista de etapas será gerada aqui -->
                        </div>
                    </div>
                </div>
            </div>
            
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Fechar</button>
            </div>
        </div>
    </div>
</div>

<!-- Modal para excluir workflow -->
<div class="modal fade" id="deleteWorkflowModal" tabindex="-1" aria-labelledby="deleteWorkflowModalLabel" aria-hidden="true">
    <div class="modal-dialog">
        <div class="modal-content">
            <form method="POST" action="workflows.php">
                <input type="hidden" name="action" value="delete">
                <input type="hidden" name="id" id="delete-id">
                
                <div class="modal-header">
                    <h5 class="modal-title" id="deleteWorkflowModalLabel">Excluir Workflow</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Fechar"></button>
                </div>
                
                <div class="modal-body">
                    <p>Tem certeza que deseja excluir o workflow <strong id="delete-name"></strong>?</p>
                    <div id="delete-warning" class="alert alert-warning d-none">
                        <i class="fas fa-exclamation-triangle me-2"></i> Este workflow está sendo usado por <strong id="delete-count"></strong> documentos. Ao excluí-lo, os documentos ficarão sem workflow.
                    </div>
                </div>
                
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancelar</button>
                    <button type="submit" class="btn btn-danger">Excluir</button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- JavaScript para manipulação dos modais -->
<script>
document.addEventListener("DOMContentLoaded", function() {
    const usersList = <?php echo json_encode($users_list); ?>;

    function generateStepHtml(step = {}, index, isEdit = false) {
        const prefix = isEdit ? `edit-step-${index}` : `add-step-${index}`;
        const assignedToUserType = step.assigned_to_user_type || 'any';
        const assignedToUserId = step.assigned_to_user_id || '';

        let userOptionsHtml = 
            `<option value="" ${assignedToUserId === '' ? 'selected' : ''}>Nenhum</option>`;
        usersList.forEach(user => {
            userOptionsHtml += 
                `<option value="${user.id}" ${assignedToUserId == user.id ? 'selected' : ''}>${user.full_name} (${user.username})</option>`;
        });

        return `
            <div class="card mb-2 step-item" data-index="${index}">
                <div class="card-body">
                    <div class="d-flex justify-content-between align-items-center mb-2">
                        <h6 class="mb-0">Etapa ${index + 1}</h6>
                        <button type="button" class="btn btn-sm btn-outline-danger remove-step-btn">
                            <i class="fas fa-times"></i>
                        </button>
                    </div>
                    <div class="mb-3">
                        <label for="${prefix}-name" class="form-label">Nome da Etapa <span class="text-danger">*</span></label>
                        <input type="text" class="form-control" id="${prefix}-name" name="steps[${index}][name]" value="${step.name || ''}" required>
                    </div>
                    <div class="mb-3">
                        <label for="${prefix}-description" class="form-label">Descrição</label>
                        <textarea class="form-control" id="${prefix}-description" name="steps[${index}][description]" rows="1">${step.description || ''}</textarea>
                    </div>
                    <div class="mb-3">
                        <label for="${prefix}-assigned-type" class="form-label">Atribuído a</label>
                        <select class="form-select assigned-to-type" id="${prefix}-assigned-type" name="steps[${index}][assigned_to_user_type]">
                            <option value="any" ${assignedToUserType === 'any' ? 'selected' : ''}>Qualquer Usuário</option>
                            <option value="admin" ${assignedToUserType === 'admin' ? 'selected' : ''}>Administrador</option>
                            <option value="user" ${assignedToUserType === 'user' ? 'selected' : ''}>Usuário Específico</option>
                        </select>
                    </div>
                    <div class="mb-3 assigned-to-user-container" style="display: ${assignedToUserType === 'user' ? 'block' : 'none'}">
                        <label for="${prefix}-assigned-user" class="form-label">Selecionar Usuário</label>
                        <select class="form-select" id="${prefix}-assigned-user" name="steps[${index}][assigned_to_user_id]">
                            ${userOptionsHtml}
                        </select>
                    </div>
                </div>
            </div>
        `;
    }

    function addStep(containerId, isEdit = false, stepData = {}) {
        const container = document.getElementById(containerId);
        const index = container.children.length;
        container.insertAdjacentHTML('beforeend', generateStepHtml(stepData, index, isEdit));
        updateStepIndexes(containerId);
    }

    function updateStepIndexes(containerId) {
        const container = document.getElementById(containerId);
        Array.from(container.children).forEach((stepItem, index) => {
            stepItem.setAttribute('data-index', index);
            stepItem.querySelector('h6').textContent = `Etapa ${index + 1}`;
            stepItem.querySelectorAll('[name^="steps["]').forEach(input => {
                const oldName = input.name;
                const newName = oldName.replace(/steps\[\d+\]/, `steps[${index}]`);
                input.name = newName;
                input.id = newName.replace(/[\[\]]/g, '-').replace(/--/g, '-').slice(0, -1);
            });
            stepItem.querySelectorAll('label').forEach(label => {
                const oldFor = label.htmlFor;
                const newFor = oldFor.replace(/step-\d+-/, `step-${index}-`);
                label.htmlFor = newFor;
            });
        });
        updateStepsJson(containerId);
    }

    function updateStepsJson(containerId) {
        const container = document.getElementById(containerId);
        const steps = [];
        Array.from(container.children).forEach(stepItem => {
            const step = {};
            step.name = stepItem.querySelector('[name$="[name]"]').value;
            step.description = stepItem.querySelector('[name$="[description]"]').value;
            step.assigned_to_user_type = stepItem.querySelector('[name$="[assigned_to_user_type]"]').value;
            step.assigned_to_user_id = stepItem.querySelector('[name$="[assigned_to_user_id]"]').value;
            steps.push(step);
        });
        if (containerId === 'add-steps-container') {
            document.getElementById('add-steps-json').value = JSON.stringify(steps);
        } else if (containerId === 'edit-steps-container') {
            document.getElementById('edit-steps-json').value = JSON.stringify(steps);
        }
    }

    // Event listeners para adicionar e remover etapas
    document.getElementById('add-step-btn').addEventListener('click', () => addStep('add-steps-container'));
    document.getElementById('edit-step-btn').addEventListener('click', () => addStep('edit-steps-container', true));

    document.addEventListener('click', function(event) {
        if (event.target.classList.contains('remove-step-btn') || event.target.closest('.remove-step-btn')) {
            const button = event.target.closest('.remove-step-btn');
            const stepItem = button.closest('.step-item');
            const container = stepItem.parentNode;
            stepItem.remove();
            updateStepIndexes(container.id);
        }
    });

    // Event listeners para mostrar/esconder seleção de usuário
    document.addEventListener('change', function(event) {
        if (event.target.classList.contains('assigned-to-type')) {
            const select = event.target;
            const userContainer = select.closest('.step-item').querySelector('.assigned-to-user-container');
            if (select.value === 'user') {
                userContainer.style.display = 'block';
            } else {
                userContainer.style.display = 'none';
                userContainer.querySelector('select').value = ''; // Limpar seleção
            }
            updateStepsJson(select.closest('.modal-body').querySelector('.step-item').parentNode.id);
        }
    });

    // Manipular abertura do modal de adicionar workflow
    document.getElementById('addWorkflowModal').addEventListener('show.bs.modal', function () {
        document.getElementById('add-steps-container').innerHTML = ''; // Limpar etapas anteriores
        addStep('add-steps-container'); // Adicionar uma etapa inicial
    });

    // Manipular abertura do modal de editar workflow
    document.querySelectorAll('.edit-workflow').forEach(button => {
        button.addEventListener('click', function() {
            const id = this.getAttribute('data-id');
            const name = this.getAttribute('data-name');
            const description = this.getAttribute('data-description');
            const steps = JSON.parse(this.getAttribute('data-steps'));

            document.getElementById('edit-id').value = id;
            document.getElementById('edit-name').value = name;
            document.getElementById('edit-description').value = description;

            const editStepsContainer = document.getElementById('edit-steps-container');
            editStepsContainer.innerHTML = ''; // Limpar etapas anteriores
            steps.forEach(step => addStep('edit-steps-container', true, step));
            updateStepsJson('edit-steps-container');

            const editModal = new bootstrap.Modal(document.getElementById('editWorkflowModal'));
            editModal.show();
        });
    });

    // Manipular abertura do modal de visualizar workflow
    document.querySelectorAll('.view-workflow').forEach(button => {
        button.addEventListener('click', function() {
            const name = this.getAttribute('data-name');
            const description = this.getAttribute('data-description');
            const steps = JSON.parse(this.getAttribute('data-steps'));

            document.getElementById('view-name').textContent = name;
            
            const viewStepsDiagram = document.getElementById('view-steps-diagram');
            viewStepsDiagram.innerHTML = '';
            const viewStepsList = document.getElementById('view-steps-list');
            viewStepsList.innerHTML = '';

            steps.forEach((step, index) => {
                // Diagrama
                const stepDiv = document.createElement('div');
                stepDiv.classList.add('workflow-step-box', 'p-3', 'border', 'rounded', 'text-center', 'mx-2');
                stepDiv.innerHTML = `
                    <h6>${step.name}</h6>
                    <small>${step.description || ''}</small><br>
                    <small>Atribuído a: ${step.assigned_to_user_type === 'user' && step.assigned_to_user_id ? usersList.find(u => u.id == step.assigned_to_user_id)?.full_name || 'Usuário Desconhecido' : (step.assigned_to_user_type === 'admin' ? 'Administrador' : 'Qualquer Usuário')}</small>
                `;
                viewStepsDiagram.appendChild(stepDiv);

                if (index < steps.length - 1) {
                    const arrowDiv = document.createElement('div');
                    arrowDiv.classList.add('workflow-arrow', 'd-flex', 'align-items-center', 'justify-content-center');
                    arrowDiv.innerHTML = `<i class="fas fa-arrow-right fa-2x text-muted"></i>`;
                    viewStepsDiagram.appendChild(arrowDiv);
                }

                // Lista
                const listItem = document.createElement('div');
                listItem.classList.add('list-group-item');
                listItem.innerHTML = `
                    <div class="d-flex w-100 justify-content-between">
                        <h5 class="mb-1">${index + 1}. ${step.name}</h5>
                    </div>
                    <p class="mb-1">${step.description || 'Sem descrição.'}</p>
                    <small>Atribuído a: ${step.assigned_to_user_type === 'user' && step.assigned_to_user_id ? usersList.find(u => u.id == step.assigned_to_user_id)?.full_name || 'Usuário Desconhecido' : (step.assigned_to_user_type === 'admin' ? 'Administrador' : 'Qualquer Usuário')}</small>
                `;
                viewStepsList.appendChild(listItem);
            });

            const viewModal = new bootstrap.Modal(document.getElementById('viewWorkflowModal'));
            viewModal.show();
        });
    });

    // Manipular clique no botão de excluir
    document.querySelectorAll('.delete-workflow').forEach(button => {
        button.addEventListener('click', function() {
            const id = this.getAttribute('data-id');
            const name = this.getAttribute('data-name');
            const count = parseInt(this.getAttribute('data-count'));
            
            document.getElementById('delete-id').value = id;
            document.getElementById('delete-name').textContent = name;

            const deleteWarning = document.getElementById('delete-warning');
            const deleteCount = document.getElementById('delete-count');

            if (count > 0) {
                deleteCount.textContent = count;
                deleteWarning.classList.remove('d-none');
            } else {
                deleteWarning.classList.add('d-none');
            }
            
            const deleteModal = new bootstrap.Modal(document.getElementById('deleteWorkflowModal'));
            deleteModal.show();
        });
    });

    // Submissão dos formulários de adicionar/editar workflow
    document.getElementById('addWorkflowForm').addEventListener('submit', function(event) {
        updateStepsJson('add-steps-container');
    });

    document.getElementById('editWorkflowForm').addEventListener('submit', function(event) {
        updateStepsJson('edit-steps-container');
    });
});
</script>

<?php
// Incluir rodapé
include "includes/footer.php";
?>

