import os
import sys
from flask import Flask, send_from_directory, jsonify, request, redirect, url_for
from flask_sqlalchemy import SQLAlchemy
from flask_cors import CORS
from werkzeug.utils import secure_filename
from werkzeug.security import generate_password_hash, check_password_hash
import datetime
import json

# Inicialização do banco de dados
db = SQLAlchemy()

# Função para garantir que o diretório de uploads exista
def ensure_upload_dir():
    upload_dir = os.path.join(os.path.dirname(os.path.abspath(__file__)), 'static', 'uploads')
    if not os.path.exists(upload_dir):
        os.makedirs(upload_dir)
    return upload_dir

# Inicialização da aplicação Flask
def create_app():
    app = Flask(__name__, static_folder=os.path.join(os.path.dirname(__file__), 'static'))
    
    # Configuração de segurança
    app.config['SECRET_KEY'] = os.getenv('SECRET_KEY', 'chave_secreta_padrao_alterar_em_producao')
    app.config['JWT_SECRET_KEY'] = os.getenv('JWT_SECRET_KEY', 'jwt_chave_secreta_padrao_alterar_em_producao')
    app.config['JWT_ACCESS_TOKEN_EXPIRES'] = 3600  # 1 hora
    
    # Habilitar CORS para produção
    CORS(app)
    
    # Configuração do banco de dados
    app.config['SQLALCHEMY_DATABASE_URI'] = f"mysql+pymysql://{os.getenv('DB_USERNAME', 'root')}:{os.getenv('DB_PASSWORD', 'password')}@{os.getenv('DB_HOST', 'localhost')}:{os.getenv('DB_PORT', '3306')}/{os.getenv('DB_NAME', 'ged_db1')}"
    app.config['SQLALCHEMY_TRACK_MODIFICATIONS'] = False
    
    # Configuração de upload
    app.config['UPLOAD_FOLDER'] = ensure_upload_dir()
    app.config['MAX_CONTENT_LENGTH'] = 50 * 1024 * 1024  # 50MB max upload
    
    # Inicializar banco de dados com a aplicação
    db.init_app(app)
    
    # Importar e registrar blueprints
    from .routes.auth import auth_bp
    from .routes.document import document_bp
    from .routes.workflow import workflow_bp
    from .routes.backup import backup_bp
    from .routes.user import user_bp
    
    app.register_blueprint(auth_bp, url_prefix='/api/auth')
    app.register_blueprint(document_bp, url_prefix='/api/documents')
    app.register_blueprint(workflow_bp, url_prefix='/api/workflow')
    app.register_blueprint(backup_bp, url_prefix='/api/backup')
    app.register_blueprint(user_bp, url_prefix='/api/users')
    
    # Rota de status para verificar se a API está funcionando
    @app.route('/api/status')
    def status():
        return jsonify({"status": "online", "message": "GED API está funcionando"})
    
    # Rota para servir arquivos estáticos e SPA
    @app.route('/', defaults={'path': ''})
    @app.route('/<path:path>')
    def serve(path):
        static_folder_path = app.static_folder
        if static_folder_path is None:
            return "Static folder not configured", 404
    
        if path != "" and os.path.exists(os.path.join(static_folder_path, path)):
            return send_from_directory(static_folder_path, path)
        else:
            index_path = os.path.join(static_folder_path, 'index.html')
            if os.path.exists(index_path):
                return send_from_directory(static_folder_path, 'index.html')
            else:
                return "index.html not found", 404
    
    return app

# Ponto de entrada para execução direta
if __name__ == '__main__':
    app = create_app()
    debug_mode = os.getenv('FLASK_ENV') == 'development'
    app.run(host='0.0.0.0', port=int(os.getenv('PORT', 5000)), debug=debug_mode)
