from flask import Blueprint, request, jsonify, current_app
from werkzeug.utils import secure_filename
from flask_jwt_extended import jwt_required, get_jwt_identity
import os
import datetime
from ..models.document import Document, DocumentVersion, DocumentKeyword, Category, db
from ..models.workflow import DocumentApproval, Workflow

document_bp = Blueprint('document', __name__)

# Extensões permitidas
ALLOWED_EXTENSIONS = {'pdf', 'doc', 'docx', 'xls', 'xlsx', 'ppt', 'pptx', 'txt', 'jpg', 'jpeg', 'png', 'gif'}

def allowed_file(filename):
    return '.' in filename and filename.rsplit('.', 1)[1].lower() in ALLOWED_EXTENSIONS

@document_bp.route('/', methods=['POST'])
@jwt_required()
def upload_document():
    """Upload de um novo documento"""
    current_user = get_jwt_identity()
    user_id = current_user.get('id')
    
    # Verificar se o arquivo foi enviado
    if 'file' not in request.files:
        return jsonify({"msg": "Nenhum arquivo enviado"}), 400
    
    file = request.files['file']
    
    # Verificar se o arquivo tem nome
    if file.filename == '':
        return jsonify({"msg": "Nenhum arquivo selecionado"}), 400
    
    # Verificar se o arquivo é permitido
    if not allowed_file(file.filename):
        return jsonify({"msg": "Tipo de arquivo não permitido"}), 400
    
    # Obter dados do formulário
    title = request.form.get('title', '')
    author = request.form.get('author', '')
    category_id = request.form.get('category_id')
    keywords = request.form.getlist('keywords')
    
    # Validar dados obrigatórios
    if not title:
        return jsonify({"msg": "Título é obrigatório"}), 400
    
    # Salvar o arquivo
    filename = secure_filename(file.filename)
    timestamp = datetime.datetime.now().strftime("%Y%m%d%H%M%S")
    unique_filename = f"{timestamp}_{filename}"
    file_path = os.path.join(current_app.config['UPLOAD_FOLDER'], unique_filename)
    file.save(file_path)
    
    # Criar documento no banco de dados
    new_document = Document(
        title=title,
        author=author,
        file_path=unique_filename,
        file_type=filename.rsplit('.', 1)[1].lower(),
        file_size=os.path.getsize(file_path),
        category_id=category_id,
        created_by=user_id
    )
    
    db.session.add(new_document)
    db.session.flush()  # Para obter o ID do documento
    
    # Adicionar palavras-chave
    for keyword in keywords:
        if keyword.strip():
            new_keyword = DocumentKeyword(
                document_id=new_document.id,
                keyword=keyword.strip()
            )
            db.session.add(new_keyword)
    
    # Criar primeira versão do documento
    new_version = DocumentVersion(
        document_id=new_document.id,
        version_number=1,
        file_path=unique_filename,
        file_size=os.path.getsize(file_path),
        created_by=user_id,
        change_summary="Versão inicial"
    )
    
    db.session.add(new_version)
    
    # Verificar se há workflow associado
    workflow_id = request.form.get('workflow_id')
    if workflow_id:
        workflow = Workflow.query.get(workflow_id)
        if workflow:
            new_approval = DocumentApproval(
                document_id=new_document.id,
                workflow_id=workflow_id,
                current_step=1,
                status='pending'
            )
            db.session.add(new_approval)
    
    db.session.commit()
    
    return jsonify({
        "msg": "Documento enviado com sucesso",
        "document": {
            "id": new_document.id,
            "title": new_document.title,
            "author": new_document.author,
            "file_path": new_document.file_path,
            "file_type": new_document.file_type,
            "file_size": new_document.file_size,
            "category_id": new_document.category_id,
            "status": new_document.status,
            "created_at": new_document.created_at
        }
    }), 201

@document_bp.route('/', methods=['GET'])
@jwt_required()
def get_documents():
    """Listar documentos com filtros opcionais"""
    # Parâmetros de filtro
    title = request.args.get('title')
    author = request.args.get('author')
    category_id = request.args.get('category_id')
    status = request.args.get('status')
    start_date = request.args.get('start_date')
    end_date = request.args.get('end_date')
    
    # Construir query base
    query = Document.query
    
    # Aplicar filtros
    if title:
        query = query.filter(Document.title.like(f'%{title}%'))
    if author:
        query = query.filter(Document.author.like(f'%{author}%'))
    if category_id:
        query = query.filter(Document.category_id == category_id)
    if status:
        query = query.filter(Document.status == status)
    if start_date:
        query = query.filter(Document.created_at >= start_date)
    if end_date:
        query = query.filter(Document.created_at <= end_date)
    
    # Executar query
    documents = query.order_by(Document.created_at.desc()).all()
    
    # Formatar resultado
    result = []
    for doc in documents:
        result.append({
            "id": doc.id,
            "title": doc.title,
            "author": doc.author,
            "file_path": doc.file_path,
            "file_type": doc.file_type,
            "file_size": doc.file_size,
            "category_id": doc.category_id,
            "status": doc.status,
            "created_at": doc.created_at,
            "updated_at": doc.updated_at
        })
    
    return jsonify({"documents": result}), 200

@document_bp.route('/<int:document_id>', methods=['GET'])
@jwt_required()
def get_document(document_id):
    """Obter detalhes de um documento específico"""
    document = Document.query.get(document_id)
    
    if not document:
        return jsonify({"msg": "Documento não encontrado"}), 404
    
    # Obter palavras-chave
    keywords = DocumentKeyword.query.filter_by(document_id=document_id).all()
    keywords_list = [k.keyword for k in keywords]
    
    # Obter versões
    versions = DocumentVersion.query.filter_by(document_id=document_id).order_by(DocumentVersion.version_number.desc()).all()
    versions_list = []
    for v in versions:
        versions_list.append({
            "id": v.id,
            "version_number": v.version_number,
            "file_path": v.file_path,
            "file_size": v.file_size,
            "created_at": v.created_at,
            "created_by": v.created_by,
            "change_summary": v.change_summary
        })
    
    # Obter aprovação se existir
    approval = DocumentApproval.query.filter_by(document_id=document_id).first()
    approval_info = None
    if approval:
        approval_info = {
            "id": approval.id,
            "workflow_id": approval.workflow_id,
            "current_step": approval.current_step,
            "status": approval.status,
            "started_at": approval.started_at,
            "completed_at": approval.completed_at
        }
    
    # Formatar resultado
    result = {
        "id": document.id,
        "title": document.title,
        "author": document.author,
        "file_path": document.file_path,
        "file_type": document.file_type,
        "file_size": document.file_size,
        "category_id": document.category_id,
        "status": document.status,
        "created_at": document.created_at,
        "updated_at": document.updated_at,
        "created_by": document.created_by,
        "keywords": keywords_list,
        "versions": versions_list,
        "approval": approval_info
    }
    
    return jsonify({"document": result}), 200

@document_bp.route('/<int:document_id>', methods=['PUT'])
@jwt_required()
def update_document(document_id):
    """Atualizar um documento existente"""
    current_user = get_jwt_identity()
    user_id = current_user.get('id')
    
    document = Document.query.get(document_id)
    if not document:
        return jsonify({"msg": "Documento não encontrado"}), 404
    
    # Obter dados do formulário
    data = request.form
    
    # Atualizar campos permitidos
    if 'title' in data:
        document.title = data['title']
    if 'author' in data:
        document.author = data['author']
    if 'category_id' in data:
        document.category_id = data['category_id']
    if 'status' in data:
        document.status = data['status']
    
    # Verificar se há novo arquivo
    if 'file' in request.files:
        file = request.files['file']
        if file and file.filename != '' and allowed_file(file.filename):
            # Salvar o novo arquivo
            filename = secure_filename(file.filename)
            timestamp = datetime.datetime.now().strftime("%Y%m%d%H%M%S")
            unique_filename = f"{timestamp}_{filename}"
            file_path = os.path.join(current_app.config['UPLOAD_FOLDER'], unique_filename)
            file.save(file_path)
            
            # Atualizar informações do documento
            document.file_path = unique_filename
            document.file_type = filename.rsplit('.', 1)[1].lower()
            document.file_size = os.path.getsize(file_path)
            
            # Criar nova versão
            version_count = DocumentVersion.query.filter_by(document_id=document_id).count()
            new_version = DocumentVersion(
                document_id=document_id,
                version_number=version_count + 1,
                file_path=unique_filename,
                file_size=os.path.getsize(file_path),
                created_by=user_id,
                change_summary=data.get('change_summary', 'Atualização do documento')
            )
            db.session.add(new_version)
    
    # Atualizar palavras-chave se fornecidas
    if 'keywords' in request.form:
        # Remover palavras-chave existentes
        DocumentKeyword.query.filter_by(document_id=document_id).delete()
        
        # Adicionar novas palavras-chave
        keywords = request.form.getlist('keywords')
        for keyword in keywords:
            if keyword.strip():
                new_keyword = DocumentKeyword(
                    document_id=document_id,
                    keyword=keyword.strip()
                )
                db.session.add(new_keyword)
    
    db.session.commit()
    
    return jsonify({"msg": "Documento atualizado com sucesso"}), 200

@document_bp.route('/<int:document_id>', methods=['DELETE'])
@jwt_required()
def delete_document(document_id):
    """Excluir um documento"""
    current_user = get_jwt_identity()
    user_type = current_user.get('user_type')
    
    # Verificar se é admin ou proprietário
    if user_type != 'admin':
        return jsonify({"msg": "Acesso negado. Apenas administradores podem excluir documentos"}), 403
    
    document = Document.query.get(document_id)
    if not document:
        return jsonify({"msg": "Documento não encontrado"}), 404
    
    # Excluir documento e relacionamentos (cascade)
    db.session.delete(document)
    db.session.commit()
    
    return jsonify({"msg": "Documento excluído com sucesso"}), 200

@document_bp.route('/categories', methods=['GET'])
@jwt_required()
def get_categories():
    """Listar todas as categorias"""
    categories = Category.query.all()
    result = []
    
    for category in categories:
        result.append({
            "id": category.id,
            "name": category.name,
            "description": category.description
        })
    
    return jsonify({"categories": result}), 200

@document_bp.route('/categories', methods=['POST'])
@jwt_required()
def create_category():
    """Criar uma nova categoria"""
    current_user = get_jwt_identity()
    user_id = current_user.get('id')
    
    if not request.is_json:
        return jsonify({"msg": "Requisição inválida, JSON esperado"}), 400
    
    data = request.json
    name = data.get('name')
    description = data.get('description', '')
    
    if not name:
        return jsonify({"msg": "Nome da categoria é obrigatório"}), 400
    
    # Verificar se já existe
    existing = Category.query.filter_by(name=name).first()
    if existing:
        return jsonify({"msg": "Categoria com este nome já existe"}), 400
    
    # Criar nova categoria
    new_category = Category(
        name=name,
        description=description,
        created_by=user_id
    )
    
    db.session.add(new_category)
    db.session.commit()
    
    return jsonify({
        "msg": "Categoria criada com sucesso",
        "category": {
            "id": new_category.id,
            "name": new_category.name,
            "description": new_category.description
        }
    }), 201

@document_bp.route('/download/<int:document_id>', methods=['GET'])
@jwt_required()
def download_document(document_id):
    """Obter URL para download do documento"""
    document = Document.query.get(document_id)
    
    if not document:
        return jsonify({"msg": "Documento não encontrado"}), 404
    
    # Em um ambiente real, aqui seria gerado um link seguro para download
    # No caso do cPanel, seria um link para o arquivo no diretório de uploads
    download_url = f"/static/uploads/{document.file_path}"
    
    return jsonify({
        "msg": "URL de download gerada com sucesso",
        "download_url": download_url,
        "filename": document.file_path.split('_', 1)[1]  # Remove o timestamp do nome
    }), 200
