<?php
/**
 * Página de gerenciamento de categorias
 * Sistema de Gerenciamento Eletrônico de Documentos (GED)
 */

// Iniciar sessão e verificar autenticação
session_start();
if (!isset($_SESSION['user_id'])) {
    header('Location: login.php');
    exit;
}

// Verificar se o usuário tem permissão de administrador
if ($_SESSION['user_type'] != 'admin') {
    $_SESSION['flash_message'] = "Acesso negado. Você não tem permissão para acessar esta página.";
    $_SESSION['flash_type'] = "danger";
    header('Location: index.php');
    exit;
}

// Incluir arquivos necessários
require_once 'config/database.php';
require_once 'models/category.php';
require_once 'controllers/category_controller.php';

// Conectar ao banco de dados
$database = new Database();
$db = $database->getConnection();

// Instanciar controlador
$category_controller = new CategoryController();

// Processar ações
if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    if (isset($_POST['action'])) {
        // Adicionar categoria
        if ($_POST['action'] == 'add') {
            $name = $_POST['name'] ?? '';
            $description = $_POST['description'] ?? '';
            
            if (!empty($name)) {
                $result = $category_controller->addCategory([
                    'name' => $name,
                    'description' => $description
                ]);
                
                if ($result['success']) {
                    $_SESSION['flash_message'] = "Categoria adicionada com sucesso.";
                    $_SESSION['flash_type'] = "success";
                } else {
                    $_SESSION['flash_message'] = $result['message'];
                    $_SESSION['flash_type'] = "danger";
                }
            } else {
                $_SESSION['flash_message'] = "O nome da categoria é obrigatório.";
                $_SESSION['flash_type'] = "danger";
            }
        }
        
        // Editar categoria
        else if ($_POST['action'] == 'edit') {
            $id = $_POST['id'] ?? '';
            $name = $_POST['name'] ?? '';
            $description = $_POST['description'] ?? '';
            
            if (!empty($id) && !empty($name)) {
                $result = $category_controller->updateCategory([
                    'id' => $id,
                    'name' => $name,
                    'description' => $description
                ]);
                
                if ($result['success']) {
                    $_SESSION['flash_message'] = "Categoria atualizada com sucesso.";
                    $_SESSION['flash_type'] = "success";
                } else {
                    $_SESSION['flash_message'] = $result['message'];
                    $_SESSION['flash_type'] = "danger";
                }
            } else {
                $_SESSION['flash_message'] = "ID e nome da categoria são obrigatórios.";
                $_SESSION['flash_type'] = "danger";
            }
        }
        
        // Excluir categoria
        else if ($_POST['action'] == 'delete') {
            $id = $_POST['id'] ?? '';
            
            if (!empty($id)) {
                $result = $category_controller->deleteCategory($id);
                
                if ($result['success']) {
                    $_SESSION['flash_message'] = "Categoria excluída com sucesso.";
                    $_SESSION['flash_type'] = "success";
                } else {
                    $_SESSION['flash_message'] = $result['message'];
                    $_SESSION['flash_type'] = "danger";
                }
            } else {
                $_SESSION['flash_message'] = "ID da categoria é obrigatório.";
                $_SESSION['flash_type'] = "danger";
            }
        }
        
        // Redirecionar para evitar reenvio do formulário
        header('Location: categories.php');
        exit;
    }
}

// Obter todas as categorias
$category = new Category($db);
$categories = $category->readAll();

// Definir título da página
$page_title = getenv('APP_PAGE_TITLE');

// Incluir cabeçalho
include 'includes/header.php';
?>

<div class="row mb-4">
    <div class="col-md-6">
        <h1><i class="fas fa-tags me-2"></i> Gerenciar Categorias</h1>
    </div>
    <div class="col-md-6 text-md-end">
        <button type="button" class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#addCategoryModal">
            <i class="fas fa-plus me-1"></i> Nova Categoria
        </button>
    </div>
</div>

<!-- Lista de categorias -->
<div class="card">
    <div class="card-body">
        <?php if (count($categories) > 0): ?>
        <div class="table-responsive">
            <table class="table table-hover datatable">
                <thead>
                    <tr>
                        <th>ID</th>
                        <th>Nome</th>
                        <th>Descrição</th>
                        <th>Documentos</th>
                        <th>Data de Criação</th>
                        <th>Ações</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($categories as $cat): ?>
                    <tr>
                        <td><?php echo $cat['id']; ?></td>
                        <td><?php echo htmlspecialchars($cat['name']); ?></td>
                        <td><?php echo htmlspecialchars($cat['description'] ?? ''); ?></td>
                        <td>
                            <a href="documents.php?category_id=<?php echo $cat['id']; ?>" class="badge bg-primary text-decoration-none">
                                <?php echo $cat['document_count']; ?> documentos
                            </a>
                        </td>
                        <td><?php echo date('d/m/Y', strtotime($cat['created_at'])); ?></td>
                        <td>
                            <div class="btn-group btn-group-sm">
                                <button type="button" class="btn btn-primary edit-category" 
                                        data-id="<?php echo $cat['id']; ?>"
                                        data-name="<?php echo htmlspecialchars($cat['name']); ?>"
                                        data-description="<?php echo htmlspecialchars($cat['description'] ?? ''); ?>">
                                    <i class="fas fa-edit"></i>
                                </button>
                                <button type="button" class="btn btn-danger delete-category" 
                                        data-id="<?php echo $cat['id']; ?>"
                                        data-name="<?php echo htmlspecialchars($cat['name']); ?>"
                                        data-count="<?php echo $cat['document_count']; ?>">
                                    <i class="fas fa-trash"></i>
                                </button>
                            </div>
                        </td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
        <?php else: ?>
        <div class="alert alert-info">
            <i class="fas fa-info-circle me-2"></i> Nenhuma categoria encontrada.
        </div>
        <?php endif; ?>
    </div>
</div>

<!-- Modal para adicionar categoria -->
<div class="modal fade" id="addCategoryModal" tabindex="-1" aria-labelledby="addCategoryModalLabel" aria-hidden="true">
    <div class="modal-dialog">
        <div class="modal-content">
            <form method="POST" action="categories.php">
                <input type="hidden" name="action" value="add">
                
                <div class="modal-header">
                    <h5 class="modal-title" id="addCategoryModalLabel">Nova Categoria</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Fechar"></button>
                </div>
                
                <div class="modal-body">
                    <div class="mb-3">
                        <label for="add-name" class="form-label">Nome <span class="text-danger">*</span></label>
                        <input type="text" class="form-control" id="add-name" name="name" required>
                    </div>
                    
                    <div class="mb-3">
                        <label for="add-description" class="form-label">Descrição</label>
                        <textarea class="form-control" id="add-description" name="description" rows="3"></textarea>
                    </div>
                </div>
                
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancelar</button>
                    <button type="submit" class="btn btn-primary">Adicionar</button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Modal para editar categoria -->
<div class="modal fade" id="editCategoryModal" tabindex="-1" aria-labelledby="editCategoryModalLabel" aria-hidden="true">
    <div class="modal-dialog">
        <div class="modal-content">
            <form method="POST" action="categories.php">
                <input type="hidden" name="action" value="edit">
                <input type="hidden" name="id" id="edit-id">
                
                <div class="modal-header">
                    <h5 class="modal-title" id="editCategoryModalLabel">Editar Categoria</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Fechar"></button>
                </div>
                
                <div class="modal-body">
                    <div class="mb-3">
                        <label for="edit-name" class="form-label">Nome <span class="text-danger">*</span></label>
                        <input type="text" class="form-control" id="edit-name" name="name" required>
                    </div>
                    
                    <div class="mb-3">
                        <label for="edit-description" class="form-label">Descrição</label>
                        <textarea class="form-control" id="edit-description" name="description" rows="3"></textarea>
                    </div>
                </div>
                
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancelar</button>
                    <button type="submit" class="btn btn-primary">Salvar</button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Modal para excluir categoria -->
<div class="modal fade" id="deleteCategoryModal" tabindex="-1" aria-labelledby="deleteCategoryModalLabel" aria-hidden="true">
    <div class="modal-dialog">
        <div class="modal-content">
            <form method="POST" action="categories.php">
                <input type="hidden" name="action" value="delete">
                <input type="hidden" name="id" id="delete-id">
                
                <div class="modal-header">
                    <h5 class="modal-title" id="deleteCategoryModalLabel">Excluir Categoria</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Fechar"></button>
                </div>
                
                <div class="modal-body">
                    <p>Tem certeza que deseja excluir a categoria <strong id="delete-name"></strong>?</p>
                    <div id="delete-warning" class="alert alert-warning d-none">
                        <i class="fas fa-exclamation-triangle me-2"></i> Esta categoria contém <strong id="delete-count"></strong> documentos. Ao excluí-la, os documentos ficarão sem categoria.
                    </div>
                </div>
                
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancelar</button>
                    <button type="submit" class="btn btn-danger">Excluir</button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- JavaScript para manipulação dos modais -->
<script>
document.addEventListener('DOMContentLoaded', function() {
    // Manipular clique no botão de editar
    document.querySelectorAll('.edit-category').forEach(function(button) {
        button.addEventListener('click', function() {
            const id = this.getAttribute('data-id');
            const name = this.getAttribute('data-name');
            const description = this.getAttribute('data-description');
            
            document.getElementById('edit-id').value = id;
            document.getElementById('edit-name').value = name;
            document.getElementById('edit-description').value = description;
            
            const editModal = new bootstrap.Modal(document.getElementById('editCategoryModal'));
            editModal.show();
        });
    });
    
    // Manipular clique no botão de excluir
    document.querySelectorAll('.delete-category').forEach(function(button) {
        button.addEventListener('click', function() {
            const id = this.getAttribute('data-id');
            const name = this.getAttribute('data-name');
            const count = parseInt(this.getAttribute('data-count'));
            
            document.getElementById('delete-id').value = id;
            document.getElementById('delete-name').textContent = name;
            
            // Mostrar aviso se a categoria contiver documentos
            if (count > 0) {
                document.getElementById('delete-count').textContent = count;
                document.getElementById('delete-warning').classList.remove('d-none');
            } else {
                document.getElementById('delete-warning').classList.add('d-none');
            }
            
            const deleteModal = new bootstrap.Modal(document.getElementById('deleteCategoryModal'));
            deleteModal.show();
        });
    });
});
</script>

<?php
// Incluir rodapé
include 'includes/footer.php';
?>
