<?php
/**
 * Controlador de Backup e Restauração
 * Sistema de Gerenciamento Eletrônico de Documentos (GED)
 */

require_once __DIR__ . "/../config/database.php";
require_once __DIR__ . "/../config/config.php";
require_once __DIR__ . "/../models/audit_log.php";

class BackupController {
    private $db;
    private $backup_dir;
    private $upload_dir;

    public function __construct() {
        $database = new Database();
        $this->db = $database->getConnection();
        $this->backup_dir = BASE_PATH . "/backups";
        $this->upload_dir = UPLOAD_PATH; // Diretório de uploads de documentos

        // Criar diretório de backups se não existir
        if (!file_exists($this->backup_dir)) {
            mkdir($this->backup_dir, 0755, true);
        }
    }

    /**
     * Cria um backup completo do banco de dados e dos arquivos de documentos.
     * @return array Resultado da operação
     */
    public function createBackup() {
        $timestamp = date("Ymd_His");
        $backup_filename = getenv('APP_BACKUP_FILENAME') . $timestamp . ".zip";
        $backup_filepath = $this->backup_dir . "/" . $backup_filename;

        try {
            // 1. Exportar banco de dados
            $db_host = DB_HOST;
            $db_name = DB_NAME;
            $db_user = DB_USER;
            $db_pass = DB_PASS;
            $mysql_dump_path = getenv('APP_MYSQL_DUMP_PATH'); // Pode precisar ser o caminho completo, ex: /usr/bin/mysqldump

            $db_backup_file = $this->backup_dir . "/db_backup_" . $timestamp . ".sql";
            $command = getenv('APP_COMMAND');
            
            // Executar o comando mysqldump
            exec($command, $output, $return_var);

            if ($return_var !== 0) {
                throw new Exception("Erro ao exportar o banco de dados: " . implode("\n", $output));
            }

            // 2. Criar arquivo ZIP com banco de dados e uploads
            $zip = new ZipArchive();
            if ($zip->open($backup_filepath, ZipArchive::CREATE | ZipArchive::OVERWRITE) !== TRUE) {
                throw new Exception("Não foi possível criar o arquivo ZIP.");
            }

            // Adicionar arquivo SQL ao ZIP
            $zip->addFile($db_backup_file, basename($db_backup_file));

            // Adicionar diretório de uploads recursivamente
            $files = new RecursiveIteratorIterator(
                new RecursiveDirectoryIterator($this->upload_dir),
                RecursiveIteratorIterator::LEAVES_ONLY
            );

            foreach ($files as $name => $file) {
                if (!$file->isDir()) {
                    $filePath = $file->getRealPath();
                    $relativePath = substr($filePath, strlen($this->upload_dir) + 1);
                    $zip->addFile($filePath, "uploads/" . $relativePath);
                }
            }

            $zip->close();

            // Remover arquivo SQL temporário
            unlink($db_backup_file);

            logAudit($_SESSION["user_id"], "create_backup", "system", null, "Backup do sistema criado: " . $backup_filename);
            return ["success" => true, "message" => "Backup criado com sucesso: " . $backup_filename];

        } catch (Exception $e) {
            error_log("Erro ao criar backup: " . $e->getMessage());
            logAudit($_SESSION["user_id"], "create_backup_fail", "system", null, "Falha ao criar backup: " . $e->getMessage());
            return ["success" => false, "message" => "Erro ao criar backup: " . $e->getMessage()];
        }
    }

    /**
     * Restaura um backup a partir de um arquivo ZIP.
     * @param string $zip_filepath Caminho para o arquivo ZIP do backup
     * @return array Resultado da operação
     */
    public function restoreBackup($zip_filepath) {
        try {
            $zip = new ZipArchive;
            if ($zip->open($zip_filepath) === TRUE) {
                $extract_path = BASE_PATH . "/temp_restore";
                if (!file_exists($extract_path)) {
                    mkdir($extract_path, 0755, true);
                }
                $zip->extractTo($extract_path);
                $zip->close();

                // Encontrar o arquivo SQL dentro do ZIP extraído
                $sql_file = null;
                foreach (scandir($extract_path) as $file) {
                    if (pathinfo($file, PATHINFO_EXTENSION) == "sql") {
                        $sql_file = $extract_path . "/" . $file;
                        break;
                    }
                }

                if (!$sql_file) {
                    throw new Exception("Arquivo SQL não encontrado no backup.");
                }

                // 1. Restaurar banco de dados
                $db_host = DB_HOST;
                $db_name = DB_NAME;
                $db_user = DB_USER;
                $db_pass = DB_PASS;
                $mysql_path = getenv('APP_MYSQL_PATH'); // Pode precisar ser o caminho completo, ex: /usr/bin/mysql

                // Limpar o banco de dados atual (opcional, mas recomendado para restauração completa)
                $this->clearDatabase();

                $command = getenv('APP_COMMAND');
                exec($command, $output, $return_var);

                if ($return_var !== 0) {
                    throw new Exception("Erro ao restaurar o banco de dados: " . implode("\n", $output));
                }

                // 2. Restaurar arquivos de upload
                // Remover uploads existentes (opcional, para garantir consistência)
                $this->deleteDirectory($this->upload_dir);
                mkdir($this->upload_dir, 0755, true);

                // Mover arquivos restaurados para o diretório de uploads
                $restored_uploads_path = $extract_path . "/uploads";
                if (file_exists($restored_uploads_path)) {
                    $this->copyDirectory($restored_uploads_path, $this->upload_dir);
                }

                // Limpar arquivos temporários
                $this->deleteDirectory($extract_path);

                logAudit($_SESSION["user_id"], "restore_backup", "system", null, "Sistema restaurado a partir do backup: " . basename($zip_filepath));
                return ["success" => true, "message" => "Backup restaurado com sucesso!"];

            } else {
                throw new Exception("Não foi possível abrir o arquivo ZIP de backup.");
            }
        } catch (Exception $e) {
            error_log("Erro ao restaurar backup: " . $e->getMessage());
            logAudit($_SESSION["user_id"], "restore_backup_fail", "system", null, "Falha ao restaurar backup: " . $e->getMessage());
            return ["success" => false, "message" => "Erro ao restaurar backup: " . $e->getMessage()];
        }
    }

    /**
     * Lista os arquivos de backup existentes.
     * @return array Lista de backups
     */
    public function listBackups() {
        $backups = [];
        if (file_exists($this->backup_dir)) {
            $files = scandir($this->backup_dir);
            foreach ($files as $file) {
                if (preg_match("/^ged_backup_.*\.zip$/", $file)) {
                    $filepath = $this->backup_dir . "/" . $file;
                    $backups[] = [
                        "name" => $file,
                        "size" => filesize($filepath),
                        "time" => filemtime($filepath)
                    ];
                }
            }
        }
        // Ordenar por data de criação (mais recente primeiro)
        usort($backups, function($a, $b) { return $b["time"] - $a["time"]; });
        return $backups;
    }

    /**
     * Exclui um arquivo de backup.
     * @param string $filename Nome do arquivo de backup a ser excluído
     * @return array Resultado da operação
     */
    public function deleteBackup($filename) {
        $filepath = $this->backup_dir . "/" . basename($filename); // Usar basename para evitar Path Traversal
        if (file_exists($filepath) && preg_match("/^ged_backup_.*\.zip$/", basename($filepath))) {
            if (unlink($filepath)) {
                logAudit($_SESSION["user_id"], "delete_backup", "system", null, "Backup excluído: " . basename($filepath));
                return ["success" => true, "message" => "Backup excluído com sucesso."];
            } else {
                return ["success" => false, "message" => "Erro ao excluir o arquivo de backup."];
            }
        } else {
            return ["success" => false, "message" => "Arquivo de backup não encontrado ou inválido."];
        }
    }

    /**
     * Limpa o banco de dados (exclui todas as tabelas).
     * CUIDADO: Usar apenas em restaurações ou ambientes de teste.
     */
    private function clearDatabase() {
        $tables = [];
        $query = getenv('APP_QUERY');
        $stmt = $this->db->query($query);
        while ($row = $stmt->fetch(PDO::FETCH_NUM)) {
            $tables[] = $row[0];
        }

        $this->db->exec("SET FOREIGN_KEY_CHECKS = 0;");
        foreach ($tables as $table) {
            $this->db->exec("DROP TABLE IF EXISTS `{$table}`");
        }
        $this->db->exec("SET FOREIGN_KEY_CHECKS = 1;");
    }

    /**
     * Copia um diretório recursivamente.
     * @param string $src Fonte
     * @param string $dst Destino
     */
    private function copyDirectory($src, $dst) {
        $dir = opendir($src);
        @mkdir($dst);
        while (false !== ($file = readdir($dir))) {
            if (($file != ".") && ($file != "..")) {
                if (is_dir($src . "/" . $file)) {
                    $this->copyDirectory($src . "/" . $file, $dst . "/" . $file);
                } else {
                    copy($src . "/" . $file, $dst . "/" . $file);
                }
            }
        }
        closedir($dir);
    }

    /**
     * Exclui um diretório recursivamente.
     * @param string $dir Diretório a ser excluído
     */
    private function deleteDirectory($dir) {
        if (!file_exists($dir)) {
            return true;
        }
        if (!is_dir($dir)) {
            return unlink($dir);
        }
        foreach (scandir($dir) as $item) {
            if ($item == "." || $item == "..") {
                continue;
            }
            if (!$this->deleteDirectory($dir . DIRECTORY_SEPARATOR . $item)) {
                return false;
            }
        }
        return rmdir($dir);
    }
}
?>

