<?php
/**
 * Controlador de Categorias
 * Sistema de Gerenciamento Eletrônico de Documentos (GED)
 */

// Incluir arquivos necessários
require_once __DIR__ . '/../config/database.php';
require_once __DIR__ . '/../models/category.php';

class CategoryController {
    private $db;
    private $category;
    
    /**
     * Construtor
     */
    public function __construct() {
        $database = new Database();
        $this->db = $database->getConnection();
        $this->category = new Category($this->db);
    }
    
    /**
     * Lista todas as categorias
     * @return array Lista de categorias
     */
    public function listCategories() {
        $stmt = $this->category->readAll();
        $categories = [];
        
        while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
            $categories[] = $row;
        }
        
        return [
            'success' => true,
            'categories' => $categories
        ];
    }
    
    /**
     * Obtém uma categoria pelo ID
     * @param int $id ID da categoria
     * @return array Categoria ou erro
     */
    public function getCategory($id) {
        if ($this->category->readOne($id)) {
            return [
                'success' => true,
                'category' => [
                    'id' => $this->category->id,
                    'name' => $this->category->name,
                    'description' => $this->category->description,
                    'created_at' => $this->category->created_at,
                    'created_by' => $this->category->created_by
                ]
            ];
        } else {
            return [
                'success' => false,
                'message' => 'Categoria não encontrada'
            ];
        }
    }
    
    /**
     * Cria uma nova categoria
     * @param array $data Dados da categoria
     * @param int $user_id ID do usuário que está criando
     * @return array Resultado da criação
     */
    public function createCategory($data, $user_id) {
        // Verificar campos obrigatórios
        if (empty($data['name'])) {
            return [
                'success' => false,
                'message' => 'O nome da categoria é obrigatório'
            ];
        }
        
        // Verificar se já existe uma categoria com o mesmo nome
        $query = getenv('APP_QUERY');
        $stmt = $this->db->prepare($query);
        $stmt->bindParam(1, $data['name']);
        $stmt->execute();
        
        if ($stmt->rowCount() > 0) {
            return [
                'success' => false,
                'message' => 'Já existe uma categoria com este nome'
            ];
        }
        
        // Criar categoria
        $this->category->name = $data['name'];
        $this->category->description = $data['description'] ?? '';
        $this->category->created_by = $user_id;
        
        if ($this->category->create()) {
            // Registrar log de auditoria
            logAudit($user_id, 'create', 'category', $this->category->id, 'Categoria criada: ' . $data['name']);
            
            return [
                'success' => true,
                'message' => 'Categoria criada com sucesso',
                'category_id' => $this->category->id
            ];
        } else {
            return [
                'success' => false,
                'message' => 'Erro ao criar categoria'
            ];
        }
    }
    
    /**
     * Atualiza uma categoria existente
     * @param int $id ID da categoria
     * @param array $data Novos dados
     * @param int $user_id ID do usuário que está atualizando
     * @return array Resultado da atualização
     */
    public function updateCategory($id, $data, $user_id) {
        // Verificar se a categoria existe
        if (!$this->category->readOne($id)) {
            return [
                'success' => false,
                'message' => 'Categoria não encontrada'
            ];
        }
        
        // Verificar campos obrigatórios
        if (empty($data['name'])) {
            return [
                'success' => false,
                'message' => 'O nome da categoria é obrigatório'
            ];
        }
        
        // Verificar se já existe outra categoria com o mesmo nome
        $query = getenv('APP_QUERY');
        $stmt = $this->db->prepare($query);
        $stmt->bindParam(1, $data['name']);
        $stmt->bindParam(2, $id);
        $stmt->execute();
        
        if ($stmt->rowCount() > 0) {
            return [
                'success' => false,
                'message' => 'Já existe outra categoria com este nome'
            ];
        }
        
        // Atualizar categoria
        $this->category->name = $data['name'];
        $this->category->description = $data['description'] ?? $this->category->description;
        
        if ($this->category->update()) {
            // Registrar log de auditoria
            logAudit($user_id, 'update', 'category', $id, 'Categoria atualizada: ' . $data['name']);
            
            return [
                'success' => true,
                'message' => 'Categoria atualizada com sucesso'
            ];
        } else {
            return [
                'success' => false,
                'message' => 'Erro ao atualizar categoria'
            ];
        }
    }
    
    /**
     * Exclui uma categoria
     * @param int $id ID da categoria
     * @param int $user_id ID do usuário que está excluindo
     * @return array Resultado da exclusão
     */
    public function deleteCategory($id, $user_id) {
        // Verificar se a categoria existe
        if (!$this->category->readOne($id)) {
            return [
                'success' => false,
                'message' => 'Categoria não encontrada'
            ];
        }
        
        // Salvar nome para log
        $category_name = $this->category->name;
        
        // Tentar excluir
        if ($this->category->delete()) {
            // Registrar log de auditoria
            logAudit($user_id, 'delete', 'category', $id, 'Categoria excluída: ' . $category_name);
            
            return [
                'success' => true,
                'message' => 'Categoria excluída com sucesso'
            ];
        } else {
            return [
                'success' => false,
                'message' => 'Não é possível excluir esta categoria porque existem documentos associados a ela'
            ];
        }
    }
    
    /**
     * Obtém estatísticas de documentos por categoria
     * @return array Estatísticas
     */
    public function getCategoryStats() {
        $stats = $this->category->countDocuments();
        
        return [
            'success' => true,
            'stats' => $stats
        ];
    }
}
?>
