<?php
/**
 * Página de upload de documento
 * Sistema de Gerenciamento Eletrônico de Documentos (GED)
 */

// Iniciar sessão e verificar autenticação
session_start();
if (!isset($_SESSION["user_id"])) {
    header("Location: login.php");
    exit;
}

// Incluir arquivos necessários
require_once "config/database.php";
require_once "models/document.php";
require_once "models/category.php";
require_once "models/workflow.php";
require_once "controllers/document_controller.php";

// Conectar ao banco de dados
$database = new Database();
$db = $database->getConnection();

// Instanciar controladores e modelos
$document_controller = new DocumentController();
$category = new Category($db);
$workflow = new Workflow($db);

// Obter categorias e workflows para o formulário
$categories = $category->readAll();
$workflows = $workflow->readAll();

// Processar formulário de upload
$error = "";
$success = "";
if ($_SERVER["REQUEST_METHOD"] == "POST") {
    // Verificar se um arquivo foi enviado
    if (!isset($_FILES["document_file"]) || $_FILES["document_file"]["error"] == UPLOAD_ERR_NO_FILE) {
        $error = getenv('APP_ERROR');
    } else {
        // Obter dados do formulário
        $title = $_POST["title"] ?? "";
        $description = $_POST["description"] ?? "";
        $category_id = $_POST["category_id"] ?? "";
        $keywords = $_POST["keywords"] ?? "";
        $workflow_id = $_POST["workflow_id"] ?? null;
        
        // Validar campos obrigatórios
        if (empty($title) || empty($category_id)) {
            $error = getenv('APP_ERROR');
        } else {
            // Preparar dados para o controlador
            $document_data = [
                "title" => $title,
                "description" => $description,
                "category_id" => $category_id,
                "keywords" => $keywords,
                "workflow_id" => $workflow_id,
                "author_id" => $_SESSION["user_id"],
                "file" => $_FILES["document_file"]
            ];
            
            // Tentar fazer upload do documento
            $result = $document_controller->uploadDocument($document_data);
            
            if ($result["success"]) {
                $success = getenv('APP_SUCCESS');
                
                // Redirecionar para a página de visualização do documento
                $_SESSION["flash_message"] = $success;
                $_SESSION["flash_type"] = "success";
                header("Location: document_view.php?id=" . $result["document_id"]);
                exit;
            } else {
                $error = $result["message"];
            }
        }
    }
}

// Definir título da página
$page_title = getenv('APP_PAGE_TITLE');

// Incluir cabeçalho
include "includes/header.php";
?>

<div class="row mb-4">
    <div class="col-md-6">
        <h1><i class="fas fa-cloud-upload-alt me-2"></i> Upload de Documento</h1>
    </div>
    <div class="col-md-6 text-md-end">
        <a href="documents.php" class="btn btn-secondary">
            <i class="fas fa-arrow-left me-1"></i> Voltar para Documentos
        </a>
    </div>
</div>

<?php if (!empty($error)): ?>
<div class="alert alert-danger">
    <i class="fas fa-exclamation-circle me-2"></i> <?php echo $error; ?>
</div>
<?php endif; ?>

<?php if (!empty($success)): ?>
<div class="alert alert-success">
    <i class="fas fa-check-circle me-2"></i> <?php echo $success; ?>
</div>
<?php endif; ?>

<div class="card">
    <div class="card-body">
        <form method="POST" action="document_add.php" enctype="multipart/form-data">
            <div class="row">
                <div class="col-md-8">
                    <!-- Informações do documento -->
                    <div class="mb-3">
                        <label for="title" class="form-label">Título <span class="text-danger">*</span></label>
                        <input type="text" class="form-control" id="title" name="title" required>
                    </div>
                    
                    <div class="mb-3">
                        <label for="description" class="form-label">Descrição</label>
                        <textarea class="form-control" id="description" name="description" rows="3"></textarea>
                    </div>
                    
                    <div class="mb-3">
                        <label for="category_id" class="form-label">Categoria <span class="text-danger">*</span></label>
                        <select class="form-select select2" id="category_id" name="category_id" required>
                            <option value="">Selecione uma categoria</option>
                            <?php foreach ($categories as $cat): ?>
                            <option value="<?php echo $cat["id"]; ?>"><?php echo htmlspecialchars($cat["name"]); ?></option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    
                    <div class="mb-3">
                        <label for="workflow_id" class="form-label">Workflow de Aprovação (Opcional)</label>
                        <select class="form-select select2" id="workflow_id" name="workflow_id">
                            <option value="">Nenhum workflow</option>
                            <?php while ($wf = $workflows->fetch(PDO::FETCH_ASSOC)): ?>
                            <option value="<?php echo $wf["id"]; ?>"><?php echo htmlspecialchars($wf["name"]); ?></option>
                            <?php endwhile; ?>
                        </select>
                        <div class="form-text">Selecione um workflow para que o documento passe por um processo de aprovação.</div>
                    </div>

                    <div class="mb-3">
                        <label for="keywords" class="form-label">Palavras-chave</label>
                        <input type="text" class="form-control" id="keywords" name="keywords" placeholder="Separe as palavras-chave por vírgulas">
                        <div class="form-text">Exemplo: relatório, financeiro, 2023</div>
                    </div>
                </div>
                
                <div class="col-md-4">
                    <!-- Upload de arquivo -->
                    <div class="card h-100">
                        <div class="card-body">
                            <h5 class="card-title">Arquivo</h5>
                            
                            <div class="mb-3 text-center">
                                <div class="document-upload-area p-4 border rounded mb-3" id="dropArea">
                                    <i class="fas fa-cloud-upload-alt fa-3x text-primary mb-3"></i>
                                    <p>Arraste e solte o arquivo aqui ou clique para selecionar</p>
                                    <input type="file" class="form-control" id="document_file" name="document_file" style="display: none;">
                                </div>
                                
                                <div id="fileInfo" class="text-start d-none">
                                    <div class="alert alert-info">
                                        <div class="d-flex align-items-center">
                                            <i class="fas fa-file-alt fa-2x me-3"></i>
                                            <div>
                                                <div id="fileName" class="fw-bold"></div>
                                                <div id="fileSize" class="small"></div>
                                            </div>
                                            <button type="button" class="btn-close ms-auto" id="removeFile"></button>
                                        </div>
                                    </div>
                                </div>
                                
                                <button type="button" class="btn btn-outline-primary" id="browseButton">
                                    <i class="fas fa-folder-open me-1"></i> Selecionar Arquivo
                                </button>
                            </div>
                            
                            <div class="mb-3">
                                <label class="form-label">Tipos de arquivo permitidos</label>
                                <div class="form-text">
                                    <ul class="list-unstyled mb-0">
                                        <li><i class="fas fa-check-circle text-success me-1"></i> PDF (.pdf)</li>
                                        <li><i class="fas fa-check-circle text-success me-1"></i> Word (.doc, .docx)</li>
                                        <li><i class="fas fa-check-circle text-success me-1"></i> Excel (.xls, .xlsx)</li>
                                        <li><i class="fas fa-check-circle text-success me-1"></i> Imagens (.jpg, .jpeg, .png, .gif)</li>
                                        <li><i class="fas fa-check-circle text-success me-1"></i> Texto (.txt)</li>
                                    </ul>
                                </div>
                            </div>
                            
                            <div class="mb-3">
                                <label class="form-label">Tamanho máximo</label>
                                <div class="form-text">20 MB</div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            
            <hr>
            
            <div class="d-flex justify-content-between">
                <button type="button" class="btn btn-secondary" onclick="window.location.href=\'documents.php\'" >
                    <i class="fas fa-times me-1"></i> Cancelar
                </button>
                <button type="submit" class="btn btn-primary">
                    <i class="fas fa-cloud-upload-alt me-1"></i> Enviar Documento
                </button>
            </div>
        </form>
    </div>
</div>

<!-- JavaScript para upload de arquivos -->
<script>
document.addEventListener("DOMContentLoaded", function() {
    const dropArea = document.getElementById("dropArea");
    const fileInput = document.getElementById("document_file");
    const browseButton = document.getElementById("browseButton");
    const fileInfo = document.getElementById("fileInfo");
    const fileName = document.getElementById("fileName");
    const fileSize = document.getElementById("fileSize");
    const removeFile = document.getElementById("removeFile");
    
    // Função para formatar tamanho de arquivo
    function formatFileSize(bytes) {
        if (bytes === 0) return "0 Bytes";
        
        const k = 1024;
        const sizes = ["Bytes", "KB", "MB", "GB", "TB"];
        const i = Math.floor(Math.log(bytes) / Math.log(k));
        
        return parseFloat((bytes / Math.pow(k, i)).toFixed(2)) + " " + sizes[i];
    }
    
    // Função para atualizar informações do arquivo
    function updateFileInfo() {
        if (fileInput.files.length > 0) {
            const file = fileInput.files[0];
            fileName.textContent = file.name;
            fileSize.textContent = formatFileSize(file.size);
            fileInfo.classList.remove("d-none");
            dropArea.classList.add("border-primary");
        } else {
            fileInfo.classList.add("d-none");
            dropArea.classList.remove("border-primary");
        }
    }
    
    // Evento de clique no botão de navegação
    browseButton.addEventListener("click", function() {
        fileInput.click();
    });
    
    // Evento de clique na área de drop
    dropArea.addEventListener("click", function() {
        fileInput.click();
    });
    
    // Evento de alteração do input de arquivo
    fileInput.addEventListener("change", updateFileInfo);
    
    // Evento de clique no botão de remover
    removeFile.addEventListener("click", function(e) {
        e.preventDefault();
        fileInput.value = "";
        updateFileInfo();
    });
    
    // Eventos de drag and drop
    ["dragenter", "dragover", "dragleave", "drop"].forEach(eventName => {
        dropArea.addEventListener(eventName, function(e) {
            e.preventDefault();
            e.stopPropagation();
        }, false);
    });
    
    ["dragenter", "dragover"].forEach(eventName => {
        dropArea.addEventListener(eventName, function() {
            dropArea.classList.add("border-primary");
        }, false);
    });
    
    ["dragleave", "drop"].forEach(eventName => {
        dropArea.addEventListener(eventName, function() {
            dropArea.classList.remove("border-primary");
        }, false);
    });
    
    dropArea.addEventListener("drop", function(e) {
        fileInput.files = e.dataTransfer.files;
        updateFileInfo();
    }, false);
});
</script>

<!-- CSS para área de upload -->
<style>
.document-upload-area {
    cursor: pointer;
    transition: all 0.3s ease;
    text-align: center;
    padding: 2rem;
}

.document-upload-area:hover {
    background-color: rgba(var(--bs-primary-rgb), 0.05);
}
</style>

<?php
// Incluir rodapé
include "includes/footer.php";
?>

