<?php
/**
 * Página de listagem de documentos
 * Sistema de Gerenciamento Eletrônico de Documentos (GED)
 */

// Iniciar sessão e verificar autenticação
session_start();
if (!isset($_SESSION['user_id'])) {
    header('Location: login.php');
    exit;
}

// Incluir arquivos necessários
require_once 'config/database.php';
require_once 'models/document.php';
require_once 'models/category.php';
require_once 'controllers/document_controller.php';

// Conectar ao banco de dados
$database = new Database();
$db = $database->getConnection();

// Instanciar controlador
$document_controller = new DocumentController();

// Processar filtros
$filters = [];
if (isset($_GET['category_id']) && !empty($_GET['category_id'])) {
    $filters['category_id'] = $_GET['category_id'];
}
if (isset($_GET['status']) && !empty($_GET['status'])) {
    $filters['status'] = $_GET['status'];
}
if (isset($_GET['date_from']) && !empty($_GET['date_from'])) {
    $filters['date_from'] = $_GET['date_from'];
}
if (isset($_GET['date_to']) && !empty($_GET['date_to'])) {
    $filters['date_to'] = $_GET['date_to'];
}
if (isset($_GET['search']) && !empty($_GET['search'])) {
    $filters['search'] = $_GET['search'];
}

// Paginação
$page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
$per_page = 10;

// Obter documentos
$result = $document_controller->listDocuments($filters, $page, $per_page);
$documents = $result['documents'];
$pagination = $result['pagination'];

// Obter categorias para filtro
$category = new Category($db);
$categories = $category->readAll();

// Definir título da página
$page_title = getenv('APP_PAGE_TITLE');

// Incluir cabeçalho
include 'includes/header.php';
?>

<div class="row mb-4">
    <div class="col-md-6">
        <h1><i class="fas fa-file-alt me-2"></i> Documentos</h1>
    </div>
    <div class="col-md-6 text-md-end">
        <a href="document_add.php" class="btn btn-primary">
            <i class="fas fa-plus me-1"></i> Novo Documento
        </a>
    </div>
</div>

<!-- Filtros -->
<div class="card mb-4">
    <div class="card-header bg-light">
        <a class="text-decoration-none text-dark" data-bs-toggle="collapse" href="#collapseFilters">
            <i class="fas fa-filter me-1"></i> Filtros
        </a>
    </div>
    <div class="collapse" id="collapseFilters">
        <div class="card-body">
            <form method="GET" action="documents.php" class="row g-3">
                <div class="col-md-3">
                    <label for="category_id" class="form-label">Categoria</label>
                    <select name="category_id" id="category_id" class="form-select select2">
                        <option value="">Todas as categorias</option>
                        <?php foreach ($categories as $cat): ?>
                        <option value="<?php echo $cat['id']; ?>" <?php echo (isset($filters['category_id']) && $filters['category_id'] == $cat['id']) ? 'selected' : ''; ?>>
                            <?php echo htmlspecialchars($cat['name']); ?>
                        </option>
                        <?php endforeach; ?>
                    </select>
                </div>
                
                <div class="col-md-3">
                    <label for="status" class="form-label">Status</label>
                    <select name="status" id="status" class="form-select">
                        <option value="">Todos os status</option>
                        <option value="draft" <?php echo (isset($filters['status']) && $filters['status'] == 'draft') ? 'selected' : ''; ?>>Rascunho</option>
                        <option value="pending" <?php echo (isset($filters['status']) && $filters['status'] == 'pending') ? 'selected' : ''; ?>>Pendente</option>
                        <option value="approved" <?php echo (isset($filters['status']) && $filters['status'] == 'approved') ? 'selected' : ''; ?>>Aprovado</option>
                        <option value="rejected" <?php echo (isset($filters['status']) && $filters['status'] == 'rejected') ? 'selected' : ''; ?>>Rejeitado</option>
                    </select>
                </div>
                
                <div class="col-md-3">
                    <label for="date_from" class="form-label">Data Inicial</label>
                    <input type="date" class="form-control" id="date_from" name="date_from" value="<?php echo $filters['date_from'] ?? ''; ?>">
                </div>
                
                <div class="col-md-3">
                    <label for="date_to" class="form-label">Data Final</label>
                    <input type="date" class="form-control" id="date_to" name="date_to" value="<?php echo $filters['date_to'] ?? ''; ?>">
                </div>
                
                <div class="col-md-6">
                    <label for="search" class="form-label">Pesquisar</label>
                    <input type="text" class="form-control" id="search" name="search" placeholder="Título, descrição, palavras-chave..." value="<?php echo $filters['search'] ?? ''; ?>">
                </div>
                
                <div class="col-12">
                    <button type="submit" class="btn btn-primary">
                        <i class="fas fa-search me-1"></i> Filtrar
                    </button>
                    <a href="documents.php" class="btn btn-secondary">
                        <i class="fas fa-times me-1"></i> Limpar Filtros
                    </a>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Lista de documentos -->
<div class="card">
    <div class="card-body">
        <?php if (count($documents) > 0): ?>
        <div class="table-responsive">
            <table class="table table-hover datatable">
                <thead>
                    <tr>
                        <th>Título</th>
                        <th>Categoria</th>
                        <th>Autor</th>
                        <th>Status</th>
                        <th>Data</th>
                        <th>Ações</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($documents as $doc): ?>
                    <tr>
                        <td>
                            <a href="document_view.php?id=<?php echo $doc['id']; ?>">
                                <?php echo htmlspecialchars($doc['title']); ?>
                            </a>
                        </td>
                        <td><?php echo htmlspecialchars($doc['category_name']); ?></td>
                        <td><?php echo htmlspecialchars($doc['author_name']); ?></td>
                        <td>
                            <?php if ($doc['status'] == 'approved'): ?>
                            <span class="badge bg-success">Aprovado</span>
                            <?php elseif ($doc['status'] == 'pending'): ?>
                            <span class="badge bg-warning text-dark">Pendente</span>
                            <?php elseif ($doc['status'] == 'rejected'): ?>
                            <span class="badge bg-danger">Rejeitado</span>
                            <?php else: ?>
                            <span class="badge bg-secondary">Rascunho</span>
                            <?php endif; ?>
                        </td>
                        <td><?php echo date('d/m/Y', strtotime($doc['created_at'])); ?></td>
                        <td>
                            <div class="btn-group btn-group-sm">
                                <a href="document_view.php?id=<?php echo $doc['id']; ?>" class="btn btn-info" title="Visualizar">
                                    <i class="fas fa-eye"></i>
                                </a>
                                <a href="document_download.php?id=<?php echo $doc['id']; ?>" class="btn btn-success" title="Download">
                                    <i class="fas fa-download"></i>
                                </a>
                                <?php if ($_SESSION['user_id'] == $doc['author_id'] || $_SESSION['user_type'] == 'admin'): ?>
                                <a href="document_edit.php?id=<?php echo $doc['id']; ?>" class="btn btn-primary" title="Editar">
                                    <i class="fas fa-edit"></i>
                                </a>
                                <a href="document_delete.php?id=<?php echo $doc['id']; ?>" class="btn btn-danger" title="Excluir" onclick="return confirm('Tem certeza que deseja excluir este documento?');">
                                    <i class="fas fa-trash"></i>
                                </a>
                                <?php endif; ?>
                            </div>
                        </td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
        
        <!-- Paginação -->
        <?php if ($pagination['total_pages'] > 1): ?>
        <nav aria-label="Navegação de página">
            <ul class="pagination justify-content-center mt-4">
                <?php if ($pagination['current_page'] > 1): ?>
                <li class="page-item">
                    <a class="page-link" href="?page=1<?php echo isset($_GET['category_id']) ? '&category_id=' . $_GET['category_id'] : ''; ?><?php echo isset($_GET['status']) ? '&status=' . $_GET['status'] : ''; ?><?php echo isset($_GET['search']) ? '&search=' . $_GET['search'] : ''; ?>">
                        <i class="fas fa-angle-double-left"></i>
                    </a>
                </li>
                <li class="page-item">
                    <a class="page-link" href="?page=<?php echo $pagination['current_page'] - 1; ?><?php echo isset($_GET['category_id']) ? '&category_id=' . $_GET['category_id'] : ''; ?><?php echo isset($_GET['status']) ? '&status=' . $_GET['status'] : ''; ?><?php echo isset($_GET['search']) ? '&search=' . $_GET['search'] : ''; ?>">
                        <i class="fas fa-angle-left"></i>
                    </a>
                </li>
                <?php endif; ?>
                
                <?php
                $start_page = max(1, $pagination['current_page'] - 2);
                $end_page = min($pagination['total_pages'], $pagination['current_page'] + 2);
                
                for ($i = $start_page; $i <= $end_page; $i++):
                ?>
                <li class="page-item <?php echo $i == $pagination['current_page'] ? 'active' : ''; ?>">
                    <a class="page-link" href="?page=<?php echo $i; ?><?php echo isset($_GET['category_id']) ? '&category_id=' . $_GET['category_id'] : ''; ?><?php echo isset($_GET['status']) ? '&status=' . $_GET['status'] : ''; ?><?php echo isset($_GET['search']) ? '&search=' . $_GET['search'] : ''; ?>">
                        <?php echo $i; ?>
                    </a>
                </li>
                <?php endfor; ?>
                
                <?php if ($pagination['current_page'] < $pagination['total_pages']): ?>
                <li class="page-item">
                    <a class="page-link" href="?page=<?php echo $pagination['current_page'] + 1; ?><?php echo isset($_GET['category_id']) ? '&category_id=' . $_GET['category_id'] : ''; ?><?php echo isset($_GET['status']) ? '&status=' . $_GET['status'] : ''; ?><?php echo isset($_GET['search']) ? '&search=' . $_GET['search'] : ''; ?>">
                        <i class="fas fa-angle-right"></i>
                    </a>
                </li>
                <li class="page-item">
                    <a class="page-link" href="?page=<?php echo $pagination['total_pages']; ?><?php echo isset($_GET['category_id']) ? '&category_id=' . $_GET['category_id'] : ''; ?><?php echo isset($_GET['status']) ? '&status=' . $_GET['status'] : ''; ?><?php echo isset($_GET['search']) ? '&search=' . $_GET['search'] : ''; ?>">
                        <i class="fas fa-angle-double-right"></i>
                    </a>
                </li>
                <?php endif; ?>
            </ul>
        </nav>
        <?php endif; ?>
        
        <?php else: ?>
        <div class="alert alert-info">
            <i class="fas fa-info-circle me-2"></i> Nenhum documento encontrado.
        </div>
        <?php endif; ?>
    </div>
</div>

<?php
// Incluir rodapé
include 'includes/footer.php';
?>
