<?php
/**
 * Página de gerenciamento de usuários
 * Sistema de Gerenciamento Eletrônico de Documentos (GED)
 */

// Iniciar sessão e verificar autenticação
session_start();
if (!isset($_SESSION["user_id"])) {
    header("Location: login.php");
    exit;
}

// Verificar se o usuário tem permissão de administrador
if ($_SESSION["user_type"] != "admin") {
    $_SESSION["flash_message"] = "Acesso negado. Você não tem permissão para acessar esta página.";
    $_SESSION["flash_type"] = "danger";
    header("Location: index.php");
    exit;
}

// Incluir arquivos necessários
require_once "config/database.php";
require_once "models/user.php";
require_once "controllers/auth_controller.php";

// Conectar ao banco de dados
$database = new Database();
$db = $database->getConnection();

// Instanciar controlador
$auth_controller = new AuthController();

// Processar ações
if ($_SERVER["REQUEST_METHOD"] == "POST") {
    if (isset($_POST["action"])) {
        // Adicionar usuário
        if ($_POST["action"] == "add") {
            $username = $_POST["username"] ?? "";
            $password = $_POST["password"] ?? "";
            $name = $_POST["name"] ?? "";
            $email = $_POST["email"] ?? "";
            $user_type = $_POST["user_type"] ?? "user";
            
            if (!empty($username) && !empty($password) && !empty($name) && !empty($email)) {
                $result = $auth_controller->registerUser([
                    "username" => $username,
                    "password" => $password,
                    "name" => $name,
                    "email" => $email,
                    "user_type" => $user_type
                ]);
                
                if ($result["success"]) {
                    $_SESSION["flash_message"] = "Usuário adicionado com sucesso.";
                    $_SESSION["flash_type"] = "success";
                } else {
                    $_SESSION["flash_message"] = $result["message"];
                    $_SESSION["flash_type"] = "danger";
                }
            } else {
                $_SESSION["flash_message"] = "Por favor, preencha todos os campos obrigatórios.";
                $_SESSION["flash_type"] = "danger";
            }
        }
        
        // Editar usuário
        else if ($_POST["action"] == "edit") {
            $id = $_POST["id"] ?? "";
            $username = $_POST["username"] ?? "";
            $name = $_POST["name"] ?? "";
            $email = $_POST["email"] ?? "";
            $user_type = $_POST["user_type"] ?? "user";
            $password = $_POST["password"] ?? ""; // Senha opcional
            
            if (!empty($id) && !empty($username) && !empty($name) && !empty($email)) {
                $user_data = [
                    "id" => $id,
                    "username" => $username,
                    "name" => $name,
                    "email" => $email,
                    "user_type" => $user_type
                ];
                if (!empty($password)) {
                    $user_data["password"] = $password;
                }

                $result = $auth_controller->updateUser($user_data);
                
                if ($result["success"]) {
                    $_SESSION["flash_message"] = "Usuário atualizado com sucesso.";
                    $_SESSION["flash_type"] = "success";
                } else {
                    $_SESSION["flash_message"] = $result["message"];
                    $_SESSION["flash_type"] = "danger";
                }
            } else {
                $_SESSION["flash_message"] = "Por favor, preencha todos os campos obrigatórios.";
                $_SESSION["flash_type"] = "danger";
            }
        }
        
        // Excluir usuário
        else if ($_POST["action"] == "delete") {
            $id = $_POST["id"] ?? "";
            
            if (!empty($id)) {
                $result = $auth_controller->deleteUser($id);
                
                if ($result["success"]) {
                    $_SESSION["flash_message"] = "Usuário excluído com sucesso.";
                    $_SESSION["flash_type"] = "success";
                } else {
                    $_SESSION["flash_message"] = $result["message"];
                    $_SESSION["flash_type"] = "danger";
                }
            } else {
                $_SESSION["flash_message"] = "ID do usuário é obrigatório.";
                $_SESSION["flash_type"] = "danger";
            }
        }
        
        // Redirecionar para evitar reenvio do formulário
        header("Location: users.php");
        exit;
    }
}

// Obter todos os usuários
$user = new User($db);
$users = $user->readAll();

// Definir título da página
$page_title = getenv('APP_PAGE_TITLE');

// Incluir cabeçalho
include "includes/header.php";
?>

<div class="row mb-4">
    <div class="col-md-6">
        <h1><i class="fas fa-users me-2"></i> Gerenciar Usuários</h1>
    </div>
    <div class="col-md-6 text-md-end">
        <button type="button" class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#addUserModal">
            <i class="fas fa-plus me-1"></i> Novo Usuário
        </button>
    </div>
</div>

<!-- Lista de usuários -->
<div class="card">
    <div class="card-body">
        <?php if (count($users) > 0): ?>
        <div class="table-responsive">
            <table class="table table-hover datatable">
                <thead>
                    <tr>
                        <th>ID</th>
                        <th>Usuário</th>
                        <th>Nome</th>
                        <th>Email</th>
                        <th>Tipo</th>
                        <th>Status</th>
                        <th>Data de Criação</th>
                        <th>Ações</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($users as $u): ?>
                    <tr>
                        <td><?php echo $u["id"]; ?></td>
                        <td><?php echo htmlspecialchars($u["username"]); ?></td>
                        <td><?php echo htmlspecialchars($u["name"]); ?></td>
                        <td><?php echo htmlspecialchars($u["email"]); ?></td>
                        <td>
                            <?php if ($u["user_type"] == "admin"): ?>
                                <span class="badge bg-danger">Administrador</span>
                            <?php else: ?>
                                <span class="badge bg-secondary">Usuário Comum</span>
                            <?php endif; ?>
                        </td>
                        <td>
                            <?php if ($u["status"] == "active"): ?>
                                <span class="badge bg-success">Ativo</span>
                            <?php else: ?>
                                <span class="badge bg-warning text-dark">Inativo</span>
                            <?php endif; ?>
                        </td>
                        <td><?php echo date("d/m/Y", strtotime($u["created_at"])); ?></td>
                        <td>
                            <div class="btn-group btn-group-sm">
                                <button type="button" class="btn btn-primary edit-user" 
                                        data-id="<?php echo $u["id"]; ?>"
                                        data-username="<?php echo htmlspecialchars($u["username"]); ?>"
                                        data-name="<?php echo htmlspecialchars($u["name"]); ?>"
                                        data-email="<?php echo htmlspecialchars($u["email"]); ?>"
                                        data-user_type="<?php echo htmlspecialchars($u["user_type"]); ?>">
                                    <i class="fas fa-edit"></i>
                                </button>
                                <button type="button" class="btn btn-danger delete-user" 
                                        data-id="<?php echo $u["id"]; ?>"
                                        data-username="<?php echo htmlspecialchars($u["username"]); ?>">
                                    <i class="fas fa-trash"></i>
                                </button>
                            </div>
                        </td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
        <?php else: ?>
        <div class="alert alert-info">
            <i class="fas fa-info-circle me-2"></i> Nenhum usuário encontrado.
        </div>
        <?php endif; ?>
    </div>
</div>

<!-- Modal para adicionar usuário -->
<div class="modal fade" id="addUserModal" tabindex="-1" aria-labelledby="addUserModalLabel" aria-hidden="true">
    <div class="modal-dialog">
        <div class="modal-content">
            <form method="POST" action="users.php">
                <input type="hidden" name="action" value="add">
                
                <div class="modal-header">
                    <h5 class="modal-title" id="addUserModalLabel">Novo Usuário</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Fechar"></button>
                </div>
                
                <div class="modal-body">
                    <div class="mb-3">
                        <label for="add-username" class="form-label">Nome de Usuário <span class="text-danger">*</span></label>
                        <input type="text" class="form-control" id="add-username" name="username" required>
                    </div>
                    
                    <div class="mb-3">
                        <label for="add-password" class="form-label">Senha <span class="text-danger">*</span></label>
                        <input type="password" class="form-control" id="add-password" name="password" required>
                    </div>

                    <div class="mb-3">
                        <label for="add-name" class="form-label">Nome Completo <span class="text-danger">*</span></label>
                        <input type="text" class="form-control" id="add-name" name="name" required>
                    </div>

                    <div class="mb-3">
                        <label for="add-email" class="form-label">Email <span class="text-danger">*</span></label>
                        <input type="email" class="form-control" id="add-email" name="email" required>
                    </div>

                    <div class="mb-3">
                        <label for="add-user_type" class="form-label">Tipo de Usuário</label>
                        <select class="form-select" id="add-user_type" name="user_type">
                            <option value="user">Usuário Comum</option>
                            <option value="admin">Administrador</option>
                        </select>
                    </div>
                </div>
                
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancelar</button>
                    <button type="submit" class="btn btn-primary">Adicionar</button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Modal para editar usuário -->
<div class="modal fade" id="editUserModal" tabindex="-1" aria-labelledby="editUserModalLabel" aria-hidden="true">
    <div class="modal-dialog">
        <div class="modal-content">
            <form method="POST" action="users.php">
                <input type="hidden" name="action" value="edit">
                <input type="hidden" name="id" id="edit-id">
                
                <div class="modal-header">
                    <h5 class="modal-title" id="editUserModalLabel">Editar Usuário</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Fechar"></button>
                </div>
                
                <div class="modal-body">
                    <div class="mb-3">
                        <label for="edit-username" class="form-label">Nome de Usuário <span class="text-danger">*</span></label>
                        <input type="text" class="form-control" id="edit-username" name="username" required>
                    </div>
                    
                    <div class="mb-3">
                        <label for="edit-name" class="form-label">Nome Completo <span class="text-danger">*</span></label>
                        <input type="text" class="form-control" id="edit-name" name="name" required>
                    </div>

                    <div class="mb-3">
                        <label for="edit-email" class="form-label">Email <span class="text-danger">*</span></label>
                        <input type="email" class="form-control" id="edit-email" name="email" required>
                    </div>

                    <div class="mb-3">
                        <label for="edit-user_type" class="form-label">Tipo de Usuário</label>
                        <select class="form-select" id="edit-user_type" name="user_type">
                            <option value="user">Usuário Comum</option>
                            <option value="admin">Administrador</option>
                        </select>
                    </div>

                    <div class="mb-3">
                        <label for="edit-password" class="form-label">Nova Senha (opcional)</label>
                        <input type="password" class="form-control" id="edit-password" name="password" placeholder="Deixe em branco para manter a senha atual">
                    </div>
                </div>
                
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancelar</button>
                    <button type="submit" class="btn btn-primary">Salvar Alterações</button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Modal para excluir usuário -->
<div class="modal fade" id="deleteUserModal" tabindex="-1" aria-labelledby="deleteUserModalLabel" aria-hidden="true">
    <div class="modal-dialog">
        <div class="modal-content">
            <form method="POST" action="users.php">
                <input type="hidden" name="action" value="delete">
                <input type="hidden" name="id" id="delete-id">
                
                <div class="modal-header">
                    <h5 class="modal-title" id="deleteUserModalLabel">Excluir Usuário</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Fechar"></button>
                </div>
                
                <div class="modal-body">
                    <p>Tem certeza que deseja excluir o usuário <strong id="delete-username"></strong>?</p>
                    <div class="alert alert-warning">
                        <i class="fas fa-exclamation-triangle me-2"></i> Esta ação é irreversível e removerá todos os dados associados a este usuário.
                    </div>
                </div>
                
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancelar</button>
                    <button type="submit" class="btn btn-danger">Excluir</button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- JavaScript para manipulação dos modais -->
<script>
document.addEventListener("DOMContentLoaded", function() {
    // Manipular clique no botão de editar
    document.querySelectorAll(".edit-user").forEach(function(button) {
        button.addEventListener("click", function() {
            const id = this.getAttribute("data-id");
            const username = this.getAttribute("data-username");
            const name = this.getAttribute("data-name");
            const email = this.getAttribute("data-email");
            const user_type = this.getAttribute("data-user_type");
            
            document.getElementById("edit-id").value = id;
            document.getElementById("edit-username").value = username;
            document.getElementById("edit-name").value = name;
            document.getElementById("edit-email").value = email;
            document.getElementById("edit-user_type").value = user_type;
            
            const editModal = new bootstrap.Modal(document.getElementById("editUserModal"));
            editModal.show();
        });
    });
    
    // Manipular clique no botão de excluir
    document.querySelectorAll(".delete-user").forEach(function(button) {
        button.addEventListener("click", function() {
            const id = this.getAttribute("data-id");
            const username = this.getAttribute("data-username");
            
            document.getElementById("delete-id").value = id;
            document.getElementById("delete-username").textContent = username;
            
            const deleteModal = new bootstrap.Modal(document.getElementById("deleteUserModal"));
            deleteModal.show();
        });
    });
});
</script>

<?php
// Incluir rodapé
include "includes/footer.php";
?>

