from flask import Blueprint, request, jsonify
from werkzeug.security import generate_password_hash, check_password_hash
from flask_jwt_extended import create_access_token, jwt_required, get_jwt_identity
import datetime
from ..models.user import User, db

auth_bp = Blueprint('auth', __name__)

@auth_bp.route('/login', methods=['POST'])
def login():
    """Autenticação de usuários"""
    if not request.is_json:
        return jsonify({"msg": "Requisição inválida, JSON esperado"}), 400

    username = request.json.get('username', None)
    password = request.json.get('password', None)

    if not username or not password:
        return jsonify({"msg": "Usuário e senha são obrigatórios"}), 400

    user = User.query.filter_by(username=username).first()

    if user and check_password_hash(user.password_hash, password):
        # Atualizar último login
        user.last_login = datetime.datetime.now()
        db.session.commit()

        # Criar token JWT
        access_token = create_access_token(
            identity={
                'id': user.id,
                'username': user.username,
                'user_type': user.user_type
            }
        )
        return jsonify({
            "msg": "Login realizado com sucesso",
            "token": access_token,
            "user": {
                "id": user.id,
                "username": user.username,
                "email": user.email,
                "full_name": user.full_name,
                "user_type": user.user_type
            }
        }), 200
    else:
        return jsonify({"msg": "Credenciais inválidas"}), 401

@auth_bp.route('/register', methods=['POST'])
@jwt_required()
def register():
    """Registro de novos usuários (apenas admin)"""
    current_user = get_jwt_identity()
    
    # Verificar se o usuário atual é admin
    if current_user.get('user_type') != 'admin':
        return jsonify({"msg": "Acesso negado. Apenas administradores podem registrar novos usuários"}), 403

    if not request.is_json:
        return jsonify({"msg": "Requisição inválida, JSON esperado"}), 400

    data = request.json
    username = data.get('username')
    email = data.get('email')
    password = data.get('password')
    full_name = data.get('full_name')
    user_type = data.get('user_type', 'user')

    if not username or not email or not password:
        return jsonify({"msg": "Usuário, email e senha são obrigatórios"}), 400

    # Verificar se usuário ou email já existem
    if User.query.filter_by(username=username).first():
        return jsonify({"msg": "Nome de usuário já existe"}), 400
    
    if User.query.filter_by(email=email).first():
        return jsonify({"msg": "Email já está em uso"}), 400

    # Criar novo usuário
    new_user = User(
        username=username,
        email=email,
        password_hash=generate_password_hash(password),
        full_name=full_name,
        user_type=user_type
    )

    db.session.add(new_user)
    db.session.commit()

    return jsonify({
        "msg": "Usuário registrado com sucesso",
        "user": {
            "id": new_user.id,
            "username": new_user.username,
            "email": new_user.email,
            "full_name": new_user.full_name,
            "user_type": new_user.user_type
        }
    }), 201

@auth_bp.route('/change-password', methods=['POST'])
@jwt_required()
def change_password():
    """Alteração de senha"""
    current_user = get_jwt_identity()
    user_id = current_user.get('id')

    if not request.is_json:
        return jsonify({"msg": "Requisição inválida, JSON esperado"}), 400

    data = request.json
    current_password = data.get('current_password')
    new_password = data.get('new_password')

    if not current_password or not new_password:
        return jsonify({"msg": "Senha atual e nova senha são obrigatórias"}), 400

    user = User.query.get(user_id)
    if not user:
        return jsonify({"msg": "Usuário não encontrado"}), 404

    if not check_password_hash(user.password_hash, current_password):
        return jsonify({"msg": "Senha atual incorreta"}), 401

    user.password_hash = generate_password_hash(new_password)
    db.session.commit()

    return jsonify({"msg": "Senha alterada com sucesso"}), 200

@auth_bp.route('/profile', methods=['GET'])
@jwt_required()
def get_profile():
    """Obter perfil do usuário atual"""
    current_user = get_jwt_identity()
    user_id = current_user.get('id')

    user = User.query.get(user_id)
    if not user:
        return jsonify({"msg": "Usuário não encontrado"}), 404

    return jsonify({
        "user": {
            "id": user.id,
            "username": user.username,
            "email": user.email,
            "full_name": user.full_name,
            "user_type": user.user_type,
            "created_at": user.created_at,
            "last_login": user.last_login
        }
    }), 200

@auth_bp.route('/profile', methods=['PUT'])
@jwt_required()
def update_profile():
    """Atualizar perfil do usuário atual"""
    current_user = get_jwt_identity()
    user_id = current_user.get('id')

    if not request.is_json:
        return jsonify({"msg": "Requisição inválida, JSON esperado"}), 400

    user = User.query.get(user_id)
    if not user:
        return jsonify({"msg": "Usuário não encontrado"}), 404

    data = request.json
    
    # Atualizar campos permitidos
    if 'email' in data:
        # Verificar se o email já está em uso por outro usuário
        existing_user = User.query.filter_by(email=data['email']).first()
        if existing_user and existing_user.id != user_id:
            return jsonify({"msg": "Email já está em uso por outro usuário"}), 400
        user.email = data['email']
    
    if 'full_name' in data:
        user.full_name = data['full_name']

    db.session.commit()

    return jsonify({
        "msg": "Perfil atualizado com sucesso",
        "user": {
            "id": user.id,
            "username": user.username,
            "email": user.email,
            "full_name": user.full_name,
            "user_type": user.user_type
        }
    }), 200
