<?php
/**
 * Controlador de Autenticação
 * Sistema de Gerenciamento Eletrônico de Documentos (GED)
 */

// Iniciar sessão se ainda não estiver iniciada
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

// Incluir arquivos necessários
require_once __DIR__ . '/../config/database.php';
require_once __DIR__ . '/../models/user.php';

class AuthController {
    private $db;
    private $user;
    
    /**
     * Construtor
     */
    public function __construct() {
        $database = new Database();
        $this->db = $database->getConnection();
        $this->user = new User($this->db);
    }
    
    /**
     * Processa o login do usuário
     * @param string $username Nome de usuário
     * @param string $password Senha
     * @return array Resultado do login (sucesso/erro)
     */
    public function login($username, $password) {
        // Verificar se os campos estão preenchidos
        if (empty($username) || empty($password)) {
            return [
                'success' => false,
                'message' => 'Usuário e senha são obrigatórios'
            ];
        }
        
        // Buscar usuário pelo nome de usuário
        if (!$this->user->readByUsername($username)) {
            return [
                'success' => false,
                'message' => 'Usuário ou senha inválidos'
            ];
        }
        
        // Verificar se o usuário está ativo
        if (!$this->user->is_active) {
            return [
                'success' => false,
                'message' => 'Conta desativada. Entre em contato com o administrador'
            ];
        }
        
        // Verificar senha
        if (!$this->user->verifyPassword($password)) {
            return [
                'success' => false,
                'message' => 'Usuário ou senha inválidos'
            ];
        }
        
        // Atualizar último login
        $this->user->updateLastLogin();
        
        // Obter permissões do usuário
        $permissions = $this->user->getPermissions();
        
        // Criar sessão
        $_SESSION['user_id'] = $this->user->id;
        $_SESSION['username'] = $this->user->username;
        $_SESSION['user_type'] = $this->user->user_type;
        $_SESSION['full_name'] = $this->user->full_name;
        $_SESSION['permissions'] = array_map(function($p) { return $p['name']; }, $permissions);
        $_SESSION['login_time'] = time();
        
        // Regenerar ID de sessão para segurança
        session_regenerate_id(true);
        
        return [
            'success' => true,
            'message' => 'Login realizado com sucesso',
            'user' => [
                'id' => $this->user->id,
                'username' => $this->user->username,
                'email' => $this->user->email,
                'full_name' => $this->user->full_name,
                'user_type' => $this->user->user_type
            ]
        ];
    }
    
    /**
     * Processa o logout do usuário
     * @return array Resultado do logout
     */
    public function logout() {
        // Limpar todas as variáveis de sessão
        $_SESSION = array();
        
        // Destruir o cookie da sessão
        if (ini_get("session.use_cookies")) {
            $params = session_get_cookie_params();
            setcookie(session_name(), '', time() - 42000,
                $params["path"], $params["domain"],
                $params["secure"], $params["httponly"]
            );
        }
        
        // Destruir a sessão
        session_destroy();
        
        return [
            'success' => true,
            'message' => 'Logout realizado com sucesso'
        ];
    }
    
    /**
     * Verifica se o usuário está autenticado
     * @return bool Verdadeiro se autenticado, falso caso contrário
     */
    public function isAuthenticated() {
        return isset($_SESSION['user_id']);
    }
    
    /**
     * Verifica se o usuário tem uma permissão específica
     * @param string $permission Nome da permissão
     * @return bool Verdadeiro se tem permissão, falso caso contrário
     */
    public function hasPermission($permission) {
        // Se não estiver autenticado, não tem permissão
        if (!$this->isAuthenticated()) {
            return false;
        }
        
        // Administradores têm todas as permissões
        if ($_SESSION['user_type'] === 'admin') {
            return true;
        }
        
        // Verificar na lista de permissões da sessão
        return in_array($permission, $_SESSION['permissions']);
    }
    
    /**
     * Obtém o usuário atual
     * @return User|null Objeto do usuário ou null se não autenticado
     */
    public function getCurrentUser() {
        if (!$this->isAuthenticated()) {
            return null;
        }
        
        $this->user->readOne($_SESSION['user_id']);
        return $this->user;
    }
    
    /**
     * Registra um novo usuário (apenas para administradores)
     * @param array $data Dados do usuário
     * @return array Resultado do registro
     */
    public function register($data) {
        // Verificar se o usuário atual é administrador
        if (!$this->isAuthenticated() || $_SESSION['user_type'] !== 'admin') {
            return [
                'success' => false,
                'message' => 'Acesso negado. Apenas administradores podem registrar novos usuários'
            ];
        }
        
        // Verificar campos obrigatórios
        if (empty($data['username']) || empty($data['email']) || empty($data['password'])) {
            return [
                'success' => false,
                'message' => 'Usuário, email e senha são obrigatórios'
            ];
        }
        
        // Verificar se o usuário já existe
        $temp_user = new User($this->db);
        if ($temp_user->readByUsername($data['username'])) {
            return [
                'success' => false,
                'message' => 'Nome de usuário já existe'
            ];
        }
        
        // Criar novo usuário
        $this->user->username = $data['username'];
        $this->user->email = $data['email'];
        $this->user->password_hash = password_hash($data['password'], PASSWORD_BCRYPT);
        $this->user->full_name = $data['full_name'] ?? '';
        $this->user->user_type = $data['user_type'] ?? 'user';
        $this->user->is_active = true;
        
        if ($this->user->create()) {
            return [
                'success' => true,
                'message' => 'Usuário registrado com sucesso',
                'user_id' => $this->user->id
            ];
        } else {
            return [
                'success' => false,
                'message' => 'Erro ao registrar usuário'
            ];
        }
    }
    
    /**
     * Altera a senha do usuário atual
     * @param string $current_password Senha atual
     * @param string $new_password Nova senha
     * @return array Resultado da alteração
     */
    public function changePassword($current_password, $new_password) {
        // Verificar se está autenticado
        if (!$this->isAuthenticated()) {
            return [
                'success' => false,
                'message' => 'Usuário não autenticado'
            ];
        }
        
        // Obter usuário atual
        $this->user->readOne($_SESSION['user_id']);
        
        // Verificar senha atual
        if (!$this->user->verifyPassword($current_password)) {
            return [
                'success' => false,
                'message' => 'Senha atual incorreta'
            ];
        }
        
        // Atualizar senha
        if ($this->user->updatePassword($new_password)) {
            return [
                'success' => true,
                'message' => 'Senha alterada com sucesso'
            ];
        } else {
            return [
                'success' => false,
                'message' => 'Erro ao alterar senha'
            ];
        }
    }
}
?>
