<?php
/**
 * Controlador de Usuários
 * Sistema de Gerenciamento Eletrônico de Documentos (GED)
 */

require_once __DIR__ . "/../config/database.php";
require_once __DIR__ . "/../models/user.php";
require_once __DIR__ . "/../models/user_permission.php";
require_once __DIR__ . "/../models/permission.php"; // Adicionado
require_once __DIR__ . "/../models/audit_log.php";

class UserController {
    private $db;
    private $user;
    private $user_permission;
    private $permission; // Adicionado

    public function __construct() {
        $database = new Database();
        $this->db = $database->getConnection();
        $this->user = new User($this->db);
        $this->user_permission = new UserPermission($this->db);
        $this->permission = new Permission($this->db); // Instanciado
    }

    public function createUser($data) {
        if (empty($data["username"]) || empty($data["email"]) || empty($data["password"]) || empty($data["full_name"]) || empty($data["user_type"])) {
            return ["success" => false, "message" => "Todos os campos obrigatórios devem ser preenchidos."];
        }

        $this->user->username = $data["username"];
        if ($this->user->readByUsername($this->user->username)) {
            return ["success" => false, "message" => "Nome de usuário já existe."];
        }

        $this->user->email = $data["email"];
        $this->user->password_hash = password_hash($data["password"], PASSWORD_BCRYPT);
        $this->user->full_name = $data["full_name"];
        $this->user->user_type = $data["user_type"];
        $this->user->is_active = isset($data["is_active"]) ? 1 : 0;

        if ($this->user->create()) {
            // Atribuir permissões padrão ou específicas
            if (isset($data["permissions"]) && is_array($data["permissions"])) {
                foreach ($data["permissions"] as $permission_name) {
                    $this->permission->name = $permission_name;
                    $permission_id = $this->permission->getIdByName();
                    if ($permission_id) {
                        $this->user_permission->user_id = $this->user->id;
                        $this->user_permission->permission_id = $permission_id;
                        $this->user_permission->create();
                    }
                }
            }
            $user_id_for_audit = isset($_SESSION["user_id"]) ? $_SESSION["user_id"] : (defined('TEST_USER_ID') ? TEST_USER_ID : 1);
            logAudit($user_id_for_audit, "create", "user", $this->user->id, "Usuário criado: " . $this->user->username);
            return ["success" => true, "message" => "Usuário criado com sucesso.", "user_id" => $this->user->id];
        } else {
            return ["success" => false, "message" => "Erro ao criar usuário."];
        }
    }

    public function getUser($id) {
        if ($this->user->readOne($id)) {
            $permissions = [];
            $this->user_permission->user_id = $id;
            $stmt = $this->user_permission->readByUserId();
            while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
                $permissions[] = $row["permission_name"];
            }
            return ["success" => true, "user" => [
                "id" => $this->user->id,
                "username" => $this->user->username,
                "email" => $this->user->email,
                "full_name" => $this->user->full_name,
                "user_type" => $this->user->user_type,
                "is_active" => $this->user->is_active,
                "created_at" => $this->user->created_at,
                "last_login" => $this->user->last_login,
                "permissions" => $permissions
            ]];
        } else {
            return ["success" => false, "message" => "Usuário não encontrado."];
        }
    }

    public function getAllUsers() {
        $stmt = $this->user->readAll();
        $users = [];
        while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
            $users[] = $row;
        }
        return $users;
    }

    public function updateUser($data) {
        if (empty($data["id"]) || empty($data["email"]) || empty($data["full_name"]) || empty($data["user_type"])) {
            return ["success" => false, "message" => "Todos os campos obrigatórios devem ser preenchidos."];
        }

        $this->user->id = $data["id"];
        if (!$this->user->readOne($this->user->id)) {
            return ["success" => false, "message" => "Usuário não encontrado."];
        }

        $this->user->email = $data["email"];
        $this->user->full_name = $data["full_name"];
        $this->user->user_type = $data["user_type"];
        $this->user->is_active = isset($data["is_active"]) ? 1 : 0;

        if ($this->user->update()) {
            // Atualizar permissões
            $this->user_permission->user_id = $data["id"];
            $this->user_permission->deleteAllByUserId();
            if (isset($data["permissions"]) && is_array($data["permissions"])) {
                foreach ($data["permissions"] as $permission_name) {
                    $this->permission->name = $permission_name;
                    $permission_id = $this->permission->getIdByName();
                    if ($permission_id) {
                        $this->user_permission->user_id = $data["id"];
                        $this->user_permission->permission_id = $permission_id;
                        $this->user_permission->create();
                    }
                }
            }
            $user_id_for_audit = isset($_SESSION["user_id"]) ? $_SESSION["user_id"] : (defined('TEST_USER_ID') ? TEST_USER_ID : 1);
            logAudit($user_id_for_audit, "update", "user", $data["id"], "Usuário atualizado: " . $this->user->username);
            return ["success" => true, "message" => "Usuário atualizado com sucesso."];
        } else {
            return ["success" => false, "message" => "Erro ao atualizar usuário."];
        }
    }

    public function deleteUser($id) {
        $this->user->id = $id;
        if (!$this->user->readOne($this->user->id)) {
            return ["success" => false, "message" => "Usuário não encontrado."];
        }
        $username = $this->user->username;

        // Excluir permissões do usuário
        $this->user_permission->user_id = $id;
        $this->user_permission->deleteAllByUserId();

        if ($this->user->delete()) {
            $user_id_for_audit = isset($_SESSION["user_id"]) ? $_SESSION["user_id"] : (defined('TEST_USER_ID') ? TEST_USER_ID : 1);
            logAudit($user_id_for_audit, "delete", "user", $id, "Usuário excluído: " . $username);
            return ["success" => true, "message" => "Usuário excluído com sucesso."];
        } else {
            return ["success" => false, "message" => "Erro ao excluir usuário."];
        }
    }

    public function updatePassword($id, $new_password) {
        $this->user->id = $id;
        if (!$this->user->readOne($this->user->id)) {
            return ["success" => false, "message" => "Usuário não encontrado."];
        }

        if ($this->user->updatePassword($new_password)) {
            $user_id_for_audit = isset($_SESSION["user_id"]) ? $_SESSION["user_id"] : (defined('TEST_USER_ID') ? TEST_USER_ID : 1);
            logAudit($user_id_for_audit, "update_password", "user", $id, "Senha do usuário " . $this->user->username . " atualizada.");
            return ["success" => true, "message" => "Senha atualizada com sucesso."];
        } else {
            return ["success" => false, "message" => "Erro ao atualizar senha."];
        }
    }
}
?>

