<?php
/**
 * Página de gerenciamento de permissões
 * Sistema de Gerenciamento Eletrônico de Documentos (GED)
 */

// Iniciar sessão e verificar autenticação
session_start();
if (!isset($_SESSION["user_id"])) {
    header("Location: login.php");
    exit;
}

// Verificar se o usuário tem permissão de administrador
if ($_SESSION["user_type"] != "admin") {
    $_SESSION["flash_message"] = "Acesso negado. Você não tem permissão para acessar esta página.";
    $_SESSION["flash_type"] = "danger";
    header("Location: index.php");
    exit;
}

// Incluir arquivos necessários
require_once "config/database.php";
require_once "models/permission.php";
require_once "controllers/permission_controller.php";

// Conectar ao banco de dados
$database = new Database();
$db = $database->getConnection();

// Instanciar controlador
$permission_controller = new PermissionController();

// Processar ações
if ($_SERVER["REQUEST_METHOD"] == "POST") {
    if (isset($_POST["action"])) {
        // Adicionar permissão
        if ($_POST["action"] == "add") {
            $name = $_POST["name"] ?? "";
            $description = $_POST["description"] ?? "";
            
            if (!empty($name) && !empty($description)) {
                $result = $permission_controller->createPermission([
                    "name" => $name,
                    "description" => $description
                ]);
                
                if ($result["success"]) {
                    $_SESSION["flash_message"] = "Permissão adicionada com sucesso.";
                    $_SESSION["flash_type"] = "success";
                } else {
                    $_SESSION["flash_message"] = $result["message"];
                    $_SESSION["flash_type"] = "danger";
                }
            } else {
                $_SESSION["flash_message"] = "Por favor, preencha todos os campos obrigatórios.";
                $_SESSION["flash_type"] = "danger";
            }
        }
        
        // Editar permissão
        else if ($_POST["action"] == "edit") {
            $id = $_POST["id"] ?? "";
            $name = $_POST["name"] ?? "";
            $description = $_POST["description"] ?? "";
            
            if (!empty($id) && !empty($name) && !empty($description)) {
                $result = $permission_controller->updatePermission([
                    "id" => $id,
                    "name" => $name,
                    "description" => $description
                ]);
                
                if ($result["success"]) {
                    $_SESSION["flash_message"] = "Permissão atualizada com sucesso.";
                    $_SESSION["flash_type"] = "success";
                } else {
                    $_SESSION["flash_message"] = $result["message"];
                    $_SESSION["flash_type"] = "danger";
                }
            } else {
                $_SESSION["flash_message"] = "Por favor, preencha todos os campos obrigatórios.";
                $_SESSION["flash_type"] = "danger";
            }
        }
        
        // Excluir permissão
        else if ($_POST["action"] == "delete") {
            $id = $_POST["id"] ?? "";
            
            if (!empty($id)) {
                $result = $permission_controller->deletePermission($id);
                
                if ($result["success"]) {
                    $_SESSION["flash_message"] = "Permissão excluída com sucesso.";
                    $_SESSION["flash_type"] = "success";
                } else {
                    $_SESSION["flash_message"] = $result["message"];
                    $_SESSION["flash_type"] = "danger";
                }
            } else {
                $_SESSION["flash_message"] = "ID da permissão é obrigatório.";
                $_SESSION["flash_type"] = "danger";
            }
        }
        
        // Redirecionar para evitar reenvio do formulário
        header("Location: permissions.php");
        exit;
    }
}

// Obter todas as permissões
$permissions = $permission_controller->getAllPermissions();

// Definir título da página
$page_title = "Gerenciar Permissões";

// Incluir cabeçalho
include "includes/header.php";
?>

<div class="row mb-4">
    <div class="col-md-6">
        <h1><i class="fas fa-shield-alt me-2"></i> Gerenciar Permissões</h1>
    </div>
    <div class="col-md-6 text-md-end">
        <button type="button" class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#addPermissionModal">
            <i class="fas fa-plus me-1"></i> Nova Permissão
        </button>
    </div>
</div>

<!-- Lista de permissões -->
<div class="card">
    <div class="card-body">
        <?php if (count($permissions) > 0): ?>
        <div class="table-responsive">
            <table class="table table-hover datatable">
                <thead>
                    <tr>
                        <th>ID</th>
                        <th>Nome</th>
                        <th>Descrição</th>
                        <th>Ações</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($permissions as $p): ?>
                    <tr>
                        <td><?php echo $p["id"]; ?></td>
                        <td><?php echo htmlspecialchars($p["name"]); ?></td>
                        <td><?php echo htmlspecialchars($p["description"]); ?></td>
                        <td>
                            <div class="btn-group btn-group-sm">
                                <button type="button" class="btn btn-primary edit-permission" 
                                        data-id="<?php echo $p["id"]; ?>"
                                        data-name="<?php echo htmlspecialchars($p["name"]); ?>"
                                        data-description="<?php echo htmlspecialchars($p["description"]); ?>">
                                    <i class="fas fa-edit"></i>
                                </button>
                                <button type="button" class="btn btn-danger delete-permission" 
                                        data-id="<?php echo $p["id"]; ?>"
                                        data-name="<?php echo htmlspecialchars($p["name"]); ?>">
                                    <i class="fas fa-trash"></i>
                                </button>
                            </div>
                        </td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
        <?php else: ?>
        <div class="alert alert-info">
            <i class="fas fa-info-circle me-2"></i> Nenhuma permissão encontrada.
        </div>
        <?php endif; ?>
    </div>
</div>

<!-- Modal para adicionar permissão -->
<div class="modal fade" id="addPermissionModal" tabindex="-1" aria-labelledby="addPermissionModalLabel" aria-hidden="true">
    <div class="modal-dialog">
        <div class="modal-content">
            <form method="POST" action="permissions.php">
                <input type="hidden" name="action" value="add">
                
                <div class="modal-header">
                    <h5 class="modal-title" id="addPermissionModalLabel">Nova Permissão</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Fechar"></button>
                </div>
                
                <div class="modal-body">
                    <div class="mb-3">
                        <label for="add-name" class="form-label">Nome da Permissão <span class="text-danger">*</span></label>
                        <input type="text" class="form-control" id="add-name" name="name" required>
                    </div>
                    
                    <div class="mb-3">
                        <label for="add-description" class="form-label">Descrição <span class="text-danger">*</span></label>
                        <textarea class="form-control" id="add-description" name="description" rows="3" required></textarea>
                    </div>
                </div>
                
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancelar</button>
                    <button type="submit" class="btn btn-primary">Adicionar</button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Modal para editar permissão -->
<div class="modal fade" id="editPermissionModal" tabindex="-1" aria-labelledby="editPermissionModalLabel" aria-hidden="true">
    <div class="modal-dialog">
        <div class="modal-content">
            <form method="POST" action="permissions.php">
                <input type="hidden" name="action" value="edit">
                <input type="hidden" name="id" id="edit-id">
                
                <div class="modal-header">
                    <h5 class="modal-title" id="editPermissionModalLabel">Editar Permissão</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Fechar"></button>
                </div>
                
                <div class="modal-body">
                    <div class="mb-3">
                        <label for="edit-name" class="form-label">Nome da Permissão <span class="text-danger">*</span></label>
                        <input type="text" class="form-control" id="edit-name" name="name" required>
                    </div>
                    
                    <div class="mb-3">
                        <label for="edit-description" class="form-label">Descrição <span class="text-danger">*</span></label>
                        <textarea class="form-control" id="edit-description" name="description" rows="3" required></textarea>
                    </div>
                </div>
                
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancelar</button>
                    <button type="submit" class="btn btn-primary">Salvar Alterações</button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Modal para excluir permissão -->
<div class="modal fade" id="deletePermissionModal" tabindex="-1" aria-labelledby="deletePermissionModalLabel" aria-hidden="true">
    <div class="modal-dialog">
        <div class="modal-content">
            <form method="POST" action="permissions.php">
                <input type="hidden" name="action" value="delete">
                <input type="hidden" name="id" id="delete-id">
                
                <div class="modal-header">
                    <h5 class="modal-title" id="deletePermissionModalLabel">Excluir Permissão</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Fechar"></button>
                </div>
                
                <div class="modal-body">
                    <p>Tem certeza que deseja excluir a permissão <strong id="delete-name"></strong>?</p>
                    <div class="alert alert-warning">
                        <i class="fas fa-exclamation-triangle me-2"></i> Esta ação é irreversível e removerá a permissão de todos os usuários associados.
                    </div>
                </div>
                
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancelar</button>
                    <button type="submit" class="btn btn-danger">Excluir</button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- JavaScript para manipulação dos modais -->
<script>
document.addEventListener("DOMContentLoaded", function() {
    // Manipular clique no botão de editar
    document.querySelectorAll(".edit-permission").forEach(function(button) {
        button.addEventListener("click", function() {
            const id = this.getAttribute("data-id");
            const name = this.getAttribute("data-name");
            const description = this.getAttribute("data-description");
            
            document.getElementById("edit-id").value = id;
            document.getElementById("edit-name").value = name;
            document.getElementById("edit-description").value = description;
            
            var editModal = new bootstrap.Modal(document.getElementById("editPermissionModal"));
            editModal.show();
        });
    });

    // Manipular clique no botão de excluir
    document.querySelectorAll(".delete-permission").forEach(function(button) {
        button.addEventListener("click", function() {
            const id = this.getAttribute("data-id");
            const name = this.getAttribute("data-name");
            
            document.getElementById("delete-id").value = id;
            document.getElementById("delete-name").textContent = name;
            
            var deleteModal = new bootstrap.Modal(document.getElementById("deletePermissionModal"));
            deleteModal.show();
        });
    });
});
</script>

<?php
// Incluir rodapé
include "includes/footer.php";
?>

