<?php
/**
 * Página de logs de auditoria
 * Sistema de Gerenciamento Eletrônico de Documentos (GED)
 */

// Iniciar sessão e verificar autenticação
session_start();
if (!isset($_SESSION['user_id'])) {
    header('Location: login.php');
    exit;
}

// Verificar se o usuário tem permissão de administrador
if ($_SESSION['user_type'] != 'admin') {
    $_SESSION['flash_message'] = "Acesso negado. Você não tem permissão para acessar esta página.";
    $_SESSION['flash_type'] = "danger";
    header('Location: index.php');
    exit;
}

// Incluir arquivos necessários
require_once 'config/database.php';
require_once 'models/audit_log.php';
require_once 'controllers/audit_log_controller.php';

// Conectar ao banco de dados
$database = new Database();
$db = $database->getConnection();

// Instanciar controlador
$audit_log_controller = new AuditLogController();

// Definir filtros e paginação
$page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
$limit = isset($_GET['limit']) ? (int)$_GET['limit'] : 50;
$user_id = isset($_GET['user_id']) ? $_GET['user_id'] : null;
$action_type = isset($_GET['action_type']) ? $_GET['action_type'] : null;
$entity_type = isset($_GET['entity_type']) ? $_GET['entity_type'] : null;
$date_from = isset($_GET['date_from']) ? $_GET['date_from'] : null;
$date_to = isset($_GET['date_to']) ? $_GET['date_to'] : null;

// Obter logs de auditoria
$result = $audit_log_controller->getLogs([
    'page' => $page,
    'limit' => $limit,
    'user_id' => $user_id,
    'action_type' => $action_type,
    'entity_type' => $entity_type,
    'date_from' => $date_from,
    'date_to' => $date_to
]);

$logs = $result['logs'];
$total_logs = $result['total'];
$total_pages = ceil($total_logs / $limit);

// Obter usuários para filtro
$users = $audit_log_controller->getUsers();

// Definir título da página
$page_title = "Logs de Auditoria";

// Incluir cabeçalho
include 'includes/header.php';
?>

<div class="row mb-4">
    <div class="col-md-6">
        <h1><i class="fas fa-history me-2"></i> Logs de Auditoria</h1>
    </div>
    <div class="col-md-6 text-md-end">
        <button type="button" class="btn btn-primary" data-bs-toggle="collapse" data-bs-target="#filterCollapse">
            <i class="fas fa-filter me-1"></i> Filtros
        </button>
        <a href="audit_logs_export.php" class="btn btn-success">
            <i class="fas fa-file-excel me-1"></i> Exportar
        </a>
    </div>
</div>

<!-- Filtros -->
<div class="collapse mb-4" id="filterCollapse">
    <div class="card">
        <div class="card-body">
            <form method="GET" action="audit_logs.php" class="row g-3">
                <div class="col-md-4">
                    <label for="user_id" class="form-label">Usuário</label>
                    <select class="form-select select2" id="user_id" name="user_id">
                        <option value="">Todos os usuários</option>
                        <?php foreach ($users as $user): ?>
                        <option value="<?php echo $user['id']; ?>" <?php echo ($user_id == $user['id']) ? 'selected' : ''; ?>>
                            <?php echo htmlspecialchars($user['username']); ?> (<?php echo htmlspecialchars($user['name']); ?>)
                        </option>
                        <?php endforeach; ?>
                    </select>
                </div>
                
                <div class="col-md-4">
                    <label for="action_type" class="form-label">Tipo de Ação</label>
                    <select class="form-select" id="action_type" name="action_type">
                        <option value="">Todas as ações</option>
                        <option value="login" <?php echo ($action_type == 'login') ? 'selected' : ''; ?>>Login</option>
                        <option value="logout" <?php echo ($action_type == 'logout') ? 'selected' : ''; ?>>Logout</option>
                        <option value="create" <?php echo ($action_type == 'create') ? 'selected' : ''; ?>>Criação</option>
                        <option value="update" <?php echo ($action_type == 'update') ? 'selected' : ''; ?>>Atualização</option>
                        <option value="delete" <?php echo ($action_type == 'delete') ? 'selected' : ''; ?>>Exclusão</option>
                        <option value="view" <?php echo ($action_type == 'view') ? 'selected' : ''; ?>>Visualização</option>
                        <option value="download" <?php echo ($action_type == 'download') ? 'selected' : ''; ?>>Download</option>
                        <option value="approve" <?php echo ($action_type == 'approve') ? 'selected' : ''; ?>>Aprovação</option>
                        <option value="reject" <?php echo ($action_type == 'reject') ? 'selected' : ''; ?>>Rejeição</option>
                    </select>
                </div>
                
                <div class="col-md-4">
                    <label for="entity_type" class="form-label">Tipo de Entidade</label>
                    <select class="form-select" id="entity_type" name="entity_type">
                        <option value="">Todas as entidades</option>
                        <option value="user" <?php echo ($entity_type == 'user') ? 'selected' : ''; ?>>Usuário</option>
                        <option value="document" <?php echo ($entity_type == 'document') ? 'selected' : ''; ?>>Documento</option>
                        <option value="category" <?php echo ($entity_type == 'category') ? 'selected' : ''; ?>>Categoria</option>
                        <option value="workflow" <?php echo ($entity_type == 'workflow') ? 'selected' : ''; ?>>Workflow</option>
                        <option value="system" <?php echo ($entity_type == 'system') ? 'selected' : ''; ?>>Sistema</option>
                    </select>
                </div>
                
                <div class="col-md-4">
                    <label for="date_from" class="form-label">Data Inicial</label>
                    <input type="date" class="form-control" id="date_from" name="date_from" value="<?php echo $date_from; ?>">
                </div>
                
                <div class="col-md-4">
                    <label for="date_to" class="form-label">Data Final</label>
                    <input type="date" class="form-control" id="date_to" name="date_to" value="<?php echo $date_to; ?>">
                </div>
                
                <div class="col-md-4">
                    <label for="limit" class="form-label">Registros por página</label>
                    <select class="form-select" id="limit" name="limit">
                        <option value="25" <?php echo ($limit == 25) ? 'selected' : ''; ?>>25</option>
                        <option value="50" <?php echo ($limit == 50) ? 'selected' : ''; ?>>50</option>
                        <option value="100" <?php echo ($limit == 100) ? 'selected' : ''; ?>>100</option>
                        <option value="250" <?php echo ($limit == 250) ? 'selected' : ''; ?>>250</option>
                    </select>
                </div>
                
                <div class="col-12">
                    <button type="submit" class="btn btn-primary">
                        <i class="fas fa-search me-1"></i> Filtrar
                    </button>
                    <a href="audit_logs.php" class="btn btn-secondary">
                        <i class="fas fa-times me-1"></i> Limpar Filtros
                    </a>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Resumo dos filtros aplicados -->
<?php if ($user_id || $action_type || $entity_type || $date_from || $date_to): ?>
<div class="alert alert-info mb-4">
    <h5><i class="fas fa-filter me-2"></i> Filtros aplicados</h5>
    <div class="row">
        <?php if ($user_id): ?>
        <div class="col-md-3">
            <strong>Usuário:</strong> 
            <?php 
            foreach ($users as $user) {
                if ($user['id'] == $user_id) {
                    echo htmlspecialchars($user['username']);
                    break;
                }
            }
            ?>
        </div>
        <?php endif; ?>
        
        <?php if ($action_type): ?>
        <div class="col-md-3">
            <strong>Tipo de Ação:</strong> 
            <?php 
            $action_types = [
                'login' => 'Login',
                'logout' => 'Logout',
                'create' => 'Criação',
                'update' => 'Atualização',
                'delete' => 'Exclusão',
                'view' => 'Visualização',
                'download' => 'Download',
                'approve' => 'Aprovação',
                'reject' => 'Rejeição'
            ];
            echo $action_types[$action_type] ?? $action_type;
            ?>
        </div>
        <?php endif; ?>
        
        <?php if ($entity_type): ?>
        <div class="col-md-3">
            <strong>Tipo de Entidade:</strong> 
            <?php 
            $entity_types = [
                'user' => 'Usuário',
                'document' => 'Documento',
                'category' => 'Categoria',
                'workflow' => 'Workflow',
                'system' => 'Sistema'
            ];
            echo $entity_types[$entity_type] ?? $entity_type;
            ?>
        </div>
        <?php endif; ?>
        
        <?php if ($date_from || $date_to): ?>
        <div class="col-md-3">
            <strong>Período:</strong> 
            <?php 
            if ($date_from && $date_to) {
                echo date('d/m/Y', strtotime($date_from)) . ' até ' . date('d/m/Y', strtotime($date_to));
            } else if ($date_from) {
                echo 'A partir de ' . date('d/m/Y', strtotime($date_from));
            } else {
                echo 'Até ' . date('d/m/Y', strtotime($date_to));
            }
            ?>
        </div>
        <?php endif; ?>
    </div>
</div>
<?php endif; ?>

<!-- Lista de logs -->
<div class="card">
    <div class="card-body">
        <?php if (count($logs) > 0): ?>
        <div class="table-responsive">
            <table class="table table-hover">
                <thead>
                    <tr>
                        <th>ID</th>
                        <th>Data/Hora</th>
                        <th>Usuário</th>
                        <th>Ação</th>
                        <th>Entidade</th>
                        <th>ID Entidade</th>
                        <th>Detalhes</th>
                        <th>IP</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($logs as $log): ?>
                    <tr>
                        <td><?php echo $log['id']; ?></td>
                        <td><?php echo date('d/m/Y H:i:s', strtotime($log['created_at'])); ?></td>
                        <td><?php echo htmlspecialchars($log['username']); ?></td>
                        <td>
                            <?php 
                            $action_badges = [
                                'login' => '<span class="badge bg-success">Login</span>',
                                'logout' => '<span class="badge bg-secondary">Logout</span>',
                                'create' => '<span class="badge bg-primary">Criação</span>',
                                'update' => '<span class="badge bg-info text-dark">Atualização</span>',
                                'delete' => '<span class="badge bg-danger">Exclusão</span>',
                                'view' => '<span class="badge bg-light text-dark">Visualização</span>',
                                'download' => '<span class="badge bg-warning text-dark">Download</span>',
                                'approve' => '<span class="badge bg-success">Aprovação</span>',
                                'reject' => '<span class="badge bg-danger">Rejeição</span>'
                            ];
                            echo $action_badges[$log['action_type']] ?? $log['action_type'];
                            ?>
                        </td>
                        <td>
                            <?php 
                            $entity_badges = [
                                'user' => '<span class="badge bg-primary">Usuário</span>',
                                'document' => '<span class="badge bg-info text-dark">Documento</span>',
                                'category' => '<span class="badge bg-success">Categoria</span>',
                                'workflow' => '<span class="badge bg-warning text-dark">Workflow</span>',
                                'system' => '<span class="badge bg-secondary">Sistema</span>'
                            ];
                            echo $entity_badges[$log['entity_type']] ?? $log['entity_type'];
                            ?>
                        </td>
                        <td>
                            <?php if ($log['entity_id'] && $log['entity_type'] == 'document'): ?>
                            <a href="document_view.php?id=<?php echo $log['entity_id']; ?>" class="text-decoration-none">
                                <?php echo $log['entity_id']; ?>
                            </a>
                            <?php else: ?>
                            <?php echo $log['entity_id'] ?: '-'; ?>
                            <?php endif; ?>
                        </td>
                        <td>
                            <button type="button" class="btn btn-sm btn-outline-secondary view-details" 
                                    data-bs-toggle="tooltip" data-bs-placement="left" 
                                    title="<?php echo htmlspecialchars($log['details']); ?>">
                                <i class="fas fa-info-circle"></i>
                            </button>
                        </td>
                        <td><?php echo $log['ip_address']; ?></td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
        
        <!-- Paginação -->
        <?php if ($total_pages > 1): ?>
        <div class="d-flex justify-content-between align-items-center mt-3">
            <div>
                Mostrando <?php echo count($logs); ?> de <?php echo $total_logs; ?> registros
            </div>
            <nav aria-label="Navegação de página">
                <ul class="pagination">
                    <?php if ($page > 1): ?>
                    <li class="page-item">
                        <a class="page-link" href="?page=1<?php echo $user_id ? "&user_id=$user_id" : ''; ?><?php echo $action_type ? "&action_type=$action_type" : ''; ?><?php echo $entity_type ? "&entity_type=$entity_type" : ''; ?><?php echo $date_from ? "&date_from=$date_from" : ''; ?><?php echo $date_to ? "&date_to=$date_to" : ''; ?><?php echo $limit ? "&limit=$limit" : ''; ?>">
                            <i class="fas fa-angle-double-left"></i>
                        </a>
                    </li>
                    <li class="page-item">
                        <a class="page-link" href="?page=<?php echo $page - 1; ?><?php echo $user_id ? "&user_id=$user_id" : ''; ?><?php echo $action_type ? "&action_type=$action_type" : ''; ?><?php echo $entity_type ? "&entity_type=$entity_type" : ''; ?><?php echo $date_from ? "&date_from=$date_from" : ''; ?><?php echo $date_to ? "&date_to=$date_to" : ''; ?><?php echo $limit ? "&limit=$limit" : ''; ?>">
                            <i class="fas fa-angle-left"></i>
                        </a>
                    </li>
                    <?php endif; ?>
                    
                    <?php
                    $start_page = max(1, $page - 2);
                    $end_page = min($total_pages, $page + 2);
                    
                    for ($i = $start_page; $i <= $end_page; $i++):
                    ?>
                    <li class="page-item <?php echo ($i == $page) ? 'active' : ''; ?>">
                        <a class="page-link" href="?page=<?php echo $i; ?><?php echo $user_id ? "&user_id=$user_id" : ''; ?><?php echo $action_type ? "&action_type=$action_type" : ''; ?><?php echo $entity_type ? "&entity_type=$entity_type" : ''; ?><?php echo $date_from ? "&date_from=$date_from" : ''; ?><?php echo $date_to ? "&date_to=$date_to" : ''; ?><?php echo $limit ? "&limit=$limit" : ''; ?>">
                            <?php echo $i; ?>
                        </a>
                    </li>
                    <?php endfor; ?>
                    
                    <?php if ($page < $total_pages): ?>
                    <li class="page-item">
                        <a class="page-link" href="?page=<?php echo $page + 1; ?><?php echo $user_id ? "&user_id=$user_id" : ''; ?><?php echo $action_type ? "&action_type=$action_type" : ''; ?><?php echo $entity_type ? "&entity_type=$entity_type" : ''; ?><?php echo $date_from ? "&date_from=$date_from" : ''; ?><?php echo $date_to ? "&date_to=$date_to" : ''; ?><?php echo $limit ? "&limit=$limit" : ''; ?>">
                            <i class="fas fa-angle-right"></i>
                        </a>
                    </li>
                    <li class="page-item">
                        <a class="page-link" href="?page=<?php echo $total_pages; ?><?php echo $user_id ? "&user_id=$user_id" : ''; ?><?php echo $action_type ? "&action_type=$action_type" : ''; ?><?php echo $entity_type ? "&entity_type=$entity_type" : ''; ?><?php echo $date_from ? "&date_from=$date_from" : ''; ?><?php echo $date_to ? "&date_to=$date_to" : ''; ?><?php echo $limit ? "&limit=$limit" : ''; ?>">
                            <i class="fas fa-angle-double-right"></i>
                        </a>
                    </li>
                    <?php endif; ?>
                </ul>
            </nav>
        </div>
        <?php endif; ?>
        
        <?php else: ?>
        <div class="alert alert-info">
            <i class="fas fa-info-circle me-2"></i> Nenhum log de auditoria encontrado.
        </div>
        <?php endif; ?>
    </div>
</div>

<!-- JavaScript para tooltips -->
<script>
document.addEventListener('DOMContentLoaded', function() {
    // Inicializar tooltips
    var tooltipTriggerList = [].slice.call(document.querySelectorAll('[data-bs-toggle="tooltip"]'));
    var tooltipList = tooltipTriggerList.map(function (tooltipTriggerEl) {
        return new bootstrap.Tooltip(tooltipTriggerEl);
    });
    
    // Mostrar detalhes em modal quando clicado
    document.querySelectorAll('.view-details').forEach(function(button) {
        button.addEventListener('click', function() {
            const details = this.getAttribute('data-bs-original-title');
            
            // Criar modal dinamicamente
            const modalHtml = `
                <div class="modal fade" id="detailsModal" tabindex="-1" aria-hidden="true">
                    <div class="modal-dialog">
                        <div class="modal-content">
                            <div class="modal-header">
                                <h5 class="modal-title">Detalhes do Log</h5>
                                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Fechar"></button>
                            </div>
                            <div class="modal-body">
                                <p>${details}</p>
                            </div>
                            <div class="modal-footer">
                                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Fechar</button>
                            </div>
                        </div>
                    </div>
                </div>
            `;
            
            // Adicionar modal ao DOM
            const modalContainer = document.createElement('div');
            modalContainer.innerHTML = modalHtml;
            document.body.appendChild(modalContainer);
            
            // Mostrar modal
            const modal = new bootstrap.Modal(document.getElementById('detailsModal'));
            modal.show();
            
            // Remover modal do DOM quando fechado
            document.getElementById('detailsModal').addEventListener('hidden.bs.modal', function() {
                document.body.removeChild(modalContainer);
            });
        });
    });
});
</script>

<?php
// Incluir rodapé
include 'includes/footer.php';
?>
