<?php
/**
 * Controlador de Documentos
 * Sistema de Gerenciamento Eletrônico de Documentos (GED)
 */

// Incluir arquivos necessários
require_once __DIR__ . "/../config/database.php";
require_once __DIR__ . "/../models/document.php";
require_once __DIR__ . "/../models/category.php";
require_once __DIR__ . "/../models/document_version.php";
require_once __DIR__ . "/../models/audit_log.php";
require_once __DIR__ . "/../models/workflow.php";
require_once __DIR__ . "/../models/workflow_step.php";
require_once __DIR__ . "/../models/workflow_approval.php";

class DocumentController {
    private $db;
    private $document;
    private $document_version;
    private $workflow;
    private $workflow_step;
    private $workflow_approval;
    
    /**
     * Construtor
     */
    public function __construct() {
        $database = new Database();
        $this->db = $database->getConnection();
        $this->document = new Document($this->db);
        $this->document_version = new DocumentVersion($this->db);
        $this->workflow = new Workflow($this->db);
        $this->workflow_step = new WorkflowStep($this->db);
        $this->workflow_approval = new WorkflowApproval($this->db);
    }
    
    /**
     * Processa o upload de um novo documento
     * @param array $data Dados do documento
     * @return array Resultado do upload
     */
    public function uploadDocument($data) {
        // Verificar campos obrigatórios
        if (empty($data["title"])) {
            return [
                "success" => false,
                "message" => "O título do documento é obrigatório"
            ];
        }
        
        if (!isset($data["file"]) || empty($data["file"]["name"])) {
            return [
                "success" => false,
                "message" => "Nenhum arquivo foi selecionado"
            ];
        }
        
        $file = $data["file"];

        // Verificar se houve erro no upload
        if ($file["error"] !== UPLOAD_ERR_OK) {
            $error_message = $this->getUploadErrorMessage($file["error"]);
            return [
                "success" => false,
                "message" => "Erro no upload: " . $error_message
            ];
        }
        
        // Verificar tipo de arquivo permitido
        $allowed_extensions = explode(",", ALLOWED_EXTENSIONS); // Usar constante do config.php
        $file_extension = strtolower(pathinfo($file["name"], PATHINFO_EXTENSION));
        
        if (!in_array($file_extension, $allowed_extensions)) {
            return [
                "success" => false,
                "message" => "Tipo de arquivo não permitido. Extensões permitidas: " . implode(", ", $allowed_extensions)
            ];
        }
        
        // Verificar tamanho máximo
        $max_size = MAX_FILE_SIZE;
        if ($file["size"] > $max_size) {
            return [
                "success" => false,
                "message" => "O arquivo excede o tamanho máximo permitido (" . MAX_UPLOAD_SIZE . "MB)"
            ];
        }
        
        // Criar diretório de uploads se não existir
        $upload_dir = UPLOAD_PATH . "/" . date("Y/m");
        if (!file_exists($upload_dir)) {
            mkdir($upload_dir, 0755, true);
        }
        
        // Gerar nome de arquivo único
        $unique_file_name = uniqid() . "_" . time() . "." . $file_extension;
        $file_path = $upload_dir . "/" . $unique_file_name;
        
        // Mover arquivo para o diretório de uploads
        if (!move_uploaded_file($file["tmp_name"], $file_path)) {
            return [
                "success" => false,
                "message" => "Erro ao mover o arquivo para o diretório de uploads"
            ];
        }
        
        // Salvar documento no banco de dados
        $this->document->title = $data["title"];
        $this->document->description = $data["description"] ?? null;
        $this->document->file_path = str_replace(BASE_PATH, "", $file_path); // Salvar caminho relativo
        $this->document->file_name = $file["name"];
        $this->document->file_type = $file["type"];
        $this->document->file_size = $file["size"];
        $this->document->category_id = $data["category_id"] ?? null;
        $this->document->status = $data["status"] ?? "draft";
        $this->document->created_by = $data["author_id"];
        $this->document->workflow_id = $data["workflow_id"] ?? null;
        
        if ($this->document->create()) {
            // Salvar palavras-chave
            if (!empty($data["keywords"])) {
                $keywords_array = array_map("trim", explode(",", $data["keywords"]));
                $this->document->setKeywords($this->document->id, $keywords_array);
            }
            
            // Criar primeira versão do documento
            $this->document_version->document_id = $this->document->id;
            $this->document_version->version_number = 1;
            $this->document_version->file_path = $this->document->file_path;
            $this->document_version->file_name = $this->document->file_name;
            $this->document_version->file_size = $this->document->file_size;
            $this->document_version->file_type = $this->document->file_type;
            $this->document_version->created_by = $this->document->created_by;
            $this->document_version->changes = "Versão inicial do documento";
            $this->document_version->create();

            // Iniciar processo de aprovação se um workflow foi selecionado
            if (!empty($data["workflow_id"])) {
                $this->workflow->readOne($data["workflow_id"]);
                $steps = $this->workflow->getSteps();
                if (!empty($steps)) {
                    $first_step = $steps[0];
                    $this->document->updateWorkflowStatus($this->document->id, $data["workflow_id"], $first_step["id"]);

                    $this->workflow_approval->document_id = $this->document->id;
                    $this->workflow_approval->workflow_step_id = $first_step["id"];
                    $this->workflow_approval->approver_id = $data["author_id"]; // Ou o primeiro aprovador definido no workflow
                    $this->workflow_approval->status = "pending";
                    $this->workflow_approval->comments = "Processo de aprovação iniciado.";
                    $this->workflow_approval->create();

                    $this->document->status = "pending";
                    $this->document->update();
                }
            }

            // Registrar log de auditoria
            logAudit($data["author_id"], "create", "document", $this->document->id, "Documento criado: " . $data["title"]);
            
            return [
                "success" => true,
                "message" => "Documento enviado com sucesso",
                "document_id" => $this->document->id
            ];
        } else {
            // Remover arquivo se falhar ao salvar no banco
            unlink($file_path);
            
            return [
                "success" => false,
                "message" => "Erro ao salvar o documento no banco de dados"
            ];
        }
    }
    
    /**
     * Obtém um documento pelo ID
     * @param int $id ID do documento
     * @return array Documento ou erro
     */
    public function getDocument($id) {
        if ($this->document->readOne($id)) {
            // Obter palavras-chave
            $keywords = $this->document->getKeywords($id);
            
            // Obter versões
            $versions_stmt = $this->document_version->getVersionsByDocumentId($id);
            $versions = [];
            while ($row = $versions_stmt->fetch(PDO::FETCH_ASSOC)) {
                $versions[] = $row;
            }
            
            return [
                "success" => true,
                "document" => [
                    "id" => $this->document->id,
                    "title" => $this->document->title,
                    "description" => $this->document->description,
                    "file_path" => $this->document->file_path,
                    "file_name" => $this->document->file_name,
                    "file_type" => $this->document->file_type,
                    "file_size" => $this->document->file_size,
                    "category_id" => $this->document->category_id,
                    "category_name" => $this->document->category_name,
                    "status" => $this->document->status,
                    "created_at" => $this->document->created_at,
                    "updated_at" => $this->document->updated_at,
                    "created_by" => $this->document->created_by,
                    "author_name" => $this->document->author_name,
                    "keywords" => $keywords,
                    "versions" => $versions,
                    "workflow_id" => $this->document->workflow_id,
                    "current_step_id" => $this->document->current_step_id
                ]
            ];
        } else {
            return [
                "success" => false,
                "message" => "Documento não encontrado"
            ];
        }
    }
    
    /**
     * Atualiza um documento existente
     * @param array $data Novos dados
     * @return array Resultado da atualização
     */
    public function updateDocument($data) {
        // Verificar se o documento existe
        if (!$this->document->readOne($data["id"])) {
            return [
                "success" => false,
                "message" => "Documento não encontrado"
            ];
        }
        
        // Verificar campos obrigatórios
        if (empty($data["title"])) {
            return [
                "success" => false,
                "message" => "O título do documento é obrigatório"
            ];
        }
        
        // Atualizar dados
        $this->document->id = $data["id"];
        $this->document->title = $data["title"];
        $this->document->description = $data["description"] ?? $this->document->description;
        $this->document->category_id = $data["category_id"] ?? $this->document->category_id;
        $this->document->status = $data["status"] ?? $this->document->status;
        $this->document->workflow_id = $data["workflow_id"] ?? $this->document->workflow_id;
        
        if ($this->document->update()) {
            // Atualizar palavras-chave
            if (isset($data["keywords"])) {
                $keywords_array = array_map("trim", explode(",", $data["keywords"]));
                $this->document->setKeywords($data["id"], $keywords_array);
            }
            
            // Registrar log de auditoria
            logAudit($data["user_id"], "update", "document", $data["id"], "Documento atualizado: " . $data["title"]);
            
            return [
                "success" => true,
                "message" => "Documento atualizado com sucesso"
            ];
        } else {
            return [
                "success" => false,
                "message" => "Erro ao atualizar o documento"
            ];
        }
    }

    /**
     * Faz upload de uma nova versão de um documento existente
     * @param int $document_id ID do documento
     * @param array $file Dados do arquivo
     * @param int $user_id ID do usuário que está fazendo o upload
     * @param string $change_summary Resumo das alterações
     * @return array Resultado do upload da nova versão
     */
    public function uploadNewDocumentVersion($document_id, $file, $user_id, $change_summary = "") {
        // Verificar se o documento existe
        if (!$this->document->readOne($document_id)) {
            return [
                "success" => false,
                "message" => "Documento não encontrado"
            ];
        }

        // Verificar se houve erro no upload
        if ($file["error"] !== UPLOAD_ERR_OK) {
            $error_message = $this->getUploadErrorMessage($file["error"]);
            return [
                "success" => false,
                "message" => "Erro no upload da nova versão: " . $error_message
            ];
        }

        // Verificar tipo de arquivo permitido
        $allowed_extensions = explode(",", ALLOWED_EXTENSIONS); 
        $file_extension = strtolower(pathinfo($file["name"], PATHINFO_EXTENSION));
        
        if (!in_array($file_extension, $allowed_extensions)) {
            return [
                "success" => false,
                "message" => "Tipo de arquivo não permitido para nova versão. Extensões permitidas: " . implode(", ", $allowed_extensions)
            ];
        }
        
        // Verificar tamanho máximo
        $max_size = MAX_FILE_SIZE; 
        if ($file["size"] > $max_size) {
            return [
                "success" => false,
                "message" => "O arquivo da nova versão excede o tamanho máximo permitido (" . MAX_UPLOAD_SIZE . "MB)"
            ];
        }

        // Criar diretório de uploads se não existir
        $upload_dir = UPLOAD_PATH . "/" . date("Y/m");
        if (!file_exists($upload_dir)) {
            mkdir($upload_dir, 0755, true);
        }

        // Gerar nome de arquivo único para a nova versão
        $unique_file_name = uniqid() . "_" . time() . "." . $file_extension;
        $new_file_path = $upload_dir . "/" . $unique_file_name;

        // Mover arquivo para o diretório de uploads
        if (!move_uploaded_file($file["tmp_name"], $new_file_path)) {
            return [
                "success" => false,
                "message" => "Erro ao mover o novo arquivo para o diretório de uploads"
            ];
        }

        // Obter o próximo número de versão
        $latest_version_number = $this->document_version->getLatestVersionNumber($document_id);
        $new_version_number = $latest_version_number + 1;

        // Criar nova entrada na tabela de versões
        $this->document_version->document_id = $document_id;
        $this->document_version->version_number = $new_version_number;
        $this->document_version->file_path = str_replace(BASE_PATH, "", $new_file_path);
        $this->document_version->file_name = $file["name"];
        $this->document_version->file_size = $file["size"];
        $this->document_version->file_type = $file["type"];
        $this->document_version->created_by = $user_id;
        $this->document_version->changes = !empty($change_summary) ? $change_summary : "Atualização de arquivo";

        if ($this->document_version->create()) {
            // Atualizar o documento principal para apontar para o novo arquivo
            $this->document->id = $document_id;
            $this->document->file_path = $this->document_version->file_path;
            $this->document->file_name = $this->document_version->file_name;
            $this->document->file_type = $this->document_version->file_type;
            $this->document->file_size = $this->document_version->file_size;
            $this->document->updateFileDetails(); // Método para atualizar apenas os detalhes do arquivo

            logAudit($user_id, "new_version", "document", $document_id, "Nova versão do documento adicionada: " . $this->document->title . " (v" . $new_version_number . ")");

            return [
                "success" => true,
                "message" => "Nova versão do documento enviada com sucesso!"
            ];
        } else {
            // Remover o arquivo recém-enviado se a criação da versão falhar
            unlink($new_file_path);
            return [
                "success" => false,
                "message" => "Erro ao registrar a nova versão do documento no banco de dados."
            ];
        }
    }

    /**
     * Restaura uma versão anterior de um documento
     * @param int $document_id ID do documento
     * @param int $version_number Número da versão a ser restaurada
     * @param int $user_id ID do usuário que está restaurando
     * @return array Resultado da restauração
     */
    public function restoreDocumentVersion($document_id, $version_number, $user_id) {
        // Obter a versão a ser restaurada
        $version_data = $this->document_version->getVersionByDocumentAndNumber($document_id, $version_number);

        if (!$version_data) {
            return [
                "success" => false,
                "message" => "Versão do documento não encontrada."
            ];
        }

        // Obter o documento principal
        if (!$this->document->readOne($document_id)) {
            return [
                "success" => false,
                "message" => "Documento principal não encontrado."
            ];
        }

        // Criar uma nova versão do documento atual antes de restaurar
        $latest_version_number = $this->document_version->getLatestVersionNumber($document_id);
        $new_version_number_for_current = $latest_version_number + 1;

        $this->document_version->document_id = $document_id;
        $this->document_version->version_number = $new_version_number_for_current;
        $this->document_version->file_path = $this->document->file_path;
        $this->document_version->file_name = $this->document->file_name;
        $this->document_version->file_size = $this->document->file_size;
        $this->document_version->file_type = $this->document->file_type;
        $this->document_version->created_by = $user_id;
        $this->document_version->changes = "Backup automático antes da restauração da v" . $version_number;
        $this->document_version->create();

        // Atualizar o documento principal com os dados da versão restaurada
        $this->document->id = $document_id;
        $this->document->file_path = $version_data["file_path"];
        $this->document->file_name = $version_data["file_name"];
        $this->document->file_type = $version_data["file_type"];
        $this->document->file_size = $version_data["file_size"];
        
        if ($this->document->updateFileDetails()) {
            logAudit($user_id, "restore_version", "document", $document_id, "Documento restaurado para a versão: " . $version_number);
            return [
                "success" => true,
                "message" => "Documento restaurado para a versão " . $version_number . " com sucesso!"
            ];
        } else {
            return [
                "success" => false,
                "message" => "Erro ao restaurar a versão do documento."
            ];
        }
    }

    /**
     * Exclui um documento
     * @param int $id ID do documento
     * @param int $user_id ID do usuário que está excluindo
     * @return array Resultado da exclusão
     */
    public function deleteDocument($id, $user_id) {
        if (!$this->document->readOne($id)) {
            return [
                "success" => false,
                "message" => "Documento não encontrado"
            ];
        }

        $document_title = $this->document->title;
        $file_path = BASE_PATH . $this->document->file_path;

        // Excluir todas as versões do documento
        $this->document_version->document_id = $id;
        $this->document_version->deleteByDocumentId();

        // Excluir aprovações relacionadas
        $this->workflow_approval->document_id = $id;
        $this->workflow_approval->deleteByDocumentId();

        if ($this->document->delete()) {
            // Tentar remover o arquivo físico
            if (file_exists($file_path)) {
                unlink($file_path);
            }
            logAudit($user_id, "delete", "document", $id, "Documento excluído: " . $document_title);
            return [
                "success" => true,
                "message" => "Documento excluído com sucesso"
            ];
        } else {
            return [
                "success" => false,
                "message" => "Erro ao excluir o documento"
            ];
        }
    }

    /**
     * Obtém uma mensagem de erro de upload com base no código de erro.
     * @param int $error_code Código de erro do upload.
     * @return string Mensagem de erro.
     */
    private function getUploadErrorMessage($error_code) {
        switch ($error_code) {
            case UPLOAD_ERR_INI_SIZE:
                return "O arquivo enviado excede a diretiva upload_max_filesize no php.ini.";
            case UPLOAD_ERR_FORM_SIZE:
                return "O arquivo enviado excede a diretiva MAX_FILE_SIZE que foi especificada no formulário HTML.";
            case UPLOAD_ERR_PARTIAL:
                return "O upload do arquivo foi feito apenas parcialmente.";
            case UPLOAD_ERR_NO_FILE:
                return "Nenhum arquivo foi enviado.";
            case UPLOAD_ERR_NO_TMP_DIR:
                return "Faltando uma pasta temporária.";
            case UPLOAD_ERR_CANT_WRITE:
                return "Falha ao gravar o arquivo em disco.";
            case UPLOAD_ERR_EXTENSION:
                return "Uma extensão do PHP interrompeu o upload do arquivo.";
            default:
                return "Erro de upload desconhecido.";
        }
    }

    /**
     * Obtém o histórico de aprovações de um documento.
     * @param int $document_id ID do documento.
     * @return array Histórico de aprovações.
     */
    public function getDocumentApprovalHistory($document_id) {
        $this->workflow_approval->document_id = $document_id;
        $stmt = $this->workflow_approval->readByDocumentId();
        $history = [];
        while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
            $history[] = $row;
        }
        return $history;
    }

    /**
     * Verifica se o usuário pode acessar o documento (ainda não implementado).
     * @param int $user_id ID do usuário.
     * @param int $document_id ID do documento.
     * @return bool Verdadeiro se o usuário pode acessar, falso caso contrário.
     */
    public function userCanAccessDocument($user_id, $document_id) {
        // TODO: Implementar lógica de permissão de acesso ao documento
        // Por enquanto, permite acesso se for o autor ou admin
        $this->document->readOne($document_id);
        if ($this->document->created_by == $user_id || $_SESSION["user_type"] == "admin") {
            return true;
        }
        return false;
    }

    /**
     * Verifica se o usuário pode aprovar um documento (ainda não implementado).
     * @param int $user_id ID do usuário.
     * @param int $document_id ID do documento.
     * @return bool Verdadeiro se o usuário pode aprovar, falso caso contrário.
     */
    public function userCanApproveDocument($user_id, $document_id) {
        // TODO: Implementar lógica de permissão de aprovação de documento
        // Por enquanto, permite aprovação se for admin ou se for o aprovador da etapa atual
        $this->document->readOne($document_id);
        if ($this->document->status == "pending" && $this->document->current_step_id) {
            $this->workflow_step->id = $this->document->current_step_id;
            $this->workflow_step->readOne();
            if ($this->workflow_step->assigned_to_user_type == "any" || 
                ($this->workflow_step->assigned_to_user_type == "user" && $this->workflow_step->assigned_to_user_id == $user_id) ||
                ($this->workflow_step->assigned_to_user_type == "admin" && $_SESSION["user_type"] == "admin")) {
                return true;
            }
        }
        return false;
    }
}

// Função auxiliar para logs de auditoria (se não estiver em um arquivo separado)
if (!function_exists('logAudit')) {
    function logAudit($user_id, $action, $entity_type, $entity_id, $details) {
        global $db; // Acessa a conexão do banco de dados global
        $audit_log = new AuditLog($db);
        $audit_log->user_id = $user_id;
        $audit_log->action = $action;
        $audit_log->entity_type = $entity_type;
        $audit_log->entity_id = $entity_id;
        $audit_log->details = $details;
        $audit_log->create();
    }
}

?>

