<?php
/**
 * Controller para gerenciamento de configurações do sistema
 * Sistema de Gerenciamento Eletrônico de Documentos (GED)
 */

class SettingsController {
    private $db;
    
    public function __construct() {
        $database = new Database();
        $this->db = $database->getConnection();
    }
    
    /**
     * Obtém todas as configurações do sistema
     * @return array Configurações do sistema
     */
    public function getAllSettings() {
        try {
            $query = getenv('APP_QUERY');
            $stmt = $this->db->prepare($query);
            $stmt->execute();
            
            $settings = [];
            while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
                $settings[$row['setting_key']] = $row['setting_value'];
            }
            
            return $settings;
        } catch (PDOException $e) {
            // Se a tabela não existir, retorna configurações padrão
            if ($e->getCode() == '42S02') {
                return $this->getDefaultSettings();
            }
            
            error_log("Erro ao obter configurações: " . $e->getMessage());
            return $this->getDefaultSettings();
        }
    }
    
    /**
     * Obtém uma configuração específica
     * @param string $key Chave da configuração
     * @param mixed $default Valor padrão caso a configuração não exista
     * @return mixed Valor da configuração
     */
    public function getSetting($key, $default = null) {
        try {
            $query = getenv('APP_QUERY');
            $stmt = $this->db->prepare($query);
            $stmt->bindParam(':key', $key);
            $stmt->execute();
            
            if ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
                return $row['setting_value'];
            }
            
            return $default;
        } catch (PDOException $e) {
            error_log("Erro ao obter configuração: " . $e->getMessage());
            return $default;
        }
    }
    
    /**
     * Atualiza uma configuração específica
     * @param string $key Chave da configuração
     * @param mixed $value Valor da configuração
     * @return bool Sucesso da operação
     */
    public function updateSetting($key, $value) {
        try {
            // Verificar se a configuração já existe
            $query = getenv('APP_QUERY');
            $stmt = $this->db->prepare($query);
            $stmt->bindParam(':key', $key);
            $stmt->execute();
            
            if ($stmt->fetchColumn() > 0) {
                // Atualizar configuração existente
                $query = getenv('APP_QUERY');
            } else {
                // Inserir nova configuração
                $query = getenv('APP_QUERY');
            }
            
            $stmt = $this->db->prepare($query);
            $stmt->bindParam(':key', $key);
            $stmt->bindParam(':value', $value);
            
            return $stmt->execute();
        } catch (PDOException $e) {
            error_log("Erro ao atualizar configuração: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Atualiza configurações gerais do sistema
     * @param array $data Dados das configurações
     * @return array Resultado da operação
     */
    public function updateGeneralSettings($data) {
        try {
            foreach ($data as $key => $value) {
                $this->updateSetting($key, $value);
            }
            
            // Registrar ação no log de auditoria
            $user_id = $_SESSION['user_id'] ?? 0;
            $this->logAction($user_id, 'update_settings', 'general', 'Configurações gerais atualizadas');
            
            return ['success' => true];
        } catch (Exception $e) {
            error_log("Erro ao atualizar configurações gerais: " . $e->getMessage());
            return ['success' => false, 'message' => 'Erro ao atualizar configurações: ' . $e->getMessage()];
        }
    }
    
    /**
     * Atualiza configurações de e-mail
     * @param array $data Dados das configurações
     * @return array Resultado da operação
     */
    public function updateEmailSettings($data) {
        try {
            foreach ($data as $key => $value) {
                $this->updateSetting($key, $value);
            }
            
            // Registrar ação no log de auditoria
            $user_id = $_SESSION['user_id'] ?? 0;
            $this->logAction($user_id, 'update_settings', 'email', 'Configurações de e-mail atualizadas');
            
            return ['success' => true];
        } catch (Exception $e) {
            error_log("Erro ao atualizar configurações de e-mail: " . $e->getMessage());
            return ['success' => false, 'message' => 'Erro ao atualizar configurações: ' . $e->getMessage()];
        }
    }
    
    /**
     * Atualiza configurações de backup
     * @param array $data Dados das configurações
     * @return array Resultado da operação
     */
    public function updateBackupSettings($data) {
        try {
            foreach ($data as $key => $value) {
                $this->updateSetting($key, $value);
            }
            
            // Registrar ação no log de auditoria
            $user_id = $_SESSION['user_id'] ?? 0;
            $this->logAction($user_id, 'update_settings', 'backup', 'Configurações de backup atualizadas');
            
            return ['success' => true];
        } catch (Exception $e) {
            error_log("Erro ao atualizar configurações de backup: " . $e->getMessage());
            return ['success' => false, 'message' => 'Erro ao atualizar configurações: ' . $e->getMessage()];
        }
    }
    
    /**
     * Atualiza configurações de aparência
     * @param array $data Dados das configurações
     * @return array Resultado da operação
     */
    public function updateAppearanceSettings($data) {
        try {
            foreach ($data as $key => $value) {
                $this->updateSetting($key, $value);
            }
            
            // Registrar ação no log de auditoria
            $user_id = $_SESSION['user_id'] ?? 0;
            $this->logAction($user_id, 'update_settings', 'appearance', 'Configurações de aparência atualizadas');
            
            return ['success' => true];
        } catch (Exception $e) {
            error_log("Erro ao atualizar configurações de aparência: " . $e->getMessage());
            return ['success' => false, 'message' => 'Erro ao atualizar configurações: ' . $e->getMessage()];
        }
    }
    
    /**
     * Obtém estatísticas do sistema
     * @param string $type Tipo de estatística
     * @return mixed Valor da estatística
     */
    public function getStatistics($type) {
        try {
            switch ($type) {
                case 'users':
                    $query = getenv('APP_QUERY');
                    $stmt = $this->db->prepare($query);
                    $stmt->execute();
                    return $stmt->fetchColumn();
                
                case 'documents':
                    $query = getenv('APP_QUERY');
                    $stmt = $this->db->prepare($query);
                    $stmt->execute();
                    return $stmt->fetchColumn();
                
                case 'categories':
                    $query = getenv('APP_QUERY');
                    $stmt = $this->db->prepare($query);
                    $stmt->execute();
                    return $stmt->fetchColumn();
                
                case 'disk_usage':
                    // Calcular espaço usado pelos documentos
                    $query = getenv('APP_QUERY');
                    $stmt = $this->db->prepare($query);
                    $stmt->execute();
                    $bytes = $stmt->fetchColumn() ?: 0;
                    
                    // Converter para formato legível
                    if ($bytes < 1024) {
                        return $bytes . " bytes";
                    } elseif ($bytes < 1048576) {
                        return round($bytes / 1024, 2) . " KB";
                    } elseif ($bytes < 1073741824) {
                        return round($bytes / 1048576, 2) . " MB";
                    } else {
                        return round($bytes / 1073741824, 2) . " GB";
                    }
                
                default:
                    return 0;
            }
        } catch (PDOException $e) {
            error_log("Erro ao obter estatísticas: " . $e->getMessage());
            return 0;
        }
    }
    
    /**
     * Realiza verificações do sistema
     * @return array Lista de verificações e seus status
     */
    public function performSystemChecks() {
        $checks = [];
        
        // Verificar permissões de diretório de uploads
        $upload_dir = getenv('APP_UPLOAD_DIR');
        $checks[] = [
            'name' => 'Diretório de uploads com permissão de escrita',
            'status' => is_writable($upload_dir) ? 'ok' : 'error'
        ];
        
        // Verificar extensão PDO MySQL
        $checks[] = [
            'name' => 'Extensão PDO MySQL',
            'status' => extension_loaded('pdo_mysql') ? 'ok' : 'error'
        ];
        
        // Verificar extensão GD (para manipulação de imagens)
        $checks[] = [
            'name' => 'Extensão GD (processamento de imagens)',
            'status' => extension_loaded('gd') ? 'ok' : 'warning'
        ];
        
        // Verificar extensão Zip
        $checks[] = [
            'name' => 'Extensão Zip',
            'status' => extension_loaded('zip') ? 'ok' : 'warning'
        ];
        
        // Verificar versão do PHP
        $checks[] = [
            'name' => 'Versão do PHP (7.4+)',
            'status' => version_compare(PHP_VERSION, '7.4.0') >= 0 ? 'ok' : 'warning'
        ];
        
        // Verificar limite de upload
        $upload_max = ini_get('upload_max_filesize');
        $post_max = ini_get('post_max_size');
        $upload_mb = $this->convertToMB($upload_max);
        $post_mb = $this->convertToMB($post_max);
        
        $checks[] = [
            'name' => 'Limite de upload (recomendado: 10MB+)',
            'status' => $upload_mb >= 10 ? 'ok' : 'warning'
        ];
        
        $checks[] = [
            'name' => 'Limite de POST (recomendado: 10MB+)',
            'status' => $post_mb >= 10 ? 'ok' : 'warning'
        ];
        
        // Verificar conexão com banco de dados
        try {
            $this->db->query("SELECT 1");
            $checks[] = [
                'name' => 'Conexão com banco de dados',
                'status' => 'ok'
            ];
        } catch (PDOException $e) {
            $checks[] = [
                'name' => 'Conexão com banco de dados',
                'status' => 'error'
            ];
        }
        
        return $checks;
    }
    
    /**
     * Converte string de tamanho (como '10M') para MB
     * @param string $size String de tamanho
     * @return float Tamanho em MB
     */
    private function convertToMB($size) {
        $unit = strtoupper(substr($size, -1));
        $value = (float)$size;
        
        switch ($unit) {
            case 'G':
                $value *= 1024;
                break;
            case 'K':
                $value /= 1024;
                break;
            case 'M':
                // Já está em MB
                break;
        }
        
        return $value;
    }
    
    /**
     * Registra ação no log de auditoria
     * @param int $user_id ID do usuário
     * @param string $action Ação realizada
     * @param string $target Alvo da ação
     * @param string $description Descrição da ação
     * @return bool Sucesso da operação
     */
    private function logAction($user_id, $action, $target, $description) {
        try {
            $query = getenv('APP_QUERY');
            
            $stmt = $this->db->prepare($query);
            $stmt->bindParam(':user_id', $user_id);
            $stmt->bindParam(':action', $action);
            $stmt->bindParam(':target', $target);
            $stmt->bindParam(':description', $description);
            
            $ip_address = $_SERVER['REMOTE_ADDR'] ?? '0.0.0.0';
            $stmt->bindParam(':ip_address', $ip_address);
            
            return $stmt->execute();
        } catch (PDOException $e) {
            error_log("Erro ao registrar ação no log: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Retorna configurações padrão do sistema
     * @return array Configurações padrão
     */
    private function getDefaultSettings() {
        return [
            'system_name' => 'Sistema GED',
            'allow_registration' => '0',
            'max_upload_size' => '10',
            'allowed_extensions' => 'pdf,doc,docx,xls,xlsx,jpg,jpeg,png',
            'default_language' => 'pt_BR',
            'items_per_page' => '20',
            'smtp_host' => '',
            'smtp_port' => '587',
            'smtp_secure' => 'tls',
            'smtp_auth' => '1',
            'smtp_username' => '',
            'smtp_password' => '',
            'mail_from' => '',
            'mail_from_name' => 'Sistema GED',
            'auto_backup' => '0',
            'backup_frequency' => 'daily',
            'backup_time' => '00:00',
            'backup_retention' => '7',
            'backup_include_files' => '1',
            'backup_destination' => 'local',
            'default_theme' => 'light',
            'primary_color' => '#0d6efd',
            'secondary_color' => '#6c757d',
            'installation_date' => date('Y-m-d'),
            'last_update' => date('Y-m-d')
        ];
    }
}
?>
