<?php
/**
 * Modelo de Log de Auditoria
 * Sistema de Gerenciamento Eletrônico de Documentos (GED)
 */

class AuditLog {
    // Conexão com o banco e nome da tabela
    private $conn;
    private $table_name = getenv('APP_TABLE_NAME');

    // Propriedades do objeto
    public $id;
    public $user_id;
    public $action;
    public $resource_type;
    public $resource_id;
    public $details;
    public $ip_address;
    public $user_agent;
    public $created_at;

    /**
     * Construtor com $db como conexão com o banco de dados
     * @param PDO $db Conexão com o banco de dados
     */
    public function __construct($db) {
        $this->conn = $db;
    }

    /**
     * Cria um novo registro de log
     * @return bool Verdadeiro se criado com sucesso, falso caso contrário
     */
    public function create() {
        $query = getenv('APP_QUERY') . $this->table_name . "                  (user_id, action, resource_type, resource_id, details, ip_address, user_agent)                  VALUES (?, ?, ?, ?, ?, ?, ?)";
        
        $stmt = $this->conn->prepare($query);
        
        // Sanitizar
        $this->details = htmlspecialchars(strip_tags($this->details));
        $this->ip_address = htmlspecialchars(strip_tags($this->ip_address));
        $this->user_agent = htmlspecialchars(strip_tags($this->user_agent));
        
        // Vincular valores
        $stmt->bindParam(1, $this->user_id);
        $stmt->bindParam(2, $this->action);
        $stmt->bindParam(3, $this->resource_type);
        $stmt->bindParam(4, $this->resource_id);
        $stmt->bindParam(5, $this->details);
        $stmt->bindParam(6, $this->ip_address);
        $stmt->bindParam(7, $this->user_agent);
        
        // Executar
        if ($stmt->execute()) {
            $this->id = $this->conn->lastInsertId();
            return true;
        }
        
        return false;
    }

    /**
     * Lê logs com filtros e paginação
     * @param array $filters Filtros
     * @param int $page Página atual
     * @param int $per_page Itens por página
     * @return array Logs e informações de paginação
     */
    public function readAll($filters = [], $page = 1, $per_page = 50) {
        // Construir consulta base
        $query = getenv('APP_QUERY') . $this->table_name . " al
                  LEFT JOIN user u ON al.user_id = u.id
                  WHERE 1=1";
        
        $params = [];
        
        // Aplicar filtros
        if (!empty($filters['user_id'])) {
            $query .= " AND al.user_id = ?";
            $params[] = $filters['user_id'];
        }
        
        if (!empty($filters['action'])) {
            $query .= " AND al.action = ?";
            $params[] = $filters['action'];
        }
        
        if (!empty($filters['resource_type'])) {
            $query .= " AND al.resource_type = ?";
            $params[] = $filters['resource_type'];
        }
        
        if (!empty($filters['entity_id'])) {
            $query .= " AND al.entity_id = ?";
            $params[] = $filters['entity_id'];
        }
        
        if (!empty($filters['start_date'])) {
            $query .= " AND al.created_at >= ?";
            $params[] = $filters['start_date'] . ' 00:00:00';
        }
        
        if (!empty($filters['end_date'])) {
            $query .= " AND al.created_at <= ?";
            $params[] = $filters['end_date'] . ' 23:59:59';
        }
        
        if (!empty($filters['search'])) {
            $query .= " AND (al.details LIKE ? OR u.username LIKE ?)";
            $search_term = getenv('APP_SEARCH_TERM') . $filters['search'] . "%";
            $params[] = $search_term;
            $params[] = $search_term;
        }
        
        // Contar total de resultados para paginação
        $count_query = str_replace("SELECT al.*, u.username", "SELECT COUNT(*) as total", $query);
        $stmt = $this->conn->prepare($count_query);
        
        for ($i = 0; $i < count($params); $i++) {
            $stmt->bindParam($i + 1, $params[$i]);
        }
        
        $stmt->execute();
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        $total = $row['total'];
        
        // Adicionar ordenação e paginação
        $query .= " ORDER BY al.created_at DESC";
        
        // Calcular offset para paginação
        $offset = ($page - 1) * $per_page;
        $query .= " LIMIT ?, ?";
        
        $stmt = $this->conn->prepare($query);
        
        // Vincular parâmetros de filtro
        for ($i = 0; $i < count($params); $i++) {
            $stmt->bindParam($i + 1, $params[$i]);
        }
        
        // Vincular parâmetros de paginação
        $stmt->bindParam(count($params) + 1, $offset, PDO::PARAM_INT);
        $stmt->bindParam(count($params) + 2, $per_page, PDO::PARAM_INT);
        
        $stmt->execute();
        
        $logs = [];
        while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
            $logs[] = $row;
        }
        
        // Calcular total de páginas
        $total_pages = ceil($total / $per_page);
        
        return [
            'logs' => $logs,
            'pagination' => [
                'total' => $total,
                'per_page' => $per_page,
                'current_page' => $page,
                'total_pages' => $total_pages
            ]
        ];
    }

    /**
     * Lê um único log pelo ID
     * @param int $id ID do log
     * @return bool Verdadeiro se encontrado, falso caso contrário
     */
    public function readOne($id) {
        $query = getenv('APP_QUERY') . $this->table_name . " al
                  LEFT JOIN user u ON al.user_id = u.id
                  WHERE al.id = ? LIMIT 0,1";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(1, $id);
        $stmt->execute();

        if ($stmt->rowCount() > 0) {
            $row = $stmt->fetch(PDO::FETCH_ASSOC);
            
            $this->id = $row['id'];
            $this->user_id = $row['user_id'];
            $this->action = $row['action'];
            $this->resource_type = $row['resource_type'];
            $this->resource_id = $row["resource_id"];
            $this->details = $row['details'];
            $this->ip_address = $row['ip_address'];
            $this->user_agent = $row['user_agent'];
            $this->created_at = $row['created_at'];
            
            return true;
        }
        
        return false;
    }

    /**
     * Obtém estatísticas de atividade por usuário
     * @param string $start_date Data de início (formato Y-m-d)
     * @param string $end_date Data de fim (formato Y-m-d)
     * @return array Estatísticas
     */
    public function getUserActivityStats($start_date = null, $end_date = null) {
        $query = getenv('APP_QUERY') . $this->table_name . " al ON u.id = al.user_id";
        
        $params = [];
        
        if ($start_date && $end_date) {
            $query .= " AND al.created_at BETWEEN ? AND ?";
            $params[] = $start_date . ' 00:00:00';
            $params[] = $end_date . ' 23:59:59';
        }
        
        $query .= " GROUP BY u.id ORDER BY action_count DESC";
        
        $stmt = $this->conn->prepare($query);
        
        for ($i = 0; $i < count($params); $i++) {
            $stmt->bindParam($i + 1, $params[$i]);
        }
        
        $stmt->execute();
        
        $stats = [];
        while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
            $stats[] = $row;
        }
        
        return $stats;
    }

    /**
     * Obtém estatísticas de atividade por tipo de entidade
     * @param string $start_date Data de início (formato Y-m-d)
     * @param string $end_date Data de fim (formato Y-m-d)
     * @return array Estatísticas
     */
    public function getEntityActivityStats($start_date = null, $end_date = null) {
        $query = getenv('APP_QUERY') . $this->table_name;
        
        $params = [];
        
        if ($start_date && $end_date) {
            $query .= " WHERE created_at BETWEEN ? AND ?";
            $params[] = $start_date . ' 00:00:00';
            $params[] = $end_date . ' 23:59:59';
        }
        
        $query .= " GROUP BY entity_type ORDER BY action_count DESC";
        
        $stmt = $this->conn->prepare($query);
        
        for ($i = 0; $i < count($params); $i++) {
            $stmt->bindParam($i + 1, $params[$i]);
        }
        
        $stmt->execute();
        
        $stats = [];
        while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
            $stats[] = $row;
        }
        
        return $stats;
    }

    /**
     * Obtém estatísticas de atividade por ação
     * @param string $start_date Data de início (formato Y-m-d)
     * @param string $end_date Data de fim (formato Y-m-d)
     * @return array Estatísticas
     */
    public function getActionStats($start_date = null, $end_date = null) {
        $query = getenv('APP_QUERY') . $this->table_name;
        
        $params = [];
        
        if ($start_date && $end_date) {
            $query .= " WHERE created_at BETWEEN ? AND ?";
            $params[] = $start_date . ' 00:00:00';
            $params[] = $end_date . ' 23:59:59';
        }
        
        $query .= " GROUP BY action ORDER BY action_count DESC";
        
        $stmt = $this->conn->prepare($query);
        
        for ($i = 0; $i < count($params); $i++) {
            $stmt->bindParam($i + 1, $params[$i]);
        }
        
        $stmt->execute();
        
        $stats = [];
        while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
            $stats[] = $row;
        }
        
        return $stats;
    }

    /**
     * Obtém histórico de atividade de um documento específico
     * @param int $document_id ID do documento
     * @return array Histórico de atividade
     */
    public function getDocumentHistory($document_id) {
        $query = getenv('APP_QUERY') . $this->table_name . " al
                  LEFT JOIN user u ON al.user_id = u.id
                  WHERE al.entity_type = 'document' AND al.entity_id = ?
                  ORDER BY al.created_at DESC";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(1, $document_id);
        $stmt->execute();
        
        $history = [];
        while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
            $history[] = $row;
        }
        
        return $history;
    }
}

/**
 * Função auxiliar para registrar logs de auditoria
 * @param int $user_id ID do usuário
 * @param string $action Ação realizada
 * @param string $entity_type Tipo de entidade
 * @param int $entity_id ID da entidade
 * @param string $details Detalhes da ação
 * @return bool Verdadeiro se registrado com sucesso, falso caso contrário
 */
function logAudit($user_id, $action, $resource_type, $entity_id, $details) {
    global $db;
    
    // Se a conexão global não estiver disponível, criar uma nova
    if (!isset($db)) {
        require_once __DIR__ . '/../config/database.php';
        $database = new Database();
        $db = $database->getConnection();
    }
    
    $audit_log = new AuditLog($db);
    $audit_log->user_id = $user_id;
    $audit_log->action = $action;
    $audit_log->resource_type = $resource_type;
    $audit_log->entity_id = $entity_id;
    $audit_log->details = $details;
    
    // Obter IP e User Agent
    $audit_log->ip_address = $_SERVER['REMOTE_ADDR'] ?? '0.0.0.0';
    $audit_log->user_agent = $_SERVER['HTTP_USER_AGENT'] ?? 'Unknown';
    
    return $audit_log->create();
}
?>
