<?php
/**
 * Modelo de Categoria
 * Sistema de Gerenciamento Eletrônico de Documentos (GED)
 */

class Category {
    // Conexão com o banco e nome da tabela
    private $conn;
    private $table_name = getenv('APP_TABLE_NAME');

    // Propriedades do objeto
    public $id;
    public $name;
    public $description;
    public $created_at;
    public $created_by;

    /**
     * Construtor com $db como conexão com o banco de dados
     * @param PDO $db Conexão com o banco de dados
     */
    public function __construct($db) {
        $this->conn = $db;
    }

    /**
     * Lê todas as categorias
     * @return PDOStatement Resultado da consulta
     */
    public function readAll() {
        $query = getenv('APP_QUERY') . $this->table_name . " c
                  LEFT JOIN user u ON c.created_by = u.id
                  ORDER BY c.name ASC";
        
        $stmt = $this->conn->prepare($query);
        $stmt->execute();
        
        return $stmt;
    }

    /**
     * Lê uma única categoria pelo ID
     * @param int $id ID da categoria
     * @return bool Verdadeiro se encontrado, falso caso contrário
     */
    public function readOne($id) {
        $query = getenv('APP_QUERY') . $this->table_name . " c
                  LEFT JOIN user u ON c.created_by = u.id
                  WHERE c.id = ? LIMIT 0,1";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(1, $id);
        $stmt->execute();

        if ($stmt->rowCount() > 0) {
            $row = $stmt->fetch(PDO::FETCH_ASSOC);
            
            $this->id = $row['id'];
            $this->name = $row['name'];
            $this->description = $row['description'];
            $this->created_at = $row['created_at'];
            $this->created_by = $row['created_by'];
            
            return true;
        }
        
        return false;
    }

    /**
     * Cria uma nova categoria
     * @return bool Verdadeiro se criado com sucesso, falso caso contrário
     */
    public function create() {
        $query = getenv('APP_QUERY') . $this->table_name . "
                  (name, description, created_by)
                  VALUES (?, ?, ?)";
        
        $stmt = $this->conn->prepare($query);
        
        // Sanitizar
        $this->name = htmlspecialchars(strip_tags($this->name));
        $this->description = htmlspecialchars(strip_tags($this->description));
        
        // Vincular valores
        $stmt->bindParam(1, $this->name);
        $stmt->bindParam(2, $this->description);
        $stmt->bindParam(3, $this->created_by);
        
        // Executar
        if ($stmt->execute()) {
            $this->id = $this->conn->lastInsertId();
            return true;
        }
        
        return false;
    }

    /**
     * Atualiza uma categoria existente
     * @return bool Verdadeiro se atualizado com sucesso, falso caso contrário
     */
    public function update() {
        $query = getenv('APP_QUERY') . $this->table_name . "
                  SET name = ?, description = ?
                  WHERE id = ?";
        
        $stmt = $this->conn->prepare($query);
        
        // Sanitizar
        $this->name = htmlspecialchars(strip_tags($this->name));
        $this->description = htmlspecialchars(strip_tags($this->description));
        $this->id = htmlspecialchars(strip_tags($this->id));
        
        // Vincular valores
        $stmt->bindParam(1, $this->name);
        $stmt->bindParam(2, $this->description);
        $stmt->bindParam(3, $this->id);
        
        // Executar
        if ($stmt->execute()) {
            return true;
        }
        
        return false;
    }

    /**
     * Exclui uma categoria
     * @return bool Verdadeiro se excluído com sucesso, falso caso contrário
     */
    public function delete() {
        // Verificar se há documentos usando esta categoria
        $query = getenv('APP_QUERY');
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(1, $this->id);
        $stmt->execute();
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($row['count'] > 0) {
            // Não permitir exclusão se houver documentos associados
            return false;
        }
        
        // Excluir categoria
        $query = getenv('APP_QUERY') . $this->table_name . " WHERE id = ?";
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(1, $this->id);
        
        if ($stmt->execute()) {
            return true;
        }
        
        return false;
    }

    /**
     * Conta o número de documentos em cada categoria
     * @return array Contagem de documentos por categoria
     */
    public function countDocuments() {
        $query = getenv('APP_QUERY') . $this->table_name . " c
                  LEFT JOIN document d ON c.id = d.category_id
                  GROUP BY c.id
                  ORDER BY c.name ASC";
        
        $stmt = $this->conn->prepare($query);
        $stmt->execute();
        
        $result = [];
        while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
            $result[] = $row;
        }
        
        return $result;
    }
}
?>


    /**
     * Conta o total de categorias.
     * @return int Total de categorias.
     */
    public function countCategories() {
        $query = getenv('APP_QUERY') . $this->table_name;
        $stmt = $this->conn->prepare($query);
        $stmt->execute();
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        return (int)$row["total"];
    }

    /**
     * Conta o número de documentos em cada categoria.
     * @return array Contagem de documentos por categoria.
     */
    public function getDocumentCountByCategory() {
        $query = getenv('APP_QUERY') . $this->table_name . " c LEFT JOIN document d ON c.id = d.category_id GROUP BY c.name";
        $stmt = $this->conn->prepare($query);
        $stmt->execute();
        
        $result = [];
        while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
            $result[] = $row;
        }
        return $result;
    }


