<?php
/**
 * Página de configurações do sistema
 * Sistema de Gerenciamento Eletrônico de Documentos (GED)
 */

// Iniciar sessão e verificar autenticação
session_start();
if (!isset($_SESSION['user_id'])) {
    header('Location: login.php');
    exit;
}

// Verificar se o usuário tem permissão de administrador
if ($_SESSION['user_type'] != 'admin') {
    $_SESSION['flash_message'] = "Acesso negado. Você não tem permissão para acessar esta página.";
    $_SESSION['flash_type'] = "danger";
    header('Location: index.php');
    exit;
}

// Incluir arquivos necessários
require_once 'config/database.php';
require_once 'config/config.php';
require_once 'controllers/settings_controller.php';

// Conectar ao banco de dados
$database = new Database();
$db = $database->getConnection();

// Instanciar controlador
$settings_controller = new SettingsController();

// Processar formulário
if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    if (isset($_POST['action'])) {
        // Atualizar configurações gerais
        if ($_POST['action'] == 'general') {
            $result = $settings_controller->updateGeneralSettings([
                'system_name' => $_POST['system_name'] ?? 'Sistema GED',
                'allow_registration' => isset($_POST['allow_registration']) ? 1 : 0,
                'max_upload_size' => $_POST['max_upload_size'] ?? 10,
                'allowed_extensions' => $_POST['allowed_extensions'] ?? 'pdf,doc,docx,xls,xlsx,jpg,jpeg,png',
                'default_language' => $_POST['default_language'] ?? 'pt_BR',
                'items_per_page' => $_POST['items_per_page'] ?? 20
            ]);
            
            if ($result['success']) {
                $_SESSION['flash_message'] = "Configurações gerais atualizadas com sucesso.";
                $_SESSION['flash_type'] = "success";
            } else {
                $_SESSION['flash_message'] = $result['message'];
                $_SESSION['flash_type'] = "danger";
            }
        }
        
        // Atualizar configurações de e-mail
        else if ($_POST['action'] == 'email') {
            $result = $settings_controller->updateEmailSettings([
                'smtp_host' => $_POST['smtp_host'] ?? '',
                'smtp_port' => $_POST['smtp_port'] ?? 587,
                'smtp_secure' => $_POST['smtp_secure'] ?? 'tls',
                'smtp_auth' => isset($_POST['smtp_auth']) ? 1 : 0,
                'smtp_username' => $_POST['smtp_username'] ?? '',
                'smtp_password' => $_POST['smtp_password'] ?? '',
                'mail_from' => $_POST['mail_from'] ?? '',
                'mail_from_name' => $_POST['mail_from_name'] ?? 'Sistema GED'
            ]);
            
            if ($result['success']) {
                $_SESSION['flash_message'] = "Configurações de e-mail atualizadas com sucesso.";
                $_SESSION['flash_type'] = "success";
            } else {
                $_SESSION['flash_message'] = $result['message'];
                $_SESSION['flash_type'] = "danger";
            }
        }
        
        // Atualizar configurações de backup
        else if ($_POST['action'] == 'backup') {
            $result = $settings_controller->updateBackupSettings([
                'auto_backup' => isset($_POST['auto_backup']) ? 1 : 0,
                'backup_frequency' => $_POST['backup_frequency'] ?? 'daily',
                'backup_time' => $_POST['backup_time'] ?? '00:00',
                'backup_retention' => $_POST['backup_retention'] ?? 7,
                'backup_include_files' => isset($_POST['backup_include_files']) ? 1 : 0,
                'backup_destination' => $_POST['backup_destination'] ?? 'local',
                'backup_cloud_provider' => $_POST['backup_cloud_provider'] ?? '',
                'backup_cloud_key' => $_POST['backup_cloud_key'] ?? '',
                'backup_cloud_secret' => $_POST['backup_cloud_secret'] ?? '',
                'backup_cloud_bucket' => $_POST['backup_cloud_bucket'] ?? ''
            ]);
            
            if ($result['success']) {
                $_SESSION['flash_message'] = "Configurações de backup atualizadas com sucesso.";
                $_SESSION['flash_type'] = "success";
            } else {
                $_SESSION['flash_message'] = $result['message'];
                $_SESSION['flash_type'] = "danger";
            }
        }
        
        // Atualizar configurações de aparência
        else if ($_POST['action'] == 'appearance') {
            $result = $settings_controller->updateAppearanceSettings([
                'default_theme' => $_POST['default_theme'] ?? 'light',
                'primary_color' => $_POST['primary_color'] ?? '#0d6efd',
                'secondary_color' => $_POST['secondary_color'] ?? '#6c757d',
                'logo_url' => $_POST['logo_url'] ?? '',
                'favicon_url' => $_POST['favicon_url'] ?? '',
                'custom_css' => $_POST['custom_css'] ?? ''
            ]);
            
            if ($result['success']) {
                $_SESSION['flash_message'] = "Configurações de aparência atualizadas com sucesso.";
                $_SESSION['flash_type'] = "success";
            } else {
                $_SESSION['flash_message'] = $result['message'];
                $_SESSION['flash_type'] = "danger";
            }
        }
        
        // Redirecionar para evitar reenvio do formulário
        header('Location: settings.php');
        exit;
    }
}

// Obter configurações atuais
$settings = $settings_controller->getAllSettings();

// Definir título da página
$page_title = getenv('APP_PAGE_TITLE');

// Incluir cabeçalho
include 'includes/header.php';
?>

<div class="row mb-4">
    <div class="col-md-6">
        <h1><i class="fas fa-cogs me-2"></i> Configurações do Sistema</h1>
    </div>
</div>

<div class="row">
    <div class="col-md-3 mb-4">
        <div class="card">
            <div class="card-header">
                <h5 class="mb-0">Menu de Configurações</h5>
            </div>
            <div class="list-group list-group-flush">
                <a href="#general" class="list-group-item list-group-item-action active" data-bs-toggle="list">
                    <i class="fas fa-sliders-h me-2"></i> Configurações Gerais
                </a>
                <a href="#email" class="list-group-item list-group-item-action" data-bs-toggle="list">
                    <i class="fas fa-envelope me-2"></i> Configurações de E-mail
                </a>
                <a href="#backup" class="list-group-item list-group-item-action" data-bs-toggle="list">
                    <i class="fas fa-database me-2"></i> Configurações de Backup
                </a>
                <a href="#appearance" class="list-group-item list-group-item-action" data-bs-toggle="list">
                    <i class="fas fa-palette me-2"></i> Aparência
                </a>
                <a href="#system" class="list-group-item list-group-item-action" data-bs-toggle="list">
                    <i class="fas fa-info-circle me-2"></i> Informações do Sistema
                </a>
            </div>
        </div>
    </div>
    
    <div class="col-md-9">
        <div class="tab-content">
            <!-- Configurações Gerais -->
            <div class="tab-pane fade show active" id="general">
                <div class="card">
                    <div class="card-header">
                        <h5 class="mb-0">Configurações Gerais</h5>
                    </div>
                    <div class="card-body">
                        <form method="POST" action="settings.php">
                            <input type="hidden" name="action" value="general">
                            
                            <div class="mb-3">
                                <label for="system_name" class="form-label">Nome do Sistema</label>
                                <input type="text" class="form-control" id="system_name" name="system_name" value="<?php echo htmlspecialchars($settings['system_name'] ?? 'Sistema GED'); ?>">
                                <div class="form-text">Nome exibido no cabeçalho e título das páginas.</div>
                            </div>
                            
                            <div class="mb-3 form-check">
                                <input type="checkbox" class="form-check-input" id="allow_registration" name="allow_registration" <?php echo ($settings['allow_registration'] ?? 0) ? 'checked' : ''; ?>>
                                <label class="form-check-label" for="allow_registration">Permitir registro de novos usuários</label>
                                <div class="form-text">Se desativado, apenas administradores podem criar novos usuários.</div>
                            </div>
                            
                            <div class="mb-3">
                                <label for="max_upload_size" class="form-label">Tamanho máximo de upload (MB)</label>
                                <input type="number" class="form-control" id="max_upload_size" name="max_upload_size" value="<?php echo htmlspecialchars($settings['max_upload_size'] ?? 10); ?>" min="1" max="100">
                                <div class="form-text">Tamanho máximo permitido para upload de arquivos em megabytes.</div>
                            </div>
                            
                            <div class="mb-3">
                                <label for="allowed_extensions" class="form-label">Extensões de arquivo permitidas</label>
                                <input type="text" class="form-control" id="allowed_extensions" name="allowed_extensions" value="<?php echo htmlspecialchars($settings['allowed_extensions'] ?? 'pdf,doc,docx,xls,xlsx,jpg,jpeg,png'); ?>">
                                <div class="form-text">Lista de extensões de arquivo permitidas, separadas por vírgula.</div>
                            </div>
                            
                            <div class="mb-3">
                                <label for="default_language" class="form-label">Idioma padrão</label>
                                <select class="form-select" id="default_language" name="default_language">
                                    <option value="pt_BR" <?php echo ($settings['default_language'] ?? 'pt_BR') == 'pt_BR' ? 'selected' : ''; ?>>Português (Brasil)</option>
                                    <option value="en_US" <?php echo ($settings['default_language'] ?? '') == 'en_US' ? 'selected' : ''; ?>>English (US)</option>
                                    <option value="es_ES" <?php echo ($settings['default_language'] ?? '') == 'es_ES' ? 'selected' : ''; ?>>Español</option>
                                </select>
                                <div class="form-text">Idioma padrão do sistema.</div>
                            </div>
                            
                            <div class="mb-3">
                                <label for="items_per_page" class="form-label">Itens por página</label>
                                <select class="form-select" id="items_per_page" name="items_per_page">
                                    <option value="10" <?php echo ($settings['items_per_page'] ?? 20) == 10 ? 'selected' : ''; ?>>10</option>
                                    <option value="20" <?php echo ($settings['items_per_page'] ?? 20) == 20 ? 'selected' : ''; ?>>20</option>
                                    <option value="50" <?php echo ($settings['items_per_page'] ?? 20) == 50 ? 'selected' : ''; ?>>50</option>
                                    <option value="100" <?php echo ($settings['items_per_page'] ?? 20) == 100 ? 'selected' : ''; ?>>100</option>
                                </select>
                                <div class="form-text">Número de itens exibidos por página nas listagens.</div>
                            </div>
                            
                            <button type="submit" class="btn btn-primary">
                                <i class="fas fa-save me-1"></i> Salvar Configurações
                            </button>
                        </form>
                    </div>
                </div>
            </div>
            
            <!-- Configurações de E-mail -->
            <div class="tab-pane fade" id="email">
                <div class="card">
                    <div class="card-header">
                        <h5 class="mb-0">Configurações de E-mail</h5>
                    </div>
                    <div class="card-body">
                        <form method="POST" action="settings.php">
                            <input type="hidden" name="action" value="email">
                            
                            <div class="mb-3">
                                <label for="smtp_host" class="form-label">Servidor SMTP</label>
                                <input type="text" class="form-control" id="smtp_host" name="smtp_host" value="<?php echo htmlspecialchars($settings['smtp_host'] ?? ''); ?>">
                                <div class="form-text">Endereço do servidor SMTP (ex: smtp.gmail.com).</div>
                            </div>
                            
                            <div class="mb-3">
                                <label for="smtp_port" class="form-label">Porta SMTP</label>
                                <input type="number" class="form-control" id="smtp_port" name="smtp_port" value="<?php echo htmlspecialchars($settings['smtp_port'] ?? 587); ?>">
                                <div class="form-text">Porta do servidor SMTP (geralmente 587 para TLS ou 465 para SSL).</div>
                            </div>
                            
                            <div class="mb-3">
                                <label for="smtp_secure" class="form-label">Segurança SMTP</label>
                                <select class="form-select" id="smtp_secure" name="smtp_secure">
                                    <option value="tls" <?php echo ($settings['smtp_secure'] ?? 'tls') == 'tls' ? 'selected' : ''; ?>>TLS</option>
                                    <option value="ssl" <?php echo ($settings['smtp_secure'] ?? '') == 'ssl' ? 'selected' : ''; ?>>SSL</option>
                                    <option value="" <?php echo ($settings['smtp_secure'] ?? '') === '' ? 'selected' : ''; ?>>Nenhuma</option>
                                </select>
                                <div class="form-text">Tipo de segurança para conexão SMTP.</div>
                            </div>
                            
                            <div class="mb-3 form-check">
                                <input type="checkbox" class="form-check-input" id="smtp_auth" name="smtp_auth" <?php echo ($settings['smtp_auth'] ?? 1) ? 'checked' : ''; ?>>
                                <label class="form-check-label" for="smtp_auth">Usar autenticação SMTP</label>
                                <div class="form-text">Marque se o servidor SMTP requer autenticação.</div>
                            </div>
                            
                            <div class="mb-3">
                                <label for="smtp_username" class="form-label">Usuário SMTP</label>
                                <input type="text" class="form-control" id="smtp_username" name="smtp_username" value="<?php echo htmlspecialchars($settings['smtp_username'] ?? ''); ?>">
                                <div class="form-text">Nome de usuário para autenticação SMTP.</div>
                            </div>
                            
                            <div class="mb-3">
                                <label for="smtp_password" class="form-label">Senha SMTP</label>
                                <input type="password" class="form-control" id="smtp_password" name="smtp_password" value="<?php echo htmlspecialchars($settings['smtp_password'] ?? ''); ?>">
                                <div class="form-text">Senha para autenticação SMTP.</div>
                            </div>
                            
                            <div class="mb-3">
                                <label for="mail_from" class="form-label">E-mail de Origem</label>
                                <input type="email" class="form-control" id="mail_from" name="mail_from" value="<?php echo htmlspecialchars($settings['mail_from'] ?? ''); ?>">
                                <div class="form-text">Endereço de e-mail que aparecerá como remetente.</div>
                            </div>
                            
                            <div class="mb-3">
                                <label for="mail_from_name" class="form-label">Nome de Origem</label>
                                <input type="text" class="form-control" id="mail_from_name" name="mail_from_name" value="<?php echo htmlspecialchars($settings['mail_from_name'] ?? 'Sistema GED'); ?>">
                                <div class="form-text">Nome que aparecerá como remetente.</div>
                            </div>
                            
                            <button type="submit" class="btn btn-primary">
                                <i class="fas fa-save me-1"></i> Salvar Configurações
                            </button>
                            
                            <button type="button" class="btn btn-info ms-2" id="test-email-btn">
                                <i class="fas fa-paper-plane me-1"></i> Testar Configurações
                            </button>
                        </form>
                    </div>
                </div>
            </div>
            
            <!-- Configurações de Backup -->
            <div class="tab-pane fade" id="backup">
                <div class="card">
                    <div class="card-header">
                        <h5 class="mb-0">Configurações de Backup</h5>
                    </div>
                    <div class="card-body">
                        <form method="POST" action="settings.php">
                            <input type="hidden" name="action" value="backup">
                            
                            <div class="mb-3 form-check">
                                <input type="checkbox" class="form-check-input" id="auto_backup" name="auto_backup" <?php echo ($settings['auto_backup'] ?? 0) ? 'checked' : ''; ?>>
                                <label class="form-check-label" for="auto_backup">Ativar backup automático</label>
                                <div class="form-text">Realiza backups automáticos do banco de dados e arquivos.</div>
                            </div>
                            
                            <div class="mb-3">
                                <label for="backup_frequency" class="form-label">Frequência de Backup</label>
                                <select class="form-select" id="backup_frequency" name="backup_frequency">
                                    <option value="daily" <?php echo ($settings['backup_frequency'] ?? 'daily') == 'daily' ? 'selected' : ''; ?>>Diário</option>
                                    <option value="weekly" <?php echo ($settings['backup_frequency'] ?? '') == 'weekly' ? 'selected' : ''; ?>>Semanal</option>
                                    <option value="monthly" <?php echo ($settings['backup_frequency'] ?? '') == 'monthly' ? 'selected' : ''; ?>>Mensal</option>
                                </select>
                                <div class="form-text">Com que frequência os backups automáticos serão realizados.</div>
                            </div>
                            
                            <div class="mb-3">
                                <label for="backup_time" class="form-label">Horário do Backup</label>
                                <input type="time" class="form-control" id="backup_time" name="backup_time" value="<?php echo htmlspecialchars($settings['backup_time'] ?? '00:00'); ?>">
                                <div class="form-text">Horário em que o backup automático será iniciado.</div>
                            </div>
                            
                            <div class="mb-3">
                                <label for="backup_retention" class="form-label">Retenção de Backup (dias)</label>
                                <input type="number" class="form-control" id="backup_retention" name="backup_retention" value="<?php echo htmlspecialchars($settings['backup_retention'] ?? 7); ?>" min="1">
                                <div class="form-text">Por quantos dias os backups serão mantidos antes de serem excluídos.</div>
                            </div>
                            
                            <div class="mb-3 form-check">
                                <input type="checkbox" class="form-check-input" id="backup_include_files" name="backup_include_files" <?php echo ($settings['backup_include_files'] ?? 1) ? 'checked' : ''; ?>>
                                <label class="form-check-label" for="backup_include_files">Incluir arquivos no backup</label>
                                <div class="form-text">Se marcado, os arquivos de documentos também serão incluídos no backup.</div>
                            </div>
                            
                            <div class="mb-3">
                                <label for="backup_destination" class="form-label">Destino do Backup</label>
                                <select class="form-select" id="backup_destination" name="backup_destination">
                                    <option value="local" <?php echo ($settings['backup_destination'] ?? 'local') == 'local' ? 'selected' : ''; ?>>Local (servidor)</option>
                                    <option value="cloud" <?php echo ($settings['backup_destination'] ?? '') == 'cloud' ? 'selected' : ''; ?>>Nuvem</option>
                                </select>
                                <div class="form-text">Onde os backups serão armazenados.</div>
                            </div>
                            
                            <div id="cloud-settings" class="<?php echo ($settings['backup_destination'] ?? 'local') == 'local' ? 'd-none' : ''; ?>">
                                <div class="mb-3">
                                    <label for="backup_cloud_provider" class="form-label">Provedor de Nuvem</label>
                                    <select class="form-select" id="backup_cloud_provider" name="backup_cloud_provider">
                                        <option value="s3" <?php echo ($settings['backup_cloud_provider'] ?? '') == 's3' ? 'selected' : ''; ?>>Amazon S3</option>
                                        <option value="dropbox" <?php echo ($settings['backup_cloud_provider'] ?? '') == 'dropbox' ? 'selected' : ''; ?>>Dropbox</option>
                                        <option value="gdrive" <?php echo ($settings['backup_cloud_provider'] ?? '') == 'gdrive' ? 'selected' : ''; ?>>Google Drive</option>
                                    </select>
                                </div>
                                
                                <div class="mb-3">
                                    <label for="backup_cloud_key" class="form-label">Chave de API</label>
                                    <input type="text" class="form-control" id="backup_cloud_key" name="backup_cloud_key" value="<?php echo htmlspecialchars($settings['backup_cloud_key'] ?? ''); ?>">
                                </div>
                                
                                <div class="mb-3">
                                    <label for="backup_cloud_secret" class="form-label">Chave Secreta</label>
                                    <input type="password" class="form-control" id="backup_cloud_secret" name="backup_cloud_secret" value="<?php echo htmlspecialchars($settings['backup_cloud_secret'] ?? ''); ?>">
                                </div>
                                
                                <div class="mb-3">
                                    <label for="backup_cloud_bucket" class="form-label">Bucket/Pasta</label>
                                    <input type="text" class="form-control" id="backup_cloud_bucket" name="backup_cloud_bucket" value="<?php echo htmlspecialchars($settings['backup_cloud_bucket'] ?? ''); ?>">
                                    <div class="form-text">Nome do bucket S3 ou pasta no Dropbox/Google Drive.</div>
                                </div>
                            </div>
                            
                            <button type="submit" class="btn btn-primary">
                                <i class="fas fa-save me-1"></i> Salvar Configurações
                            </button>
                            
                            <button type="button" class="btn btn-info ms-2" id="manual-backup-btn">
                                <i class="fas fa-download me-1"></i> Executar Backup Manual
                            </button>
                        </form>
                    </div>
                </div>
            </div>
            
            <!-- Configurações de Aparência -->
            <div class="tab-pane fade" id="appearance">
                <div class="card">
                    <div class="card-header">
                        <h5 class="mb-0">Configurações de Aparência</h5>
                    </div>
                    <div class="card-body">
                        <form method="POST" action="settings.php">
                            <input type="hidden" name="action" value="appearance">
                            
                            <div class="mb-3">
                                <label for="default_theme" class="form-label">Tema Padrão</label>
                                <select class="form-select" id="default_theme" name="default_theme">
                                    <option value="light" <?php echo ($settings['default_theme'] ?? 'light') == 'light' ? 'selected' : ''; ?>>Claro</option>
                                    <option value="dark" <?php echo ($settings['default_theme'] ?? '') == 'dark' ? 'selected' : ''; ?>>Escuro</option>
                                    <option value="auto" <?php echo ($settings['default_theme'] ?? '') == 'auto' ? 'selected' : ''; ?>>Automático (baseado no sistema)</option>
                                </select>
                                <div class="form-text">Tema padrão para novos usuários.</div>
                            </div>
                            
                            <div class="mb-3">
                                <label for="primary_color" class="form-label">Cor Primária</label>
                                <input type="color" class="form-control form-control-color" id="primary_color" name="primary_color" value="<?php echo htmlspecialchars($settings['primary_color'] ?? '#0d6efd'); ?>">
                                <div class="form-text">Cor principal do sistema.</div>
                            </div>
                            
                            <div class="mb-3">
                                <label for="secondary_color" class="form-label">Cor Secundária</label>
                                <input type="color" class="form-control form-control-color" id="secondary_color" name="secondary_color" value="<?php echo htmlspecialchars($settings['secondary_color'] ?? '#6c757d'); ?>">
                                <div class="form-text">Cor secundária do sistema.</div>
                            </div>
                            
                            <div class="mb-3">
                                <label for="logo_url" class="form-label">URL do Logo</label>
                                <input type="text" class="form-control" id="logo_url" name="logo_url" value="<?php echo htmlspecialchars($settings['logo_url'] ?? ''); ?>">
                                <div class="form-text">URL da imagem do logo (recomendado: 200x50px).</div>
                            </div>
                            
                            <div class="mb-3">
                                <label for="favicon_url" class="form-label">URL do Favicon</label>
                                <input type="text" class="form-control" id="favicon_url" name="favicon_url" value="<?php echo htmlspecialchars($settings['favicon_url'] ?? ''); ?>">
                                <div class="form-text">URL do favicon (recomendado: 32x32px).</div>
                            </div>
                            
                            <div class="mb-3">
                                <label for="custom_css" class="form-label">CSS Personalizado</label>
                                <textarea class="form-control" id="custom_css" name="custom_css" rows="5"><?php echo htmlspecialchars($settings['custom_css'] ?? ''); ?></textarea>
                                <div class="form-text">CSS personalizado para ajustar a aparência do sistema.</div>
                            </div>
                            
                            <button type="submit" class="btn btn-primary">
                                <i class="fas fa-save me-1"></i> Salvar Configurações
                            </button>
                            
                            <button type="button" class="btn btn-secondary ms-2" id="preview-theme-btn">
                                <i class="fas fa-eye me-1"></i> Pré-visualizar
                            </button>
                        </form>
                    </div>
                </div>
            </div>
            
            <!-- Informações do Sistema -->
            <div class="tab-pane fade" id="system">
                <div class="card">
                    <div class="card-header">
                        <h5 class="mb-0">Informações do Sistema</h5>
                    </div>
                    <div class="card-body">
                        <div class="row">
                            <div class="col-md-6">
                                <h6>Informações do Software</h6>
                                <table class="table table-bordered">
                                    <tr>
                                        <th>Versão do Sistema</th>
                                        <td>1.0.0</td>
                                    </tr>
                                    <tr>
                                        <th>Data de Instalação</th>
                                        <td><?php echo date('d/m/Y', strtotime($settings['installation_date'] ?? date('Y-m-d'))); ?></td>
                                    </tr>
                                    <tr>
                                        <th>Última Atualização</th>
                                        <td><?php echo date('d/m/Y', strtotime($settings['last_update'] ?? date('Y-m-d'))); ?></td>
                                    </tr>
                                    <tr>
                                        <th>Licença</th>
                                        <td>Comercial</td>
                                    </tr>
                                </table>
                            </div>
                            
                            <div class="col-md-6">
                                <h6>Informações do Servidor</h6>
                                <table class="table table-bordered">
                                    <tr>
                                        <th>Versão do PHP</th>
                                        <td><?php echo phpversion(); ?></td>
                                    </tr>
                                    <tr>
                                        <th>Versão do MySQL</th>
                                        <td><?php echo $db->getAttribute(PDO::ATTR_SERVER_VERSION); ?></td>
                                    </tr>
                                    <tr>
                                        <th>Sistema Operacional</th>
                                        <td><?php echo php_uname('s') . ' ' . php_uname('r'); ?></td>
                                    </tr>
                                    <tr>
                                        <th>Servidor Web</th>
                                        <td><?php echo $_SERVER['SERVER_SOFTWARE']; ?></td>
                                    </tr>
                                </table>
                            </div>
                        </div>
                        
                        <div class="row mt-4">
                            <div class="col-md-6">
                                <h6>Estatísticas do Sistema</h6>
                                <table class="table table-bordered">
                                    <tr>
                                        <th>Total de Usuários</th>
                                        <td><?php echo $settings_controller->getStatistics('users'); ?></td>
                                    </tr>
                                    <tr>
                                        <th>Total de Documentos</th>
                                        <td><?php echo $settings_controller->getStatistics('documents'); ?></td>
                                    </tr>
                                    <tr>
                                        <th>Total de Categorias</th>
                                        <td><?php echo $settings_controller->getStatistics('categories'); ?></td>
                                    </tr>
                                    <tr>
                                        <th>Espaço Utilizado</th>
                                        <td><?php echo $settings_controller->getStatistics('disk_usage'); ?></td>
                                    </tr>
                                </table>
                            </div>
                            
                            <div class="col-md-6">
                                <h6>Verificação do Sistema</h6>
                                <div class="list-group">
                                    <?php $checks = $settings_controller->performSystemChecks(); ?>
                                    <?php foreach ($checks as $check): ?>
                                    <div class="list-group-item d-flex justify-content-between align-items-center">
                                        <?php echo $check['name']; ?>
                                        <?php if ($check['status'] == 'ok'): ?>
                                        <span class="badge bg-success rounded-pill"><i class="fas fa-check"></i></span>
                                        <?php elseif ($check['status'] == 'warning'): ?>
                                        <span class="badge bg-warning text-dark rounded-pill"><i class="fas fa-exclamation-triangle"></i></span>
                                        <?php else: ?>
                                        <span class="badge bg-danger rounded-pill"><i class="fas fa-times"></i></span>
                                        <?php endif; ?>
                                    </div>
                                    <?php endforeach; ?>
                                </div>
                            </div>
                        </div>
                        
                        <div class="mt-4">
                            <button type="button" class="btn btn-info" id="check-updates-btn">
                                <i class="fas fa-sync me-1"></i> Verificar Atualizações
                            </button>
                            
                            <button type="button" class="btn btn-warning ms-2" id="clear-cache-btn">
                                <i class="fas fa-broom me-1"></i> Limpar Cache
                            </button>
                            
                            <button type="button" class="btn btn-danger ms-2" id="repair-db-btn">
                                <i class="fas fa-tools me-1"></i> Reparar Banco de Dados
                            </button>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Modal para teste de e-mail -->
<div class="modal fade" id="testEmailModal" tabindex="-1" aria-labelledby="testEmailModalLabel" aria-hidden="true">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="testEmailModalLabel">Testar Configurações de E-mail</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Fechar"></button>
            </div>
            <div class="modal-body">
                <div class="mb-3">
                    <label for="test-email" class="form-label">Enviar e-mail de teste para:</label>
                    <input type="email" class="form-control" id="test-email" value="<?php echo htmlspecialchars($_SESSION['email'] ?? ''); ?>">
                </div>
                <div id="test-email-result"></div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Fechar</button>
                <button type="button" class="btn btn-primary" id="send-test-email">Enviar</button>
            </div>
        </div>
    </div>
</div>

<!-- JavaScript para manipulação das configurações -->
<script>
document.addEventListener('DOMContentLoaded', function() {
    // Mostrar/ocultar configurações de nuvem com base na seleção
    document.getElementById('backup_destination').addEventListener('change', function() {
        const cloudSettings = document.getElementById('cloud-settings');
        if (this.value === 'cloud') {
            cloudSettings.classList.remove('d-none');
        } else {
            cloudSettings.classList.add('d-none');
        }
    });
    
    // Manipular clique no botão de teste de e-mail
    document.getElementById('test-email-btn').addEventListener('click', function() {
        const testEmailModal = new bootstrap.Modal(document.getElementById('testEmailModal'));
        testEmailModal.show();
    });
    
    // Enviar e-mail de teste
    document.getElementById('send-test-email').addEventListener('click', function() {
        const email = document.getElementById('test-email').value;
        const resultDiv = document.getElementById('test-email-result');
        
        if (!email) {
            resultDiv.innerHTML = '<div class="alert alert-danger">Por favor, informe um endereço de e-mail válido.</div>';
            return;
        }
        
        resultDiv.innerHTML = '<div class="alert alert-info">Enviando e-mail de teste...</div>';
        
        // Aqui você faria uma requisição AJAX para enviar o e-mail de teste
        // Por simplicidade, estamos apenas simulando o resultado
        setTimeout(function() {
            resultDiv.innerHTML = '<div class="alert alert-success">E-mail de teste enviado com sucesso para ' + email + '.</div>';
        }, 2000);
    });
    
    // Manipular clique no botão de backup manual
    document.getElementById('manual-backup-btn').addEventListener('click', function() {
        if (confirm('Deseja realmente executar um backup manual agora?')) {
            alert('Backup manual iniciado. Este processo pode levar alguns minutos.');
            // Aqui você faria uma requisição AJAX para iniciar o backup manual
        }
    });
    
    // Manipular clique no botão de pré-visualização de tema
    document.getElementById('preview-theme-btn').addEventListener('click', function() {
        const primaryColor = document.getElementById('primary_color').value;
        const secondaryColor = document.getElementById('secondary_color').value;
        const customCSS = document.getElementById('custom_css').value;
        
        // Aplicar cores temporariamente
        document.documentElement.style.setProperty('--bs-primary', primaryColor);
        document.documentElement.style.setProperty('--bs-secondary', secondaryColor);
        
        // Aplicar CSS personalizado temporariamente
        let styleElement = document.getElementById('preview-custom-css');
        if (!styleElement) {
            styleElement = document.createElement('style');
            styleElement.id = 'preview-custom-css';
            document.head.appendChild(styleElement);
        }
        styleElement.textContent = customCSS;
        
        alert('Tema aplicado temporariamente para pré-visualização. As alterações não serão salvas até que você clique em "Salvar Configurações".');
    });
    
    // Manipular clique nos botões da seção de informações do sistema
    document.getElementById('check-updates-btn').addEventListener('click', function() {
        alert('Verificando atualizações disponíveis...');
        // Aqui você faria uma requisição AJAX para verificar atualizações
    });
    
    document.getElementById('clear-cache-btn').addEventListener('click', function() {
        if (confirm('Deseja realmente limpar o cache do sistema?')) {
            alert('Cache limpo com sucesso!');
            // Aqui você faria uma requisição AJAX para limpar o cache
        }
    });
    
    document.getElementById('repair-db-btn').addEventListener('click', function() {
        if (confirm('ATENÇÃO: Esta operação pode levar algum tempo e afetar temporariamente o desempenho do sistema. Deseja realmente reparar o banco de dados?')) {
            alert('Reparo do banco de dados iniciado. Este processo pode levar alguns minutos.');
            // Aqui você faria uma requisição AJAX para reparar o banco de dados
        }
    });
});
</script>

<?php
// Incluir rodapé
include 'includes/footer.php';
?>
